"""Bottle Plugin that will pass an initialized job manager object into each
route
"""
#pylint: disable=no-self-use,too-few-public-methods
import inspect

from .job_manager import JobManager


class JobManagerPlugin(object):
    """Bottle Plugin that will pass an initialized job manager object into each
    route
    See https://bottlepy.org/docs/dev/plugindev.html#bottle.Plugin for
    documentation on the api
    """
    name = 'job_manager'
    api = 2

    def apply(self, callback, route):
        """Method used by Bottle to decorate routes
        """

        def wrapper(*args, **kwargs):
            """Function that sets up the the job_manager object based on the
            job manager filepath configuration and passes job_manager to the
            wrapped function
            """
            product_specific = route.app.config["rest_api.product_specific"]
            job_schema = product_specific.job_schema
            cdm_schema = product_specific.cdm_schema
            db_file = route.app.config.get('rest_api.db_file')
            job_manager = JobManager(job_schema, db_file, cdm_schema,product_specific)
            kwargs['job_manager'] = job_manager

            return_value = callback(*args, **kwargs)

            return return_value

        return wrapper


class SideEffect(object):
    """Bottle plugin that will pass a side effect function that will be called
    when the job information changes and the job is active
    """
    name = 'side_effect'
    api = 2

    def apply(self, callback, route):
        """Method used by Bottle to decorate routes
        """
        product_specific = route.app.config.get('rest_api.product_specific')
        arg_spec = arg_spec_from_callback(callback)

        if 'side_effect' not in arg_spec.args:
            return callback

        def wrapper(*args, **kwargs):
            """Function that adds the side effect function into the callback
            function's key word arguments
            """
            kwargs['side_effect'] = product_specific.update_job_side_effect

            return_value = callback(*args, **kwargs)

            return return_value

        return wrapper

def arg_spec_from_callback(callback):
    """Function to get the argument specification from the original function
    before it is wrapped by web_args

    Tests to see if the function is wrapped and if so looks for the original
    function in the wrapped functions closures
    """
    closures = callback.__closure__
    if closures:
        for closure in closures:
            if callable(closure.cell_contents):
                arg_spec = inspect.getargspec(closure.cell_contents)

    else:
        arg_spec = inspect.getargspec(callback)
    return arg_spec
