"""
Module containing the planned test object
"""

from .helper import html_format_test_data
from .helper import get_formated_status


class PlannedTest(object):
    """
    Class for handling planned tests that transitions the state of the
    planned test and it's information
    """
    def __init__(self, args):
        self.test_type = args['test_type']
        self.test_label = args.get('test_label', "")
        self.status = args['status']
        self.reference_info = args['reference_info']
        self.sub_type_info = args.get('sub_type_info')
        self.test_data = args['test_data']
        self.procedures = args['procedures']
        self.config = args.get('config', None)
        self.workflow_id = args.get('workflow_id', 0)

    def add_test_data(self, test_data):
        """
        Method to add test data to a planned test.  Test data will be added to the
        list of test data and the status will be set to match the verdict in the new
        test data

        Args:
            test_data (dict): dictionary of the test data to add
        """
        self._update_status(test_data)

        self._update_reference_info(test_data)

        self.test_data.append(test_data)

    def matches(self, test_data):
        """
        Method to determine if test data matches that expected by the planned test

        test_type has to match
        if there is reference info that has to match in any order of key/value pairs
        if there is not reference info matches if the planned test is in To Do status
        if there is extra subType info this has to match in addition to the reference info matching

        Args:
            test_data (dict): dictionary of the test data

        Returns:
            (bool): True if the test data matches False if not
        """

        if not test_data['test_type'] == self.test_type:
            return False

        refInfoMatches = False
        subTypeInfoMatches = False

        refInfoFound = False
        subTypeFound = False

        if self.reference_info:
            refInfoFound = True
            refInfoMatches = self._reference_info_matches(test_data)

        if self.sub_type_info:
            subTypeFound = True
            subTypeInfoMatches = self._sub_type_info_matches(test_data)

        if (refInfoFound and subTypeFound):
            return (refInfoMatches and subTypeInfoMatches)
        elif refInfoFound:
            return refInfoMatches
        else:
            return self.status == 'To Do'

    def get_html_format(self):
        """
        Method to return a dictionary with the HTMl formatting of the planned test
        so that templating the report will be easier
        """
        test_data_html_format = [html_format_test_data(test_data)
                                 for test_data
                                 in self.test_data]

        return {'test_type': self.test_type,
                'test_label': self.test_label,
                'status': get_formated_status(self.status),
                'reference_info': self.reference_info,
                'procedures': self.procedures,
                'test_data': test_data_html_format}

    def _update_status(self, test_data):
        """Method to update status based on the verdict of the test data

        Args:
            test_data (dict): dictionary of the test data
        """
        if test_data['verdict'] == 'NotApplicable':
            self.status = 'Complete'
        elif test_data['verdict'] == 'Skipped':
            self.status = 'Skipped'
        else:
            self.status = test_data['verdict']

    def _update_reference_info(self, test_data):
        """Method to have the planned test adopt the reference info of the test
        data if the planned test's reference info is empty

        Args:
            test_data (dict): dictionary of the test data
        """
        if not self.reference_info:
            self.reference_info = test_data['reference_info']

    def _reference_info_matches(self, test_data):
        """Method to determine if the reference info in the test data matches
        the reference info in the planned tests

        The reference info dictionaries can can appear in any any order becuase
        they are sorted as part of the comparision

        Args:
            test_data (dict): dictionary of the test data

        Returns:
            bool: True if matches False otherwise
        """
        our_reference_info = sorted(self.reference_info,
                                    key=lambda x: sorted(x.values()))

        incomming_reference_info = sorted(test_data['reference_info'],
                                          key=lambda x: sorted(x.values()))

        return our_reference_info == incomming_reference_info

    def _sub_type_info_matches(self, test_data):
        """Method to determine if the config details placed in the optional sub_type_info set of key-value pairs
        in the test data matches the keys in the config schema for that particular test type in the planned tests

        The config info dictionaries can can appear in any any order because
        they are sorted as part of the comparision

        Args:
            test_data (dict): dictionary of the test data

        Returns:
            bool: True if matches False otherwise
        """
        our_sub_type_info = sorted(self.sub_type_info,
                                    key=lambda x: sorted(x.values()))

        incoming_sub_type_info = sorted(test_data['sub_type_info'],
                                          key=lambda x: sorted(x.values()))

        """
        Need to know if incoming_sub_type_info.key .... is found in config.schema contents of the planned test,
        and the values are the same
        Typically only one key-value pair, such as label for manual Steps, otdrTopology for OTDRs, etc.
        but can leave the possibility of several key-value pairs just in case ?? and just test the whole block

        for key in incoming_config_info:
            if value == our_config_info.key.value: *****NEED SOME PYTHON EXPERTISE HERE
                print name
        """
        return our_sub_type_info == incoming_sub_type_info
