"""
Module containing classes and helper functions for
manipulating jobs
"""
import logging

from . import data_store
from .job import Job
from .job_adapter_viavi import load_from_cdm_string

log = logging.getLogger(__name__)


class JobManager(object):
    '''
    Class for managing job data

    Attibutes:
        job: a Job object used for tranforming the the information in a job
             None prior to loading the job or if the job does not exist
    '''

    def __init__(self, job_schema, job_manager_file, cdm_schema,product_specific=""):
        self.job_manager_file = job_manager_file
        self.data_store = data_store.DataStore(job_manager_file)
        self.job = None
        self.active_state = False
        self.job_schema = job_schema
        self.cdm_schema = cdm_schema
        self.product_specific = product_specific

    def get_job_summaries(self):
        """
        Method to return a list of dictionaries containing job id's and job names for
        all jobs in the datastore
        """
        job_eids = self.data_store.get_jobs_eids()
        return [{'job_Id': job_eid, 'job_name': 'defaultJob'} for job_eid in job_eids]

    def load_job(self, job_id):
        '''
        Method to load the job information from the datastore into a job object

        Args:
            job_id (int): the index for the job to load

        Returns:
            job (Job): object containing the job information
                       None if not in datastore
        '''
        #Early bailout if job id is not '1'
        if job_id != 1:
            self.job = None
            return self.job

        job_dict = self.data_store.read_job(job_id)

        unmarshal_result = self.job_schema().load(job_dict)
        if unmarshal_result.errors:
            job = None
            log.debug('## load_job: unmarshal_result.errors: %s', unmarshal_result.errors )
        else:
            job = Job(unmarshal_result.data)
        self.job = job

        active_job = self.data_store.read_active()['active_job']
        self.active_state = active_job == job_id

        return job

    def load_job_from_file(self, file_path):
        """
        Method to load a job from a file containing job data in json format

        Args:
            file_path (str): the file_path of the file to load
        """
        with open(file_path) as job_file:
            job_string = job_file.read()

        self.job = load_from_string(self.job_schema, job_string)
        if not self.job:
            log.debug('## load_job_from_file: Trying cdm: %s', self.cdm_schema)
            cdm_dictionary = load_from_cdm_string(job_string, self.cdm_schema)
            try:
                self.job = Job(cdm_dictionary)
            except TypeError:
                log.debug('## load_job_from_file: TypeError')
                return None

        if not self.job:
            return None
        job_dict = self.dump_job()
        self.data_store.write_job(job_dict)

        if self.active_state:
            self.product_specific.update_loaded_job_side_effect(1,self.job.status_ratio)
        else:
            self.product_specific.update_loaded_job_side_effect(0,self.job.status_ratio)

        extra = {'jobId': '1', 'jobName': 'defaultJob'}
        return self.job_schema(extra=extra).dump(self.job).data

    def save_job(self, job_data):
        '''
        Method to save job to the datastore

        Args:
            job_data (dict): dictionary of the job data to send to the datastore
        '''
        self.job = Job(job_data)
        job_dict = self.dump_job()
        self.data_store.write_job(job_dict)

        extra = {'jobId': '1', 'jobName': 'defaultJob'}
        return self.job_schema(extra=extra).dump(self.job).data

    def dump_job(self):
        """
        Method to serialize the job object as a dictionary with
        cammelCase keys
        """
        return self.job_schema().dump(self.job).data

    def add_test_data(self, test_data):
        """
        Method called when processing new test data

        Args:
            test_data (dict): the dictionary of new test data
        """
        test_plan_index = self.job.add_test_data(test_data)
        if test_plan_index is not None:
            self.product_specific.update_test_data_side_effect(test_plan_index,self.job.status_ratio)

        job_dict = self.dump_job()
        self.data_store.write_job(job_dict)
        return job_dict

    def set_active(self, active_job_eid):
        '''
        Method to set the active job in the datastore

        Args:
            active_job_eid (int): the element id of the active job or
                0 to set all jobs inactive
        '''
        if active_job_eid:
            self.data_store.set_active(int(active_job_eid))
            self.product_specific.update_loaded_job_side_effect(1,self.job.status_ratio)
        else:
            self.data_store.set_active(None)
            self.product_specific.update_loaded_job_side_effect(0,self.job.status_ratio)

    def set_active_test_plan_index(self, active_job_eid, active_job_item_index):
        '''
        Method to set the index of the active job item within a test plan

        Args:
            active_job_eid (int): the element id of the active job
            active_job_item_index (int): the index of the actively launched job item in the active job

        '''
        if active_job_eid:
           self.data_store.set_active_job_item_index(int(active_job_eid), int(active_job_item_index))

    def get_active_test_plan_index(self, active_job_eid):
        '''
        Method to get the index of the active job item within a test plan

        Args:
            active_job_eid (int): the element id of the active job

        '''
        return self.data_store.read_active_job_item_index(active_job_eid)['active_job_item_index']

    def delete_job(self):
        '''This function will gain more functionality as the API
        expands. Currently it only serves to abstract the job manager
        file away from the rest of the API'''
        self.data_store.delete_job()


def load_from_string(job_schema, input_string):
    '''
    Alternate constructor for the Job class

    Args:
        input_string (str): the json representation of the job information

    Returns:
        job (Job): the deserialized Job object
    '''
    try:
        unmarshal_result = job_schema().loads(input_string)
    except ValueError:
        return None

    if unmarshal_result.errors:
        log.debug('## load_from_string: unmarshal_result.errors: %s', unmarshal_result.errors )
        return None

    log.debug('## load_from_string: Loaded Native format job')

    return Job(unmarshal_result.data)
