'''
set of methods to convert cdm json file to a Job object
'''
import logging
from json import dumps

log = logging.getLogger(__name__)

def load_from_cdm_string(input_json, cdm_schema):
    '''
    Create a Job from a cdm json file
    '''
    log.debug('## start: load_from_cdm_string')

    try:
        cdm_job = cdm_schema().loads(input_json)
    except ValueError:
        log.debug('## except ValueError during load_from_cdm_string (TopSchema)')
        return None

    if cdm_job.errors:
        log.debug('## cdm_job.errors: %s', cdm_job.errors )
        return None

    workflow = cdm_job.data['cdm'][0]['workflow']

    log.debug('## converted workflow (after Schema) contains: %s', workflow )

    if 'workflow_type' in workflow.keys():
        workflow_type = workflow['workflow_type']
        log.debug( '## found workflow.type: %s', workflow_type )
    else:
        workflow_type = 'unknown'
        log.debug( '## unknown workflow_type' )

    if workflow_type == 'fiberJob':
        log.debug( '##-- fiberJob adaptor --##' )
        job = { 'job_number' : workflow['workorder_id'],
                'customer_name' : workflow['customer_info']['company'],
                'technician_id' : workflow['tech_info']['tech_id'],
                'contractor_id' : workflow['contractor_id'],
                'workflow_id' : workflow['workflow_id']
                }

        # Add any customer-specific workOrderAttributes key-value pairs into an appropriate
        # customer-specific object (taken from workOrderSource)
        if 'workorder_source' in workflow.keys():
            workorder_attribs_key = workflow['workorder_source']
            # This is where we add the needed orange object, but keep the code "Orange-agnostic"
            # All workOrderAttributes get placed in an object that uses the workOrderSource as the key
            wo_object = {}
            wo_object[ workorder_attribs_key ] = workflow['workorder_attributes']
            log.debug('## woA source: %s', workorder_attribs_key )

        # Note any Job Manager-specific attributes (which are held under the 'attibutes' object)
        if 'attributes' in workflow.keys():
            for attrib_key in workflow['attributes']:
                if attrib_key == 'testLocationA':
                    job['test_location'] =  get_genfield_value( workflow['attributes'], attrib_key )
                elif attrib_key == 'testLocationB':
                    job['test_location_b'] =  get_genfield_value( workflow['attributes'], attrib_key )
                elif attrib_key == 'cableId':
                    job['cable_id'] =  get_genfield_value( workflow['attributes'], attrib_key )
                elif attrib_key == 'jobComments':
                    job['job_comments'] =  get_genfield_value( workflow['attributes'], attrib_key )
                elif attrib_key == 'workOrderAttribHeaders':
                    # StrataSync only want to provide the **order** (in an array) of workOrderAttributes here (not ideal)
                    # so we have to jump through some more hoops to try to assemble what we need
                    # Again, needed for Orange, but make it work regardless of customer
                    # (Add this piece of the puzzle to the wo_object).
                    log.debug('## attribute: %s, value = %s', attrib_key, get_genfield_value( workflow['attributes'], attrib_key) )
                    if 'workorder_source' in workflow.keys():
                        wo_object[ workorder_attribs_key ][ 'headerOrder' ] = get_genfield_value( workflow['attributes'], attrib_key )
                else:
                    job[ attrib_key ] = get_genfield_value( workflow['attributes'], attrib_key )
                    log.debug('## unexpected attribute: %s, value = %s', attrib_key, get_genfield_value( workflow['attributes'], attrib_key) )

        # Only write orange-related info here (once we have assembled required information
        if 'workorder_source' in workflow.keys():
            job['wo_attribs'] = dumps( wo_object ) # Try to serialize the whole object into a string
            log.debug('## wo_attribs : %s', job['wo_attribs'] )

        # path of the logo
        if 'logoInfo' in workflow['domain_attributes'].keys() and \
            'path' in workflow['domain_attributes']['logoInfo'].keys():
            job['logo'] = workflow['domain_attributes']['logoInfo']['path']
        elif 'logo' in workflow['domain_attributes'].keys():
            job['logo'] = workflow['domain_attributes']['logo']
        else:
            job['logo'] = ''

        job['test_plan'] = get_old_cdm2_test_plan(cdm_job)
    # StrataSync does not yet support a viaviJob test type
    elif workflow_type == 'viaviJob' or workflow_type == 'ona800Job' : # Going forward this is the common CDM2.1 path for all
        log.debug( '##-- viaviJob adaptor --##' )
        job = { 'job_number' : workflow['workorder_id'],
                'customer_name' : workflow['customer_info']['company'],
                'technician_id' : workflow['tech_info']['tech_id'],
                'contractor_id' : workflow['contractor_id'],
                'workflow_id' : workflow['workflow_id']
                }

        # Add any customer-specific workOrderAttributes key-value pairs into an appropriate
        # customer-specific object (taken from workOrderSource)
        if 'workorder_source' in workflow.keys():
            workorder_attribs_key = workflow['workorder_source']
            # This is where we add the needed orange object, but keep the code "Orange-agnostic"
            # All workOrderAttributes get placed in an object that uses the workOrderSource as the key
            wo_object = {}
            wo_object[ workorder_attribs_key ] = workflow['workorder_attributes']
            log.debug('## woA source: %s', workorder_attribs_key )

        # Note any Job Manager-specific attributes (CDM2.1)
        if 'job_manager_attributes' in workflow.keys():
            log.debug( '##-- jobManagerAttributes --##' )

            for attrib_key in workflow['job_manager_attributes']:
                if attrib_key == 'testLocationA':
                    job['test_location'] =  get_job_attr_value( workflow['job_manager_attributes'], attrib_key ) # ['value']
                    log.debug( '##-- jobManagerAttributes test_location (A): %s', job['test_location'] )
                elif attrib_key == 'testLocation': # Metro doesn't use Loc A & Loc B
                    job['test_location'] =  get_job_attr_value( workflow['job_manager_attributes'], attrib_key ) # ['value']
                    log.debug( '##-- jobManagerAttributes test_location: %s', job['test_location'] )
                elif attrib_key == 'testLocationB':
                    job['test_location_b'] =  get_job_attr_value( workflow['job_manager_attributes'], attrib_key )
                    log.debug( '##-- jobManagerAttributes test_location (B): %s', job['test_location_b'] )
                elif attrib_key == 'cableId':
                    job['cable_id'] =  get_job_attr_value( workflow['job_manager_attributes'], attrib_key )
                    log.debug( '##-- jobManagerAttributes cable_id: %s', job['cable_id'] )
                elif attrib_key == 'jobComments':
                    job['job_comments'] =  get_job_attr_value( workflow['job_manager_attributes'], attrib_key )
                    log.debug( '##-- jobManagerAttributes job_comments: %s', job['job_comments'] )
                else:
                    job[ attrib_key ] = get_job_attr_value( workflow['job_manager_attributes'], attrib_key )
                    log.debug('## unexpected jobManagerAttributes: %s, value = %s', attrib_key, get_job_attr_value( workflow['job_manager_attributes'], attrib_key) )

        # Note any other attributes (CDM2.1)
        if 'attributes' in workflow.keys():
            for attrib_key in workflow['attributes']:
                if attrib_key == 'workOrderAttribHeaders':
                    # StrataSync only want to provide the **order** (in an array) of workOrderAttributes here (not ideal)
                    # so we have to jump through some more hoops to try to assemble what we need
                    # Again, needed for Orange, but make it work regardless of customer
                    # (Add this piece of the puzzle to the wo_object).
                    log.debug('## viaviJob attribute: %s, value = %s', attrib_key, get_genfield_value( workflow['attributes'], attrib_key) )
                    if 'workorder_source' in workflow.keys():
                        wo_object[ workorder_attribs_key ][ 'headerOrder' ] = get_genfield_value( workflow['attributes'], attrib_key )
                else:
                    job[ attrib_key ] = get_genfield_value( workflow['attributes'], attrib_key )
                    log.debug('## unexpected attribute: %s, value = %s', attrib_key, get_genfield_value( workflow['attributes'], attrib_key) )

        # Only write orange-related info here (once we have assembled required information
        if 'workorder_source' in workflow.keys():
            job['wo_attribs'] = dumps( wo_object ) # Try to serialize the whole object into a string
            log.debug('## wo_attribs : %s', job['wo_attribs'] )

        # path of the logo
        if 'logoInfo' in workflow['domain_attributes'].keys() and \
            'path' in workflow['domain_attributes']['logoInfo'].keys():
            job['logo'] = workflow['domain_attributes']['logoInfo']['path']
        elif 'logo' in workflow['domain_attributes'].keys():
            job['logo'] = workflow['domain_attributes']['logo']
        else:
            job['logo'] = ''

        job['test_plan'] = get_cdm2_test_plan(cdm_job)

    elif workflow_type == 'metroJob':
        # original Metro Jobs :
        log.debug( '##-- metroJob adaptor --##' )
        job = { 'job_number' : workflow['workorder_id'],
                'customer_name' : workflow['customer_info']['company'],
                'technician_id' : workflow['tech_info']['tech_id'],
                'workflow_id' : workflow['workflow_id'],
                'test_location': workflow['workorder_attributes']['testLocation']['value'],
                'logo': workflow['domain_attributes']['logo']
                }
        job['test_plan'] = get_cdm_test_plan(cdm_job)

    else:
        log.debug( '##-- unknown Job type --##' )
        job = { 'job_number' : workflow['workorder_id'],
                'customer_name' : workflow['customer_info']['company'],
                'technician_id' : workflow['tech_info']['tech_id'],
                'workflow_id' : workflow['workflow_id'],
                }
        job['test_plan'] = get_cdm2_test_plan(cdm_job)

    job['cdm_string'] = input_json

    return job

def get_cdm_test_plan(cdm_job):
    test_plan = []
    try:
        tests = cdm_job.data['cdm'][0]['tests']
    except KeyError:
        return test_plan
    for test in tests:
        planned_test = {'test_type' : test['type'],
                        'status' : get_status(test),
                        'test_label' : test['label']
                        }

        log.debug('## (Metro) planned test type= %s', planned_test['test_type'] )

        refinfolist = []
        if 'attributes' in test.keys():
            attributes = test['attributes']
            for attr in attributes.keys():
                refinfo = {}
                refinfo['key'] = attr
                refinfo['value'] = get_genfield_value(attributes[attr], 'value')
                refinfolist.append(refinfo)

        planned_test['reference_info'] = refinfolist
        planned_test['procedures'] = get_genfield_value(test, 'procedures')
        planned_test['config'] = get_genfield_value(test, 'configuration')
        planned_test['test_data'] = []
        test_plan.append(planned_test)
    return test_plan

# Needed to bridge the gap between Native Job Manager v1's expected Planned_Test schema & Otdr CDM v2
def cdm2_config_keys_dictionary( cdm1_key ):
    switch={
        'config_file':'config_file',
        'otdr_topology':'otdr_topology',
        'alarms':'alarms', # Not managed yet
        'launch_cable':'launch_cable', # Not managed yet
        'fiber':'fiber', # Not managed yet
        'setups':'setups', # Not managed yet
        'schema_version':'schema_version' # Not managed yet
        }

    return switch.get( cdm1_key,"invalid_config_key")

def get_old_cdm2_test_plan(cdm_job):
    '''
    common code is *not* working for Fiber,
    so need to return to what worked before (this)
    '''
    test_plan = []
    try:
        tests = cdm_job.data['cdm'][0]['tests']
    except KeyError:
        return test_plan
    for test in tests:
        planned_test = {'test_type' : test['type'],
                        'status' : get_status(test),
                        'test_label' : test['label']
                        }

        log.debug('## (Fiber) Planned test type = %s', planned_test['test_type'] )

        refinfolist = []
        stinfolist = []

        if 'test_locations' in test.keys():
            test_loc = test['test_locations'][0]
            log.debug('##   testLocations: %s', test_loc )

            if 'attributes' in test_loc.keys():

                for attrib_key in test_loc['attributes']:

                    if attrib_key == 'ref_info':
                        allRefInfo = get_genfield_value( test_loc['attributes'], attrib_key )
                        log.debug('##   referenceInfo attribute= %s', allRefInfo )
                        for riKey in allRefInfo:
                            refinfo = {}
                            refinfo['key'] = riKey
                            refinfo['value'] = get_genfield_value( allRefInfo, riKey )
                            refinfolist.append(refinfo)

                    elif attrib_key == 'sub_type_info':
                        if planned_test['test_type'] != 'OTDR':
                            # Bodge (OTDR not storing this info yet - breaks 'matches' test)
                            # Exclude OTDR subTypeInfo for now
                            allSubTypeInfo = get_genfield_value( test_loc['attributes'], attrib_key )
                            log.debug('##   subTypeInfo attribute= %s', allSubTypeInfo )
                            for stKey in allSubTypeInfo:
                                stinfo = {}
                                stinfo['key'] = stKey
                                stinfo['value'] = get_genfield_value( allSubTypeInfo, stKey )
                                stinfolist.append(stinfo)

                        log.debug('##   stinfolist: %s', stinfolist )
                    else:
                        log.debug('##   unknown attribute')


        log.debug('##   refinfolist: %s', refinfolist )
        planned_test['reference_info'] = refinfolist
        planned_test['sub_type_info'] = stinfolist
        planned_test['procedures'] = test.get('procedures', "")

        planned_test['workflow_id'] = test.get('workflow_id', 0)
        log.debug('##   test workflowId= %d', planned_test['workflow_id'] )

        # Deal with CDM configs (and move into Job Mgr config format (until we rework Job Mgr to be CDM natively?)
        if 'configuration' in test.keys():
            log.debug('##   configuration: %s', get_genfield_value(test, 'configuration') )
            pt_config = {}
            pt_config['test_type'] = test['type']
            allConfig = get_genfield_value(test, 'configuration')
            for configKey in allConfig:
                if configKey == 'otdr_settings':
                    # This is all getting very Fiber-specific :(
                    allOtdrConfig = get_genfield_value(allConfig, 'otdr_settings')
                    for otdrKey in allOtdrConfig:
                        log.debug('##      config key translation: %s', cdm2_config_keys_dictionary( otdrKey ) )
                        pt_config[ cdm2_config_keys_dictionary( otdrKey ) ] = get_genfield_value( allOtdrConfig, otdrKey )
                        
            planned_test['config'] = pt_config
            log.debug('##   config: %s', planned_test['config'])


        planned_test['test_data'] = []
        
        log.debug('## get_old_cdm2_test_plan: planned_test = %s', planned_test )

        test_plan.append(planned_test)

    return test_plan

def get_cdm2_test_plan(cdm_job):
    test_plan = []
    try:
        tests = cdm_job.data['cdm'][0]['tests']
    except KeyError:
        return test_plan
    for test in tests:
        planned_test = {'test_type' : test.get('type'),
                        'status' : get_status(test),
                        'test_label' : test.get('label')
                        }

        log.debug('## Planned test type = %s', planned_test.get('test_type') )

        refinfolist = []
        stinfolist = []

        if 'test_locations' in test.keys():
            test_loc = test['test_locations'][0]
            log.debug('##   testLocations: %s', test_loc )

            if 'attributes' in test_loc.keys():

                for attrib_key in test_loc['attributes']:

                    if attrib_key == 'ref_info':
                        allRefInfo = get_genfield_value( test_loc['attributes'], attrib_key )
                        log.debug('##   referenceInfo attribute= %s', allRefInfo )
                        for riKey in allRefInfo:
                            refinfo = {}
                            refinfo['key'] = riKey
                            refinfo['value'] = get_genfield_value( allRefInfo, riKey )
                            refinfolist.append(refinfo)

                    elif attrib_key == 'sub_type_info':
                        if planned_test['test_type'] != 'OTDR':
                            # Bodge (OTDR not storing this info yet - breaks 'matches' test)
                            # Exclude OTDR subTypeInfo for now
                            allSubTypeInfo = get_genfield_value( test_loc.get('attributes'), attrib_key )
                            log.debug('##   subTypeInfo attribute= %s', allSubTypeInfo )
                            for stKey in allSubTypeInfo:
                                stinfo = {}
                                stinfo['key'] = stKey
                                stinfo['value'] = get_genfield_value( allSubTypeInfo, stKey )
                                stinfolist.append(stinfo)

                        log.debug('##   stinfolist: %s', stinfolist )
                    else:
                        log.debug('##   unknown attribute')


        log.debug('##   refinfolist: %s', refinfolist )
        planned_test['reference_info'] = refinfolist
        planned_test['sub_type_info'] = stinfolist
        planned_test['procedures'] = test.get('procedures', "")

        planned_test['workflow_id'] = test.get('workflow_id', 0)
        log.debug('##   test workflowId= %d', planned_test.get('workflow_id') )

        planned_test['config'] = test.get('configuration', {})
        log.debug('##   test.configuration: %s', test.get('configuration'))
        log.debug('##   config: %s', planned_test.get('config'))
        
        #if 'test_type' not part of the 'config' object add it in
        if 'test_type' not in planned_test['config']:
            planned_test['config']['test_type'] = test.get('type')
            log.debug('##   add in job mgr testType: %s', planned_test['config']['test_type'] )

        if 'test_type' in planned_test['config'] and planned_test['config']['test_type'] == 'fromCdm':
            planned_test['config']['test_type'] = test.get('type')
            log.debug('##   changed fromCdm to CDM testType: %s', planned_test['config']['test_type'])


        planned_test['test_data'] = []
        
        log.debug('## get_cdm2_test_plan: planned_test = %s', planned_test )

        test_plan.append(planned_test)

    return test_plan

def get_status(test):
    if ('status' in test):
        return test['status']
    else:
        return 'To Do'

def get_genfield_value(structure, value):
    if (value in structure):
        return structure[value]
    else:
        return ""

def get_job_attr_value(structure, value):
    if (value in structure):
        return structure[value]['value']
    else:
        return ""
