#!/bin/bash
#
# Prepare fs for Otu
#
# This script manage :
#   - /otu directory creation or deletion
#   - Debug mode for not switching part if otu doesn't start properly
#   - Symlink for Otu.ver to /acterna/release/bin/
#   - Ensure ot trusted user are removed in enhanced security mode
#   - Ensure smartotu log file are writable from apache
#

source /usr/lib/otu_functions

inspect_repair_unused_prtn() {
    #
    #  fsck error codes 4, 8, 12 and 128 indicate a corrupted partition
    EC_CODE_4=4
    EC_CODE_8=8
    EC_CODE_12=12
    EC_CODE_128=128
    DEVICE_5="/dev/mmcblk1p5"
    DEVICE_6="/dev/mmcblk1p6"
    #
    #  Get the device reference for the unused (unmounted) partition

    current_device_prtn=`findmnt --output source --noheadings --target /`
    if [[ $current_device_prtn == "$DEVICE_5" ]]; then
        unused_prtn_device=$DEVICE_6
    elif  [[ $current_device_prtn == "$DEVICE_6" ]]; then
        unused_prtn_device=$DEVICE_5
    else
        #
        # Device for the root fs mount point is not recognizable
        #
        echo "$(date +'%B %d %H:%M:%S') $(hostname) $0:  . Exiting device/partition recovery.  The Root mount point must be $DEVICE_5 or $DEVICE_6>" >> /var/log/user.log
    fi

    #
    #  Use fsck to inspect partition
    #  Options -f -n runs a check without running repairs.
    #  Recover only for fsck error codes "4" and "128"
    umount $unused_prtn_device 2>/dev/null
    res_scan_check=`fsck -f -n -M $unused_prtn_device`
    fsck_rc=$?
    echo  "$(date +'%B %d %H:%M:%S') $(hostname) $0:  . Checking $unused_prtn_device ::  fsck error code = $fsck_rc ." >> /var/log/user.log

    case $fsck_rc in
        $EC_CODE_4 | $EC_CODE_8 | $EC_CODE_12 | $EC_CODE_128)
            #
            #  Unused partion is not sane.
            #  Recover device partition within capabilities of fsck.
            #
            echo  "$(date +'%B %d %H:%M:%S') $(hostname) $0:  . Device $unused_prtn_device contains partition errors :: Initiating fsck recovery." >> /var/log/user.log
	    fsck -f -y $unused_prtn_device
            if [[ $? -eq 0 || $? -eq 1 ]]; then
                mount $unused_prtn_device /mnt/fs/unused_part  2>/dev/null
	        echo "$(date +'%B %d %H:%M:%S') $(hostname) $0:  . Recovery complete for device $unused_prtn_device ::  fsck repair return code = $?" >> /var/log/user.log
            else
	        echo "$(date +'%B %d %H:%M:%S') $(hostname) $0:  . Unable to repair $unused_prtn_device ::  fsck repair return code = $? ::  $unused_prtn_device is not mounted." >> /var/log/user.log
            fi
	;;

        *)
            #
            # Recovery procedure is not necessary
            #
            echo "$(date +'%B %d %H:%M:%S') $(hostname) $0:  . Partition recovery is NOT required for device $unused_prtn_device ." >> /var/log/user.log
        ;;
        esac
}

# If we are not Otu but /otu directory exists remove it (otu5000 to eotdrv2 conversion by manufacturing)
is_cotu_hw
HW_OTU=$?
is_otu_sw
SW_OTU=$?
[ "${HW_OTU}" == "0" ] && [ "${SW_OTU}" == "0" ] && [ -f /otu ] && rm -rf /otu

# Do the rest only if using otu
quit_if_not_using_otu

#
#  In OTU rescue mode only, check if unused partition
#  is corrupted and recover if possible.
#
otu_rescuemode=`tr -d '\000' < /proc/device-tree/board/rescue-mode`
if [[ ! -z $otu_rescuemode  &&  $otu_rescuemode == "yes" ]]; then
    echo "$(date +'%B %d %H:%M:%S') $(hostname) $0:  . OTU is in rescue mode; inspecting $unused_prtn_device if partition is sane." >> /var/log/user.log
    inspect_repair_unused_prtn
fi
#

# Create /acterna/user directory (temporary we have to remove this directory and use /user instead)
if [ ! -d /acterna/user ] ; then
        mkdir /acterna/user
fi
if [ ! -d /acterna/user/disk ] ; then
        mount --rbind /user /acterna/user
fi

# Create /Otu directory
if [ ! -L /otu ] ;  then
    if [ -d /user/harddisk ] ; then
		mkdir -p /user/harddisk/otu
        ln -s /user/harddisk/otu /otu        
    elif [ -d /user/disk ] && [ "${HW_OTU}" == "0" ] && [ "${SW_OTU}" == "1" ] ; then
		mkdir -p /user/disk/otu
        ln -s /user/disk/otu /otu
    fi
else
	# Enforce link to otu to the harddisk if present
	if [ -d /user/harddisk ] && [ "$(readlink /otu)" != /user/harddisk/otu ] ; then
        rm /otu
        mkdir -p /user/harddisk/otu
        ln -sf /user/harddisk/otu /otu
	fi
	# Enforce /otu removal if harddisk is not present with otu hardware (do not work using flash)
	mountpoint /user/harddisk > /dev/null 2>&1
	HAS_NO_HARDDISK=$?
	if [ "${HAS_NO_HARDDISK}" == "1" ]  && [ "${HW_OTU}" == "1" ] && [ -h /otu ] ; then
	     rm -rf /otu
	fi
fi

# If the file debug exists we don't want to switch partition if Otu doesn't start
[ -f /acterna/storage/otu/debug ] && ! [ -f /otu/release/tmp/first_boot_after_upgrade.tmp ] && otu_confirm_start.sh

# if there is a otu.ver file move it to /etc/acterna/
if [ -f /acterna/release/bin/Otu.ver ] && [ ! -h /acterna/release/bin/Otu.ver ] ; then
	mv /acterna/release/bin/Otu.ver /etc/acterna/Otu.ver
fi

# Create symlink to Otu.ver file if doesn't exist
if [ ! -f /acterna/release/bin/Otu.ver ] && [ -f /etc/acterna/Otu.ver ] ; then
	ln -sf /etc/acterna/Otu.ver /acterna/release/bin/Otu.ver
fi

# In enhance security mode we have to check every boot that users have not been added during an upgrade
if [ "$(/sbin/fw_printenv -n Enhanced_Security)" = "true" ] ; then
    bash /usr/lib/jdsu/scripts/update_user_list.sh "DEL" > /dev/null 2>&1
fi

# Log partition we are using in otu log
if [ -d /otu/log ] ; then
    source /etc/otu/otuparts.sh
    CURRENT_DATE=$(date '+%Y-%m-%d %H:%M:%S.%3N')
    echo "${CURRENT_DATE} : Boot using partition ${CURRENT_PART}" >> /otu/log/boot.log
    if [ -f /proc/device-tree/board/rescue-mode ] && [ $(cat /proc/device-tree/board/rescue-mode) == "yes" ] ; then
        echo "${CURRENT_DATE} : Started in rescue mode" >> /otu/log/boot.log
        # We want to try again to boot normally at next reboot
        otu_confirm_start.sh
    fi
fi

# Temporary : Create log file required for smartotu
touch /var/log/otu_query.log
touch /var/log/smart_otu.log
chown www-data:www-data /var/log/otu_query.log
chown www-data:www-data /var/log/smart_otu.log

exit 0
