<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\serviceshelper\system;

use app\sharedmemory\SMTMemoryManager;

use app\sharedmemory\SMTIMemorySupport;

use app\util\SMTLogger;

use app\database\SMTDatabaseException;

use app\database\SMTSmartOtuDB;

use app\util\SMTResultSetDto;

use app\util\SMTUtil;

use app\serviceshelper\media\SMTEmail;

use app\util\SMTInvalidValueException;

/**
 * System configuration Dto saved in database
 * 
 * @author Sylvain Desplat
 * @SMTClassTableNameAnnotation(tablename='system_configuration', tablekey='id') 
 */
class SMTSystemConfigurationDto extends SMTResultSetDto implements SMTIMemorySupport
{
    /**
     * Time zone is in DST period. boolean type not available in Sqlite
     * 
     * @var integer 
     */
    const TIMEZONE_DST_PERIOD = 1;
    
    /**
     * Time zone is not in DST period. boolean type not available in Sqlite
     *
     * @var integer
     */
    
    const TIMEZONE_NO_DST_PERIOD = 0;
    
    /**
     * Files are added to emails. boolean type not available in Sqlite
     *
     * @var integer
     */
    const EMAIL_ATTACHED_FILE = 1;
    
    /**
     * Files are not added to emails. boolean type not available in Sqlite
     *
     * @var integer
     */    
    const EMAIL_NO_ATTACHED_FILE = 0;
    
    const FETCH_SYSTEM_CONFIGURATION = "SELECT * from system_configuration";
    
    /**
     * Only one record created in sql script SMTSql01p0002
     */
    const ID = 0;
    
    /**
     * System configuration identifier
     * @var integer
     */
    protected $id;
    
    /**
     * @var string
     */
    protected $configurationName = "default";
    
    /**
     * Time zone in seconds
     * 
     * @var integer
     */
    protected $timezoneSec;
    
    /**
     * Time zone name (GMT+01:00)
     * 
     * @var string
     */
    protected $timezoneName;
    
    /**
     * Whether time zone is in DST period
     *
     * @var integer
     */
    protected $isTimezoneDst = self::TIMEZONE_NO_DST_PERIOD;
    
    /**
     * Backup in database of email smtp server
     * @var string
     */
    protected $smtpServer = "";
    
    /**
     * Backup in database of email smtp server port
     * @var int
     */
    protected $smtpPort = SMTEmail::DEFAULT_SMTP_PORT;
    
    /**
     * Backup in database of email smtp server security protocol
     * @var string
     */
    protected $smtpSecurityProtocol = SMTEmail::DEFAULT_SMTP_SECURITY_NONE;

    /**
     * Backup in database of email smtp server login
     * @var string
     */    
    protected $smtpLogin = "";

    /**
     * Backup in database of email smtp server password
     * 
     * @var string
     */    
    protected $smtpPassword = "";
        
    /**
     * Whether files must be attached to Emails
     * 
     * @var integer
     */
    protected $isSmtpAttachFile = FALSE;
    
    /**
     * Returns 0: It is forbidden to add a new configuration to the default one except if it has been removed:
     * Should not be called: there is only one configuration!
     *
     * @param SMTSmartOtuDB $dbConnection
     *
     * @return unique configuration ID = 0
     */
    public function generateId( SMTSmartOtuDB $dbConnection )
    {    	
        SMTLogger::getInstance()->trace( "Try to add a new configuration whereas only one is allowed: ".$this->getDtoClassName(), SMTLogger::ERROR );    	 
    	return self::ID;
    }    
    
    /**
     * System configuration id
     *
     * @return integer
     */
    public function getId()
    {
    	return $this->id;
    }   
    
    /**
     *
     * @param integer $timezoneSec
     */
    public function setTimezoneSec( $timezoneSec )
    {
        if ( !isset($timezoneSec) || $timezoneSec === NULL || !SMTUtil::isInt( $timezoneSec ) )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "timezoneSec", $timezoneSec );
        }
        $this->timezoneSec = SMTUtil::convertToInt( $timezoneSec );
    }
    
    /**
     *
     * @return integer
     */
    public function getTimezoneSec()
    {
        return $this->timezoneSec;
    }

    /**
     *
     * @param string $timezoneName
     */
    public function setTimezoneName( $timezoneName)
    {
    	$this->timezoneName = $timezoneName;
    }
    
    /**
     * Whether the timezone is in the DST period
     *
     * @return boolean Whether the timezone is in the DST period
     */
    public function isTimezoneDst()
    {
    	return ( $this->isTimezoneDst == self::TIMEZONE_DST_PERIOD );
    }
    
    /**
     * Set whether the timezone is in the DST period
     * 
     * @param boolean $isTimezoneDst
     */
    public function setTimezoneDst( $isTimezoneDst )
    {
    	$this->isTimezoneDst = ( $isTimezoneDst? self::TIMEZONE_DST_PERIOD : self::TIMEZONE_NO_DST_PERIOD );
    }
    
    /**
     *
     * @return string
     */
    public function getTimezoneName()
    {
    	return $this->timezoneName;
    }
    
    /**
     *
     * @param string $name
     */
    public function setConfigurationName( $name)
    {
    	$this->configurationName = $name;
    }
    /**
     *
     * @return integer
     */
    public function getConfigurationName()
    {
    	return $this->configurationName;
    }    
    
    /**
     * @param string
     */
    public function setSmtpServer( $smtpServer )
    {
    	$this->smtpServer = $smtpServer;
    }
    
    /**
     * @return string
     */
    public function getSmtpServer()
    {
    	return $this->smtpServer;
    }
    
    /**
     * @param int
     */
    public function setSmtpPort( $smtpPort )
    {
        if ( !isset($smtpPort) || $smtpPort === NULL || !SMTUtil::isInt( $smtpPort ) || $smtpPort < 0 )
        {
        	throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "smtpPort", $smtpPort );
        }    
    	$this->smtpPort = SMTUtil::convertToInt( $smtpPort );
    }    
    
    /**
     * @return int
     */
    public function getSmtpPort()
    {
    	return $this->smtpPort;
    }    
    
    /**
     * @param string
     */
    public function setSmtpSecurityProtocol( $smtpSecurityProtocol )
    {
    	if ( !isset($smtpSecurityProtocol) || ($smtpSecurityProtocol === NULL) || ($smtpSecurityProtocol == "") )
    	{
    		throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, NULL, "smtpSecurityProtocol", $smtpSecurityProtocol );
    	}
    	$this->smtpSecurityProtocol = $smtpSecurityProtocol;
    }    
    
    public function getSmtpSecurityProtocol()
    {
        return $this->smtpSecurityProtocol;
    }
    
    /**
     * @param string
     */
    public function setSmtpLogin( $smtpLogin )
    {
    	$this->smtpLogin = $smtpLogin;
    }
    
    /**
     * @return string
     */
    public function getSmtpLogin()
    {
    	return $this->smtpLogin;
    }
    
    /**
     * @param string
     */
    public function setSmtpPassword( $smtpPassword )
    {
    	$this->smtpPassword = $smtpPassword;
    }
    
    /**
     * @return string
     */
    public function getSmtpPassword()
    {
    	return $this->smtpPassword;
    }
        
    /**
     * Whether files are attached to Emails
     *
     * @return boolean whether files (OTDR traces) are attached to Emails
     */
    public function isSmtpAttachFile()
    {
    	return ( $this->isSmtpAttachFile == self::EMAIL_ATTACHED_FILE );
    }
    
    /**
     * Set whether files (OTDR traces) are attached to Emails
     *
     * @param boolean $isSmtpAttachFile
     */
    public function setSmtpAttachFile( $isSmtpAttachFile )
    {
    	$this->isSmtpAttachFile = ( $isSmtpAttachFile? self::EMAIL_ATTACHED_FILE : self::EMAIL_NO_ATTACHED_FILE );
    }    
    
    /**
     * Fetch the system configuration from APC cache or from database if not found in cache
     *
     * @param \SQLite3 $dbConnection
     *
     * @return SMTSystemConfigurationDto
     */
    public static function fetch( \SQLite3 $dbConnection )
    {
        $systemConfigurationCache = SMTMemoryManager::fetch( SMTSystemConfigurationDto::getClass(), SMTSystemConfigurationDto::ID );
        if ( $systemConfigurationCache == NULL )
        {
        	$query = self::FETCH_SYSTEM_CONFIGURATION;
        	$statement = $dbConnection->prepare( $query );
        	 
        	if ( $statement === FALSE )
        	{
        		SMTLogger::getInstance()->trace( sprintf("Prepare Fetch statement failed. Query: %s.", $query), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
        	}
        	else
        	{
        		SMTLogger::getInstance()->trace( sprintf("Prepare Fetch statement Successfull. Query: %s.", $query), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        	}
        	 
        	$resultSet = $statement->execute();
        	 
        	if ( $resultSet === FALSE )
        	{
        		SMTLogger::getInstance()->trace( sprintf("Fetch failed. Query: %s.", $query), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
        	}
        	else
        	{
        		SMTLogger::getInstance()->trace( sprintf("Fetch successfull for Query: %s.", $query), SMTLogger::DEBUG);
        	}
        
        	$systemConfigurationCache = parent::getSingleInstanceFromResultSet($resultSet, get_class() );
        	
        	//system configuration doesn't exist, create it
        	if ( $systemConfigurationCache == NULL )
        	{
        	    $systemConfigurationCache = new SMTSystemConfigurationDto();
        	}
        	
        	//update cache
        	SMTMemoryManager::saveDto( $systemConfigurationCache );
        }
        
        return $systemConfigurationCache;
    }
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }
    
    
    
    
    /*************************************************
     * OVERRIDES SAVE... FOR STORAGE IN APC memory
     * 
     *************************************************/
    
    /**
     * Save one property of the current dto instance in a database table mapping the class hierarchy
     * and in APC shared memory.
     *
     * @param SMTSmartOtuDB $dbConnection
     * @param string $propertyName The property name to update
     *
     * @throws SMTDatabaseException
     */
    public function updateSingleProperty( SMTSmartOtuDB $dbConnection, $propertyName )
    {
    	parent::updateSingleProperty($dbConnection, $propertyName);
    	SMTMemoryManager::saveDto( $this );
    }
    
    /**
     * Save current dto instance in database tables mapping the class hierarchy and generate objectId primary key if needed
     * and in APC shared memory
     *
     * @param SMTSmartOtuDB $dbConnection ( NULL by default to implement SMTIMemorySuport save method without argument)
     *
     * @throws SMTDatabaseException, SMTAPCMemoryException
     */
    public function save( SMTSmartOtuDB $dbConnection = NULL )
    {
        if ( $dbConnection != NULL )
        {
            parent::save($dbConnection);
        }
    	SMTMemoryManager::saveDto( $this );
    }
    
    /**
     * Delete dto
     *
     * @param \SQLite3 $dbConnection ( NULL by default to implement SMTIMemorySuport delete method without argument)
     * @throws SMTDatabaseException, SMTAPCMemoryException
     */
    public function delete( SMTSmartOtuDB $dbConnection = NULL )
    {
        if ( $dbConnection != NULL )
        {
        	parent::delete($dbConnection);
        }
        SMTMemoryManager::deleteDto( $this );
    }    
    
    /**
     * Returns the dto class name.
     *
     * @return string the dto class name.
     */
    public static function getClass()
    {
    	return __CLASS__;
    }
    
    /**
     * Otu settings cache is never obsolete: it must not be purged
     *
     * @return FALSE
     */
    public function isObsolete()
    {
    	return FALSE;
    }
    
    /**
     * No obsolescence for those data: returns current time.
     *
     * @see isObsolete()
     * @return number Timestamp of the last APC memory access (used to handle obsolescence of dtos in cache)
     */
    public function getLastAccess()
    {
    	return microtime(TRUE);
    }
    
    /**
     * No obsolescence for those data: do nothing
     *
     * @see isObsolete()
     *
     */
    public function touchAccess()
    {
    }     
    
}

?>