<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\serviceshelper\system;

use app\http\SMTSessionDirectoryUtil;

use app\serviceshelper\SMTServiceHelper;

use app\serviceshelper\otdr\SMTOtdrTraceFileUtil;

use app\database\SMTDatabaseUtil;

use app\util\SMTLogger;

/**
 * Utility class for SmartOTU
 * 
 * @author Sylvain Desplat
 */
class SMTSmartOtu extends SMTServiceHelper
{
    /**
     * Check whether the Otu has no link and no test and is not a new database.
     * 
     * @param $isNewDatabase boolean Whether the database is a new database not yet created. 
     * 
     * @return boolean whether the database and all smartOtu files can be cleared: the Otu has no link and no test and is not a new database.
     */
    public static function isSmartOtuToClear( $isNewDatabase )
    {
        $isSmartOtuToBeCleared = FALSE; 
        if ( $isNewDatabase != TRUE )
        {
            //retrieve monitoring tests reference detection traces
            $files = glob( SMTOtdrTraceFileUtil::OTU_MEASURE_REF_DET_DIR."*".SMTOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION );         
            //search last backup file
            if ( $files != FALSE && !empty( $files ) )
            {
                $isSmartOtuToBeCleared = TRUE;
            }
            else 
            {
                //retrieve monitoring tests reference localisation traces
                $files = glob( SMTOtdrTraceFileUtil::OTU_MEASURE_REF_LOC_DIR."*".SMTOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION );
                //search last backup file
                if ( $files != FALSE && !empty( $files ) )
                {
                    $isSmartOtuToBeCleared = TRUE;
                }
            }
        }
        return $isSmartOtuToBeCleared;
    }
    
    /**
     * If the Otu has no link and no test, ensure that database which only stores alarms is empty and all smartOtu files (otdr traces, sessions...).
     * Used in case of mode change between SmartOtu and ONMSi: ensure that when switching to SmartOtu mode, the database is empty.
     * If database is new, don't clear it, it is not yet created!
     * 
     * @param $isNewDatabase Whether the database is a new database not yet created. 
     * 
     * @return Whether the database (and all smartOtu directories) has been cleaned and recreated .
     */
    public static function cleanUpSmartOtuIfNeeded( $isNewDatabase )
    {
        //search if OTU has links and tests
    	$isSmartOtuToBeCleared = self::isSmartOtuToClear( $isNewDatabase );
    	if ( $isSmartOtuToBeCleared  )
    	{
    	    SMTLogger::getInstance()->trace("WARNING: Database must be cleared!");
    	    $databaseFilename = SMTDatabaseUtil::CUSTOM_SMARTOTU_DATABASE_DIR."/".SMTDatabaseUtil::DEFAULT_DATABASE_FILE_NAME;
    	    
    	    $success = FALSE;
    	    
    	    //drop database if it exists
    	    if (file_exists( $databaseFilename ))
    	    {
    	        $count = 0;
    	        while ( !$success && $count < 3)
    	        {
    	    	    $success = unlink( $databaseFilename );
    	    	    usleep(10000);
    	    	    $count++;
    	        }
    	        if ( !$success )
    	        {
    	            //don't throw exception, try continuing using current database
    	            SMTLogger::getInstance()->trace( sprintf( "CRITICAL: Database file: %s couldn't be deleted and cleaned!", $databaseFilename ), SMTLogger::ERROR );
    	        }
    	        else
    	        {
    	            SMTLogger::getInstance()->trace("WARNING: Database successfully cleared!");
    	        }
    	    }
    	    else
    	    {
    	        SMTLogger::getInstance()->trace("No Database file to clear!");
    	        $success = TRUE;
    	    }

    	    //cleanup SmartOTU alarm traces:
    	    SMTLogger::getInstance()->trace("WARNING: Remove SmartOTU alarm traces!");
    	    self::removeCustomSmartOtuDirectoryIfNeeded( SMTOtdrTraceFileUtil::CUSTOM_SMARTOTU_ALARM_DIR );            
    	    
    	    //cleanup SmartOTU session directory:
    	    SMTSessionDirectoryUtil::removeSessionsDirectoriesIfNeeded( 0 );
    	    
    	    //recreate database if it could be deleted    	    
    	    return ( $success && SMTDatabaseUtil::createDatabaseFileIfNeeded() );
    	}
    	return FALSE;
    }

    /**
     * Remove the given smartotu directory.
     *
     * @param $dir String the directory to cleanup and remove
     */
    static function removeCustomSmartOtuDirectoryIfNeeded( $dir )
    {    
    	if ( is_dir( $dir ) )
    	{
    		$files = scandir($dir);
    	  
    		foreach ( $files as $file )
    		{
    			if ( $file != "." && $file != ".." )
    			{
    				if ( file_exists( $dir."/".$file) )
    				{
    					unlink( $dir."/".$file );
    				}
    			}
    		}
    		 
    		//remove dir
    		rmdir( $dir );
    	}
    }
    
}

?>