<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of Viavi Solution
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from Viavi Solution
// Copyright Viavi Solution2019. All rights reserved.
// *********************************************************
namespace app\services\api;

use app\services\security\SMTAccessControl;
use app\services\SMTIService;
use app\services\SMTService;
use app\parser\SMTOtuApi;
use app\http\SMTHttpResponse;
use app\error\SMTRestException;
use app\util\SMTInvalidValueException;
use app\serviceshelper\monitoring\SMTLinkTest;


/**
 * Public API service to retrieve public urls.
 *
 * @author Sylvain Desplat
 */
class SMTApiUrlService extends SMTService implements SMTIService
{
	const HOME_URL = "/permalink/home/%s";
	const SETUP_URL = "/permalink/setup/%s";
	const LAST_ACQUISITION_URL = "/permalink/lasttestacquisition/%d/%s";
	const TEST_SETUP_URL = "/permalink/reference/measurement/%d/%s";
	const ALARM_URL = "/permalink/alarm/%d/%s";
	
	/**
	 *
	 * @url GET /home
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getHomeUrl()
	{
		$url ="";
		
		try
		{
			$token = $this->sendReceive( SMTOtuApi::getApiToken() );
			$url = sprintf(self::HOME_URL, $token);
			
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve monitoring summary url');
		}
		
		$this->getContext()->closeSession();
		
		return $url;
	}
	
	/**
	 *
	 * @url GET /setup
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getSetupUrl()
	{
		$url ="";
		
		try
		{
			$token = $this->sendReceive( SMTOtuApi::getApiToken() );
			$url = sprintf(self::SETUP_URL, $token);
			
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve otu setup url');
		}
		
		$this->getContext()->closeSession();
		
		return $url;
	}
	
	/**
	 *
	 * @url GET /lastacquisition/{portNumber}
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getLastAcquisitionUrl($portNumber)
	{
		$url ="";
		
		try
		{
			$linkHelper = new SMTLinkTest();
			$linkHelper->setContext( $this->getContext() );
			$portCount = $linkHelper->getPortsCount();
			
			if ( intval($portNumber)> $portCount)
			{
				throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, "portNumber > port count", "portNumber", $portNumber);	
			}
			else if ( intval($portNumber) <= 0 )
			{
				throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, "portNumber <= 0", "portNumber", $portNumber);
				
			}
			$token = $this->sendReceive( SMTOtuApi::getApiToken() );
			$url = sprintf(self::LAST_ACQUISITION_URL, intval($portNumber), $token);
			
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve last acquisition url');
		}
		
		$this->getContext()->closeSession();
		
		return $url;
	}
	
	/**
	 *
	 * @url GET /test/{portNumber}
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getMonitoringTestUrl($portNumber)
	{
		$url ="";
		
		try
		{
			$linkHelper = new SMTLinkTest();
			$linkHelper->setContext( $this->getContext() );
			$portCount = $linkHelper->getPortsCount();
			
			if ( intval($portNumber)> $portCount)
			{
				throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, "portNumber > port count", "portNumber", $portNumber);
			}
			else if ( intval($portNumber) <= 0 )
			{
				throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, "portNumber <= 0", "portNumber", $portNumber);
				
			}
			$token = $this->sendReceive( SMTOtuApi::getApiToken() );
			$url = sprintf(self::TEST_SETUP_URL, intval($portNumber), $token);
			
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve monitoring test url');
		}
		
		$this->getContext()->closeSession();
		
		return $url;
	}
	
	/**
	 *
	 * @url GET /alarm/{portNumber}
	 *
	 * @access protected
	 * @class SMTAccessControl {@requires install}
	 */
	function getAlarmUrl($portNumber)
	{
		$url ="";
		
		try
		{
			$linkHelper = new SMTLinkTest();
			$linkHelper->setContext( $this->getContext() );
			$portCount = $linkHelper->getPortsCount();
			
			if ( intval($portNumber)> $portCount)
			{
				throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, "portNumber > port count", "portNumber", $portNumber);
			}
			else if ( intval($portNumber) <= 0 )
			{
				throw new SMTInvalidValueException( SMTInvalidValueException::INVALID_VALUE, "portNumber <= 0", "portNumber", $portNumber);
				
			}
			$token = $this->sendReceive( SMTOtuApi::getApiToken() );
			$url = sprintf(self::ALARM_URL, intval($portNumber), $token);
			
		}
		catch ( \Exception $e )
		{
			$this->getContext()->getLogger()->traceException($e);
			SMTHttpResponse::handleApiError($this->getContext(), SMTRestException::INTERNAL_SERVER_ERROR, SMTApiDto::FAILURE, 'Failed to retrieve alarm url');
		}
		
		$this->getContext()->closeSession();
		
		return $url;
	}
}