<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\parser;

use Luracast\Restler\RestException;

use app\error\SMTRestException;

use app\util\SMTException;

/**
 * Otu parser socket exception
 * 
 * @author Sylvain Desplat
 *
 */
class SMTSocketException extends SMTException
{
    /**
     * Couldn't retrieve OTU PORT
     * @var string
     */
    const ERROR_OTU_PORT = "ERROR_RETRIEVING_OTU_PORT";
    /**
     * Couldn't create socket
     * @var string
     */    
    const ERROR_CREATING = "ERROR_CREATING_SOCKET";
    /**
     * Could create socket but couldn't open the connection
     * 
     * @var string
     */    
    const ERROR_OPENING = "ERROR_OPENING_SOCKET_CONNECTION";
    /**
     * Could create socket but couldn't set connection properties
     * @var string
     */
    const ERROR_SETTING_OPTIONS = "ERROR_SETTING_SOCKET_OPTIONS";    
    
    /**
     * Connection is opened but write fails
     * @var string
     */
    const ERROR_WRITTING = "ERROR_WRITTING_SOCKET";
    /**
     * Connection is opened but read fails
     * @var string
     */    
    const ERROR_READING = "ERROR_READING_SOCKET";
    
    /**
     * Command sent to OTU application failed
     * @var string
     */    
    const COMMAND_FAILURE = "COMMAND_FAILURE";
    /**
     * Command sent to OTU application timed-out
     * @var string
     */    
    const COMMAND_TIMEOUT = "COMMAND_TIMEOUT";
    /**
     * Command sent to OTU application timed-out
     * @var string
     */
    const ERROR_OTU_AUTHENTIFICATION_FAILURE = "AUTHENTIFICATION_FAILURE_OTU_PARSER";
    
    /**
     * Command sent to OTU application timed-out
     * @var int
     */
    const ERROR_ACQUIRE_LOCK = "FAILURE_ACQUIRE_LOCK_CREATE_SOCKET";
    
    /**
     * OTU application is not available
     * @var string
     */    
    const ERROR_OTU_APPLICATION_NOT_AVAILABLE = "ERROR_OTU_APPLICATION_NOT_AVAILABLE";    

    /**
     * OTU application is not yet ready to process request (Otu is starting)
     * @var string
     */
    const ERROR_OTU_APPLICATION_STARTING = "ERROR_OTU_APPLICATION_STARTING";
    
    /**
     * OTU application is not available to process request because it is rebooting
     * @var string
     */
    const ERROR_OTU_APPLICATION_REBOOTING = "ERROR_OTU_APPLICATION_REBOOTING";
    
    /**
     * OTU application is not available to process request because it is upgraded
     * @var string
     */
    const ERROR_OTU_APPLICATION_UPGRADE = "ERROR_OTU_APPLICATION_UPGRADE";    
    
    /**
     * OTU application is not available to process request because it is in rescue
     * @var string
     */
    const ERROR_OTU_APPLICATION_RESCUE = "ERROR_OTU_APPLICATION_RESCUE";
    
    /**
     * Default socket Error
     * @var string
     */
    const SOCKET_ERROR = "SOCKET_ERROR";
    
    /**
     *
     * @param string $code error code among SMTSocketException error codes
     * @param string $message detailed message
     */    
    function __construct( $code = NULL, $message = NULL )
    {
    	parent::__construct( $code, $message, self::decodeHTMLCode( $code ) );
    }
    
    /**
     * Get error codes label for tracing without error codes.
     * @param string $code The exception error code (SMTSocketException)
     *
     * @return int
     */
    public function getErrorLabel()
    {
        return self::translateErrorLabel( $this->getErrorCode() );
    }
    
    /**
     * Get error codes label for tracing without localization.
     * @param int $code The exception error code (SMTSocketException)
     * 
     * @return string
     */
    public static function translateErrorLabel( $errorCode )
    {      
        return str_replace("_", " ", $errorCode);
    }
    
    /**
     * Translates exception error code
     * 
     * @param string $code The exception error code (SMTSocketException)
     * 
     * @return string the exception message
     */
    public function decodeMessage( $code )
    {
        $errorMessage = MSG_SOCKET_ERROR;
        
        switch( $code )
        {
            case self::ERROR_CREATING:
            case self::ERROR_OPENING:            
            case self::ERROR_SETTING_OPTIONS:
            case self::ERROR_ACQUIRE_LOCK:
                $errorMessage = MSG_ERROR_OPENING_SOCKET;
                break;
            case self::ERROR_WRITTING:
                $errorMessage = MSG_ERROR_WRITTING_SOCKET;
                break;                
            case self::ERROR_READING:
            	$errorMessage = MSG_ERROR_READING_SOCKET;
            	break;
            case self::COMMAND_FAILURE:
            	$errorMessage = MSG_PARSER_COMMAND_FAILURE;
            	break;
        	case self::COMMAND_TIMEOUT:
        		$errorMessage = MSG_PARSER_COMMAND_TIMEOUT;
        		break;
        	case self::ERROR_OTU_PORT:
        	    $errorMessage = MSG_ERROR_PORT_OTU;
        	    break;    		 
        	case self::ERROR_OTU_AUTHENTIFICATION_FAILURE:
        	    $errorMessage = MSG_ERROR_OTU_AUTHENTIFICATION_FAILURE;
        	    break;
        	case self::ERROR_OTU_APPLICATION_NOT_AVAILABLE:
        	    $errorMessage = MSG_ERROR_OTU_APPLICATION_NOT_AVAILABLE;
        	    break;
        	case self::ERROR_OTU_APPLICATION_STARTING:
        	    $errorMessage = MSG_ERROR_OTU_APPLICATION_STARTING;
        	    break;
        	case self::ERROR_OTU_APPLICATION_REBOOTING:
        	    $errorMessage = MSG_ERROR_OTU_APPLICATION_REBOOTING;
        	    break;        	
    	    case self::ERROR_OTU_APPLICATION_UPGRADE:
    	    	$errorMessage = MSG_ERROR_OTU_APPLICATION_UPGRADE;
    	    	break;
    	    case self::ERROR_OTU_APPLICATION_RESCUE:
    	        $errorMessage = MSG_ERROR_OTU_APPLICATION_RESCUE;
    	        break;
            default:
                break;
        }
        return $errorMessage;
    }
    
    /**
     * Translates exception error code in HTML error code
     *
     * @param string $code The exception error code (SMTSocketException)
     *
     * @return integer the HTML error code
     */
    public static function decodeHTMLCode( $code )
    {
        if ( array_key_exists( $code, RestException::$codes ) )
        {
            $htmlErrorCode = $code;
        }
        else 
        {
        	$htmlErrorCode = SMTRestException::INTERNAL_SERVER_ERROR;
        
        	switch( $code )
        	{
        		case self::ERROR_OPENING:
        		case self::ERROR_SETTING_OPTIONS:
        		case self::COMMAND_TIMEOUT:
        		case self::ERROR_OTU_PORT:
        		case self::ERROR_OTU_APPLICATION_NOT_AVAILABLE:
        		case self::ERROR_OTU_APPLICATION_RESCUE:
        		case self::ERROR_OTU_APPLICATION_REBOOTING:
        		case self::ERROR_OTU_APPLICATION_STARTING:
        		case self::ERROR_OTU_APPLICATION_UPGRADE:
        		    $htmlErrorCode = SMTRestException::SERVICE_UNAVAILABLE;
        			break;
        		case self::ERROR_WRITTING:
        		case self::ERROR_READING:
        		case self::COMMAND_FAILURE:
        			$htmlErrorCode = SMTRestException::INTERNAL_SERVER_ERROR;
        			break;    
        		default:
        		    $htmlErrorCode = SMTRestException::INTERNAL_SERVER_ERROR;
        			break;
        	}
        }
    	return $htmlErrorCode;
    }    
}

?>