<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\database\patch;

use app\serviceshelper\alarm\SMTAlarmResourceCode;

use app\serviceshelper\alarm\SMTAlarmSeverity;

use app\util\SMTLogger;

use app\database\SMTSmartOtuDB;

use app\database\SMTSqlPatch;

/**
 * First datatabase patch to create the alarm model of SmartOtu
 * Database version 1
 * Patch number 1
 *
 * @author Sylvain Desplat
 */
class SMTSql01p0001 extends SMTSqlPatch
{  
//patches:
    const SQL_CREATE_DATABASE_PATCH =
    "CREATE TABLE IF NOT EXISTS database_patch
        (
            patch_number INTEGER NOT NULL,
            db_version INTEGER NOT NULL,
            type TEXT NOT NULL DEFAULT '%s',
            description TEXT NOT NULL,
            date_created INTEGER NOT NULL DEFAULT 0
    )";
    
    static function getCreateDatabasePatchSql()
    {
    	return sprintf(self::SQL_CREATE_DATABASE_PATCH, SMTSqlPatch::SQL_PATCH_TYPE_UPDATE );    
    }
    
    const SQL_CREATE_ALARM_SEQUENCE =
    "CREATE TABLE IF NOT EXISTS alarm_sequence
     (
            sequence INTEGER NOT NULL DEFAULT 0
     )";    
    
    static function getCreateAlarmSequenceSql()
    {
    	return self::SQL_CREATE_ALARM_SEQUENCE;
    }
    
    const SQL_CREATE_ALARM_EVENT_SEQUENCE =
    "CREATE TABLE IF NOT EXISTS alarm_event_sequence
     (
            sequence INTEGER NOT NULL DEFAULT 0
     )";    
    
    static function getCreateAlarmEventSequenceSql()
    {
    	return self::SQL_CREATE_ALARM_EVENT_SEQUENCE;
    }       

    const SQL_INSERT_ALARM_SEQUENCE =
    "INSERT INTO alarm_sequence(sequence) VALUES(0)";
    
    static function getInsertAlarmSequenceSql()
    {
    	return self::SQL_INSERT_ALARM_SEQUENCE;
    }    

    const SQL_INSERT_ALARM_EVENT_SEQUENCE =
    "INSERT INTO alarm_event_sequence(sequence) VALUES(0)";
    
    static function getInsertAlarmEventSequenceSql()
    {
    	return self::SQL_INSERT_ALARM_EVENT_SEQUENCE;
    }    

    const SQL_CREATE_DATABASE_PATCH_UNIQUE_INDEX =
    "CREATE UNIQUE INDEX IF NOT EXISTS database_patch_uk ON database_patch (patch_number, db_version, type)";
    
    static function getCreateDatabasePatchUniqueIndexSql()
    {
    	return self::SQL_CREATE_DATABASE_PATCH_UNIQUE_INDEX;
    }
    
    const SQL_CREATE_OPTICAL_ALARM =
    "CREATE TABLE IF NOT EXISTS optical_alarm 
    (
        id INTEGER NOT NULL PRIMARY KEY,
        test_id INTEGER NOT NULL,
        last_update INTEGER NOT NULL DEFAULT 0,
        discriminator TEXT NOT NULL,
        specific_problem_code TEXT NOT NULL,
        alarm_type TEXT NOT NULL DEFAULT '%s',
        severity INTEGER NOT NULL DEFAULT %d,
        clear_status INTEGER NOT NULL DEFAULT %b
     )";
    
    static function getCreateOpticalAlarmSql()
    {
        return sprintf( self::SQL_CREATE_OPTICAL_ALARM, SMTAlarmResourceCode::OTU_OPTICAL_ALARM_CODE_TEXT, SMTAlarmSeverity::SEVERITY_NONE, FALSE );
    }
    
    const SQL_CREATE_OPTICAL_ALARM_DISCRIMNATOR_UNIQUE_INDEX =
    "CREATE UNIQUE INDEX IF NOT EXISTS optical_alarm_discriminator_uk ON optical_alarm (discriminator)";
    
    static function getCreateOpticalAlarmDiscriminatorUniqueIndexSql()
    {
    	return self::SQL_CREATE_OPTICAL_ALARM_DISCRIMNATOR_UNIQUE_INDEX;
    }
    
    const SQL_CREATE_OPTICAL_ALARM_EVENT =
    "CREATE TABLE IF NOT EXISTS optical_alarm_event
     (
        id INTEGER NOT NULL PRIMARY KEY,
        alarm_id INTEGER NOT NULL,
        time INTEGER NOT NULL DEFAULT 0,
        first_marker_deviation_db REAL DEFAULT 0,
        first_marker_deviation_valid INTEGER NOT NULL DEFAULT %b,
        link_loss_deviation_db REAL DEFAULT 0,
        link_loss_deviation_valid INTEGER NOT NULL DEFAULT %b,
        problem_confirmed INTEGER NOT NULL DEFAULT %b,
        severity INTEGER NOT NULL DEFAULT %d,
        FOREIGN KEY(alarm_id) REFERENCES optical_alarm(id) ON DELETE CASCADE
     )";    
    
    static function getCreateOpticalAlarmEventSql()
    {
    	return sprintf( self::SQL_CREATE_OPTICAL_ALARM_EVENT, FALSE, FALSE, FALSE, SMTAlarmSeverity::SEVERITY_NONE );
    }    
    
    const SQL_CREATE_ATTENUATION_OPTICAL_ALARM_EVENT =
    "CREATE TABLE IF NOT EXISTS attenuation_optical_alarm_event
     (
        id INTEGER NOT NULL PRIMARY KEY,
        distance_m REAL DEFAULT 0,
        distance_valid INTEGER NOT NULL DEFAULT %b,
        level_db REAL DEFAULT 0,
        level_valid INTEGER NOT NULL DEFAULT %b,
        FOREIGN KEY(id) REFERENCES optical_alarm_event(id) ON DELETE CASCADE
     )";    
    
    static function getCreateAttenuationOpticalAlarmEventSql()
    {
    	return sprintf( self::SQL_CREATE_ATTENUATION_OPTICAL_ALARM_EVENT, FALSE, FALSE );
    }    
    
    //WARNING add foreign key constraint on optical_alarm table to reference the last alarm event
    //WARNING MUST ADD the column because ALTER TABLE cannont be used to add a constraint
    //WARNING CANNOT SET NOT NULL CONSTAINT ON FOREIGN KEY TO ALLOW INSERT AN ALARM WITHOUT ITS EVENT WHICH MUST BE INSERTED AFTER (EVENT REFERENCES ALSO THE ALARM)
    //"ALTER TABLE optical_alarm ADD current_alarm_event_id INTEGER CHECK(current_alarm_event_id is not null) REFERENCES optical_alarm_event(id)";
    const SQL_CREATE_OPTICAL_ALARM_OPTICAL_ALARM_EVENT_FK =
    "ALTER TABLE optical_alarm ADD current_alarm_event_id INTEGER REFERENCES optical_alarm_event(id)";
    
    static function getCreateOpticalAlarmOpticalAlarmEventFkSql()
    {
    	return self::SQL_CREATE_OPTICAL_ALARM_OPTICAL_ALARM_EVENT_FK;
    }
    
    const SQL_CREATE_SYSTEM_ALARM =
    "CREATE TABLE IF NOT EXISTS system_alarm
     (
        id INTEGER NOT NULL PRIMARY KEY,
        last_update INTEGER NOT NULL DEFAULT 0,
        discriminator TEXT NOT NULL,
        specific_problem_code TEXT NOT NULL,
        additional_text TEXT NOT NULL DEFAULT %s,
        alarm_type TEXT NOT NULL DEFAULT %s,
        severity INTEGER NOT NULL DEFAULT %d,
        clear_status INTEGER NOT NULL DEFAULT %b,
        specific_problem_id INTEGER NOT NULL, 
        otu_additional_info_id TEXT
     )";    
    
    static function getCreateSystemAlarmSql()
    {
    	return sprintf( self::SQL_CREATE_SYSTEM_ALARM, "none", SMTAlarmResourceCode::OTU_SYSTEM_ALARM_CODE_TEXT, SMTAlarmSeverity::SEVERITY_NONE, FALSE );
    }
    
    const SQL_CREATE_SYSTEM_ALARM_DISCRIMNATOR_UNIQUE_INDEX =
    "CREATE UNIQUE INDEX IF NOT EXISTS system_alarm_discriminator_uk ON system_alarm (discriminator)";
    
    static function getCreateSystemAlarmDiscriminatorUniqueIndexSql()
    {
    	return self::SQL_CREATE_SYSTEM_ALARM_DISCRIMNATOR_UNIQUE_INDEX;
    }
    
    const SQL_CREATE_SYSTEM_ALARM_EVENT =
    "CREATE TABLE IF NOT EXISTS system_alarm_event
     (
        id INTEGER NOT NULL PRIMARY KEY,
        alarm_id INTEGER NOT NULL,
        time INTEGER NOT NULL DEFAULT 0,
        severity INTEGER NOT NULL DEFAULT %d,
        FOREIGN KEY(alarm_id) REFERENCES system_alarm(id) ON DELETE CASCADE
     )";    
    
    static function getCreateSystemAlarmEventSql()
    {
    	return sprintf( self::SQL_CREATE_SYSTEM_ALARM_EVENT, SMTAlarmSeverity::SEVERITY_NONE );
    }    
    
    //WARNING add foreign key constraint on system_alarm table to reference the last alarm event
    //WARNING MUST ADD the column because ALTER TABLE cannont be used to add a constraint
    //WARNING cannot SET NOT NULL CONSTRAINT on current_alarm_event_id foreign key, because event must be created after (event references the alarm)
    //ALTER TABLE system_alarm ADD current_alarm_event_id INTEGER CHECK(current_alarm_event_id is not null) REFERENCES system_alarm_event(id)
    const SQL_CREATE_SYSTEM_ALARM_SYSTEM_ALARM_EVENT_FK =
    "ALTER TABLE system_alarm ADD current_alarm_event_id INTEGER REFERENCES system_alarm_event(id)";
    
    static function getCreateSystemAlarmSystemAlarmEventFkSql()
    {
    	return self::SQL_CREATE_SYSTEM_ALARM_SYSTEM_ALARM_EVENT_FK;
    }
        
    function __construct()
    {
        SMTLogger::getInstance()->trace( sprintf("Execute patch %s.", __CLASS__ ), SMTLogger::INFO );
        
    	//retrieve patch version
        $className = basename(__FILE__, '.php');
        preg_match_all('!\d+!', $className, $matches);
        
        $this->dbVersion = $matches[0][0];
        $this->dbPatch = $matches[0][1];    	
    }
    
    /**
     * @return SMTSql01p0001
     */    
    public static function getInstance()
    {        
        return new SMTSql01p0001();
    }
    
    /**
     * Create database model on an empty database.
     *
     * @see \app\database\SMTSqlPatch::execute()
     */    
    public function execute( SMTSmartOtuDB $db )
    {
        $success = $this->disableForeignKeys($db);
        self::handleExecutePatchError($db, SMTSmartOtuDB::getEnableForeignKey( FALSE ), $success, __FILE__, __METHOD__, __LINE__ );
    
    	$query = self::SQL_CREATE_DATABASE_PATCH;
    	$success = $db->execWithTrace($query, SMTLogger::INFO );    
    	self::handleExecutePatchError($db, $query, $success, __FILE__, __METHOD__, __LINE__ );
    	
    	$query = self::getCreateOpticalAlarmSql();
    	$success = $db->execWithTrace($query, SMTLogger::INFO );    	
        self::handleExecutePatchError($db, $query, $success, __FILE__, __METHOD__, __LINE__ );
        
        $query = self::getCreateOpticalAlarmEventSql();
    	$success = $db->execWithTrace($query, SMTLogger::INFO );
    	self::handleExecutePatchError($db, $query, $success, __FILE__, __METHOD__, __LINE__ );
    	 
    	$query = self::getCreateAttenuationOpticalAlarmEventSql();
    	$success = $db->execWithTrace($query, SMTLogger::INFO );
    	self::handleExecutePatchError($db, $query, $success, __FILE__, __METHOD__, __LINE__ );    	 

        $query = self::getCreateOpticalAlarmOpticalAlarmEventFkSql();
    	$success = $db->execWithTrace($query, SMTLogger::INFO );
    	self::handleExecutePatchError($db, $query, $success, __FILE__, __METHOD__, __LINE__ );
        	    	 
        $query = self::getCreateSystemAlarmSql();
    	$success = $db->execWithTrace($query, SMTLogger::INFO );
    	self::handleExecutePatchError($db, $query, $success, __FILE__, __METHOD__, __LINE__ );
        	    	 
        $query = self::getCreateSystemAlarmEventSql();
    	$success = $db->execWithTrace($query, SMTLogger::INFO );
    	self::handleExecutePatchError($db, $query, $success, __FILE__, __METHOD__, __LINE__ );
        	    	 
        $query = self::getCreateSystemAlarmSystemAlarmEventFkSql();
    	$success = $db->execWithTrace($query, SMTLogger::INFO );
        self::handleExecutePatchError($db, $query, $success, __FILE__, __METHOD__, __LINE__ );		 
        	    
		$query = self::getCreateAlarmSequenceSql();
		$success = $db->execWithTrace($query, SMTLogger::INFO );
		self::handleExecutePatchError($db, $query, $success, __FILE__, __METHOD__, __LINE__ );
    	
		$query = self::getCreateAlarmEventSequenceSql();
		$success = $db->execWithTrace($query, SMTLogger::INFO );
		self::handleExecutePatchError($db, $query, $success, __FILE__, __METHOD__, __LINE__ );
    	
		$query = self::getInsertAlarmSequenceSql();
		$success = $db->execWithTrace($query, SMTLogger::INFO );
		self::handleExecutePatchError($db, $query, $success, __FILE__, __METHOD__, __LINE__ );
    		    	
		$query = self::getInsertAlarmEventSequenceSql();
		$success = $db->execWithTrace($query, SMTLogger::INFO );
		self::handleExecutePatchError($db, $query, $success, __FILE__, __METHOD__, __LINE__ );
    		    	
    	$success = $this->enableForeignKeys($db);
    	self::handleExecutePatchError($db, SMTSmartOtuDB::getEnableForeignKey( TRUE ), $success, __FILE__, __METHOD__, __LINE__ );
    	
	    $success = $this->addPatchTrace($db, 'create smartOTU alarm model');
	    self::handleExecutePatchError($db, SMTSqlPatch::SQL_TRACE_DATABASE_PATCH, $success, __FILE__, __METHOD__, __LINE__ );    	
    }
}
?>