<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\admin;    

// use app\util\SMTLogger;
use app\settings\SMTSmartOtuSettings;

/**
 * Handle the OTU mode: 
 *  - retrieve it from settings
 *  - store it in session variable
 * 
 * @author Sylvain Desplat
 *
 */
class SMTOtuMode
{        
    const OTU_MODE  = "otumode";
    const SMART_OTU = "SMARTOTU";
    const SMART_OEM = "SMARTOEM";
    const ONMSI     = "ONMSI";
    const ONMS      = "OTU";
    const OEM       = "OEM";
    const ROTAU     = "ROTAU";
    const RESCUE    = "RESCUE";
    const UFOM      = "UFOM";

    const DEFAULT_OTU_MODE = self::SMART_OTU;
    const SMART_OTU_LOGO = "SmartOtu158x38.png";
    const SMART_OEM_LOGO = "SmartOem164x38.png";
    const UFOM_LOGO      = "SmartOtuFM180x38.png";
    const OEM_LOGO       = "Oem73x38.png";
    const ROTAU_LOGO     = "Rotau111x38.png";
    const ONMS_LOGO      = "OtuOnms.png";
    const ONMSI_LOGO     = "OtuOnmsi.png";
    const OTU5000_ONMSI_LOGO     = "Otu5000Onmsi.png";
    const OTU5000_ROTAU_LOGO     = "Otu5000Rotau.png";
    const OTU8000_ONMSI_LOGO     = "Otu8000Onmsi.png";
    const OTU8000_ROTAU_LOGO     = "Otu8000Rotau.png";
    const RESCUE_LOGO    = "Rescue133x38.png";
    const SMART_OTU_DISPLAY = "SmartOTU";
    const SMART_OEM_DISPLAY = "SmartOEM";
    const ONMSI_DISPLAY     = "OTU ONMSi";
    const ONMS_DISPLAY      = "OTU ONMS";
    const OEM_DISPLAY       = "OEM";
    const ROTAU_DISPLAY     = "ROTAU";
    const RESCUE_DISPLAY    = "OTU RESCUE";
    const UFOM_DISPLAY      = "SmartOTU Fast Monitoring";
    
    
    /**
     * The supported OTU modes keys.
     *
     *
     * @return string The OTU mode key.
     */
    public static function getSupportedModes()
    {
        return array(self::SMART_OTU,self::SMART_OEM, self::ONMSI, self::ONMS, self::OEM, self::ROTAU, self::RESCUE, self::UFOM );
    }    
    
    /**
     * Whether the OTU mode is SmartOTU
     * @return boolean Whether the OTU mode is SmartOTU
     */
    public static function isSmartOtu()
    {
        return ( strcasecmp( self::getOtuMode() , self::SMART_OTU ) == 0 );
    }
    
    /**
     * Whether the OTU mode is SmartOTU or UFOM
     * @return boolean Whether the OTU mode is SmartOTU
     */
    public static function isSmartOtuFamily()
    {
    	return self::isSmartOtu() || self::isUFOM();
    }

    /**
     * Whether the OTU mode is SmartOEM
     * @return boolean Whether the OTU mode is SmartOEM
     */
    public static function isSmartOem()
    {
    	return ( strcasecmp( self::getOtuMode() , self::SMART_OEM ) == 0 );
    }
    
    /**
     * Whether the OTU mode is ONMSi
     * @return boolean Whether the OTU mode is ONMSi
     */
    public static function isONMSi()
    {
    	return ( strcasecmp( self::getOtuMode() , self::ONMSI ) == 0 );
    }
    
    /**
     * Whether the OTU mode is ONMS
     * @return boolean Whether the OTU mode is ONMS
     */
    public static function isONMS()
    {
    	return ( strcasecmp( self::getOtuMode() , self::ONMS ) == 0 );
    }
    
    /**
     * Whether the OTU mode is OEM
     * @return boolean Whether the OTU mode is OEM
     */
    public static function isROTAU()
    {
    	return ( strcasecmp( self::getOtuMode() , self::ROTAU ) == 0 );
    }
    
    /**
     * Whether the OTU mode is OEM
     * @return boolean Whether the OTU mode is OEM
     */
    public static function isOem()
    {
    	return ( strcasecmp( self::getOtuMode() , self::OEM ) == 0 );
    }  

    /**
     * Whether the OTU mode is RESCUE
     * @return boolean Whether the OTU mode is RESCUE
     */
    public static function isRESCUE()
    {
    	return ( strcasecmp( self::getOtuMode() , self::RESCUE ) == 0 );
    }
    
    /**
     * Whether the OTU mode is UFOM
     * @return boolean Whether the OTU mode is UFOM
     */
    public static function isUFOM()
    {
    	return ( strcasecmp( self::getOtuMode() , self::UFOM) == 0 );
    }
    
    /**
     * Whether the string value is an existing OTU mode
     *
     * @param string $mode
     * @return boolean
     */
    public static function isValid( $mode )
    {
    	if ( strcasecmp( $mode , self::SMART_OTU ) == 0  || 
  	         strcasecmp( $mode , self::SMART_OEM ) == 0  ||
	         strcasecmp( $mode , self::ONMSI ) == 0  ||
	         strcasecmp( $mode , self::ONMS ) == 0  ||
	         strcasecmp( $mode , self::OEM ) == 0  ||
	         strcasecmp( $mode , self::ROTAU ) == 0 || 
    	     strcasecmp( $mode , self::RESCUE ) == 0 ||
    	     strcasecmp( $mode , self::UFOM ) == 0
    	   )
    	{
    		return true;
    	}
    	return false;
    }
        
    /**
     * The OTU mode logo
     * 
     * @return string  The OTU mode logo.
     */
    public static function getOtuModeLogo()
    {
        if ( self::isSmartOtu() )
        {
            return self::SMART_OTU_LOGO;
        }
        else if ( self::isSmartOem() )
        {
        	return self::SMART_OEM_LOGO;
        }
        else if ( self::isONMSi() )
        {
        	return SMTOtuType::isCOtu_EOTDRV2()? self::OTU5000_ONMSI_LOGO : self::OTU8000_ONMSI_LOGO;
        }
        else if ( self::isONMS() )
        {
        	return self::ONMS_LOGO;
        }
        else if ( self::isOem() )
        {
        	return self::OEM_LOGO;
        }
        else if ( self::isROTAU() )
        {
            return SMTOtuType::isCOtu_EOTDRV2()? self::OTU5000_ROTAU_LOGO : self::OTU8000_ROTAU_LOGO;
        }
        else if ( self::isRESCUE() )
        {
        	return self::RESCUE_LOGO;
        }
        else if ( self::isUFOM() )
        {
        	return self::UFOM_LOGO;
        }
    }
    
    /**
     * The OTU mode display name
     *
     * @return string  The OTU mode display name.
     */
    public static function getOtuModeDisplay()
    {
    	if ( self::isSmartOtu() )
    	{
    		return self::SMART_OTU_DISPLAY;
    	}
    	else if ( self::isSmartOem() )
    	{
    		return self::SMART_OEM_DISPLAY;
    	}
    	else if ( self::isONMSi() )
    	{
    		return self::ONMSI_DISPLAY;
    	}
    	else if ( self::isONMS() )
    	{
    		return self::ONMS_DISPLAY;
    	}
    	else if ( self::isOem() )
    	{
    		return self::OEM_DISPLAY;
    	}
    	else if ( self::isROTAU() )
    	{
    		return self::ROTAU_DISPLAY;
    	}
    	else if ( self::isUFOM() )
    	{
    		return self::UFOM_DISPLAY;
    	}    	
    }
    
    /**
     * Set OTU mode in session
     *
     * @param string $mode the OTU Mode
     */
    public static function setOtuMode( $mode )
    {
    	if ( self::isValid( $mode ) )
    	{
    		$_SESSION[self::OTU_MODE] = $mode;
    	}
    }    
    
    /**
     * Return the OTU mode
     * 
     * @return string
     */
    public static function getOtuMode()
    {
        //update session current language
        if ( !isset($_SESSION[self::OTU_MODE] ) || $_SESSION[self::OTU_MODE] == NULL )
        { 
            $mode = self::retrieveOtuMode();
            if ( self::isValid( $mode ) )
                {
                self::setOtuMode( $mode );
                }
        }        
        
        $mode = isset($_SESSION[self::OTU_MODE])?$_SESSION[self::OTU_MODE]: self::DEFAULT_OTU_MODE;
        
        if ( self::isValid( $mode ) )
        {
            return $mode;
        }
        
        return self::DEFAULT_OTU_MODE;    
    }
    
    /**
     * Retrieve the OTU mode from settings
     *
     */
    private static function retrieveOtuMode()
    {	    	
    	return SMTSmartOtuSettings::getOtuMode();
    }    
    
    /**
     * Whether the OTU is in rescue mode
     * 
     * @return boolean
     */
    public static function queryIsRescueMode()
    {
        // is OTU in rescue mode
        $output = exec  ('sudo /sbin/get_otu_fs.sh');
        
        //test rescue mode
        $rescue = (strcasecmp( self::RESCUE, $output ) == 0)? true : false;
        
        return $rescue;
    }
    
}
?>