"""Module containing the the ProductSpecificBase class

This class is the base class for providing product specific functions
to rest api endpionts
"""
import logging

from rest_api.products.access_anywhere import AccessAnywhere
from rest_api.api.job_manager.schemas_viavi import make_job_schema
from rest_api.api.job_manager.cdm_schemas_viavi import make_cdm_schema
from rest_api.api import __files_api_version__
from rest_api.api import __job_manager_api_version__
from rest_api.api import __location_api_version__
from rest_api.api import __datetime_api_version__

from bottle import Bottle

log = logging.getLogger(__name__)


class ProductSpecific(object):
    """Base class for implementing default functionality for endpoints on the
    rest api

    Default methods do not interact with the system and enable happy path
    functionality of api routes
    """
    job_schema = make_job_schema()

    cdm_schema = make_cdm_schema()

    @classmethod
    def startup_side_effect(cls, db_file, report_dir, procedures_dir, jobs_dir):
        """Default side effect executed on startup of the rest api

        Args:
            db_file: the path to the database file used by product manager
        """
        log.debug('db_file = %s', db_file)
        log.debug('report_dir = %s', report_dir)
        log.debug('procedures_dir = %s', procedures_dir)
        log.debug('jobs_dir = %s', jobs_dir)

    @staticmethod
    def update_job_side_effect(job_info):
        """Default side effect executed if job state change from inactive to
        active or when top level job information changes
        """
        #log.debug('job_info = %s', job_info)
        log.debug('job updated (default - no additional actions)')

    @staticmethod
    def get_access_anywhere_code(servername):
        """Default function executed when an access anywhere code is requested

        args:
            servername (str): host name of the syetem
        returns:
            AccessAnywere: namedtupple with the access anywhere information
        """
        log.debug('servername = %s', servername)
        return AccessAnywhere(
            accessCode="None",
            connectStatus=False,
            hasIpAddress=False,
            errorMessage='This product does not support Smart Access Anywhere'
        )

    @staticmethod
    def make_instrument_info():
        """Default function to create information for the info endpoints

        returns:
            Instrument info dictionary
        ---
        infoVersion:
            type: number
            description: version of the info JSON
            example: 1
        apiSupport:
            type: array
            description: API versions supported by this instrument
            items:
              type: object
              required:
                - name
                - versions
              properties:
                name:
                  type: string
                  description: API package name
                  example: fileMgr
                versions:
                  type: number
                  description: API version(s) supported
                  example:
                    - 1
                    - 2
                    - 3
        assetInfo:
          type: object
          properties:
            serialNo:
              type: string
              description: serial number of the instrument
            assetType:
              type: string
              description: instrument family
            uniqueId:
              type: string
              description: concatenation of platform name and serial number
            model:
              type: string
              description: model of the instrument
            hwVersion:
              type: string
              description: hardware version of the instrument
            swVersion:
              type: string
              description: current software version of the instrument
            calibrationDate:
              type: string
              description: date of last calibration
            macAddress:
              type: string
              description: mac address of the instrument
            mfgDate:
              type: string
              description: manufacturing date of the instrument
            swOptions:
              items:
                type: object
                properties:
                  description:
                    type: string
                  name:
                    type: string
                  optionLicenseType:
                    type: string
                  expirationDate:
                    type: string
              type: array
              description: currently installed software options
            hwOptions:
              type: array of strings
              description: hardware options on the mainframe
            customFields:
              type: object
              properties:
                Challenge ID:
                  type: string
                Board Info:
                  type: string
        """
        info_version = {'infoVersion':1}

        api_support = {
            'apiSupport': [
#                 {
#                     'name': 'datetime',
#                     'versions': [__datetime_api_version__]
#                 },
#                 {
#                     'name': 'location',
#                     'versions': __location_api_version__[]
#                 },
#                 {
#                     'name': 'techinfo',
#                     'versions': [1]
#                 },
#                 {
#                     'name': 'filemgr',
#                     'versions': [__files_api_version__]
#                 },
#                 {
#                     'name': 'signature',
#                     'versions': [__files_api_version__]
#                 },
#                 {
#                     'name': 'testdata',
#                     'versions': [__files_api_version__]
#                 },
#                 {
#                     'name': 'jobmanager',
#                     'versions': [__job_manager_api_version__]
#                 },
#                 {
#                     'name': 'saa',
#                     'versions': [__job_manager_api_version__]
#                 },
#                {
#                    'name': 'workflow',
#                    'versions': [1]
#                }
            ]
        }

        asset_info = {
            'assetInfo': {
                'serialNo': '12345',
                'assetType': 'Off Target',
                'uniqueId': 'Off Target_12345',
                'model': 'Off Target 1',
                'hwVersion': '1.0',
                'swVersion': '1.0',
                'calibrationDate': 'N/A',
                'macAddress': 'N/A',
                'mfgDate': 'N/A',
                'swOptions': [],
                'hwOptions': [],
                'customFields': {},
            }
        }
        instrument_info = info_version.copy()
        instrument_info.update(api_support)
        instrument_info.update(asset_info)
        return instrument_info

    @staticmethod
    def update_time(time_value, location):
        """Default function to update the instrument time
        """
        print ('product_specific_base : update_time(time_value, location)')
        log.debug('time_value = %s', time_value)
        log.debug('location = %s', location)
        return True

    @staticmethod
    def update_location(geolocation):
        """Default function to update the instrument geolocation
        """
        log.debug('latitude = %s', geolocation['latitude'])
        log.debug('longitude = %s', geolocation['longitude'])
        log.debug('timestamp = %s', geolocation['timestamp'])
        log.debug('altitude = %s', geolocation['altitude'])
        log.debug('accuracy = %s', geolocation['accuracy'])
        log.debug('altitude_accuracy = %s', geolocation['altitude_accuracy'])
        return True

    @staticmethod
    def update_stratasync_info(tech_first_name, tech_last_name, tech_id, account_id, url):
        """Default function for updating stratasync information on the instrument
        """
        log.debug('tech_first_name = %s', tech_first_name)
        log.debug('test_last_name = %s', tech_last_name)
        log.debug('tech_id = %s', tech_id)
        log.debug('account_id = %s', account_id)
        log.debug('url = %s', url)
        return True

    @staticmethod
    def update_test_data_side_effect(test_plan_index,job_status_ratio):
        """Default function executed as a result of test data being updated
        """
        log.debug('test_plan_index = %s', test_plan_index)
        log.debug('jobs done:%s total jobs:%s', job_status_ratio['complete_count'],job_status_ratio['total_count'])
        return True

    @staticmethod
    def update_loaded_job_side_effect(is_job_active,job_status_ratio):
        """Default function executed as a result of
        a new job being loaded or a job being activated / deactivated
        """
        log.debug('is_job_active = %s', is_job_active)
        log.debug('jobs done:%s total jobs:%s', job_status_ratio['complete_count'],job_status_ratio['total_count'])
        return True

    @staticmethod
    def reset_firmware_upgrade_status():
        log.debug('reset_firmware_upgrade_status')
        return False

    @staticmethod
    def process_firmware_upgrade_status():
        log.debug('process_firmware_upgrade_status')
        return False

    @staticmethod
    def process_firmware_upgrade_post(request, Filename):
        log.debug('process_firmware_upgrade_post = {}'.format(Filename))
        return False

    @staticmethod
    def process_firmware_upgrade_put(request):
        log.debug('process_firmware_upgrade_put')
        return False

    @staticmethod
    def call_process_cdm_job():
        log.debug('call_process_cdm_job')
        return False

    @staticmethod
    def call_process_mta_workflow(cdm_job):
        log.debug('call_process_mta_workflow')
        return False

    #@staticmethod
    #def call_get_workorder_id_list():
    #    log.debug('call_get_workorder_id_list')
    #    return []

    #@staticmethod
    #def call_get_workorder(workorder_id_param):
    #    log.debug('call_get_workorder')
    #    return {}

    #@staticmethod
    #def call_put_active_workorder(active_workorder_json):
    #    log.debug('call_put_active_workorder')
    #    return False

    @staticmethod
    def call_notify_job_received(file_path, cdm):
        log.debug('call_notify_job_received - Default')
        return False

    @staticmethod
    def call_post_workorder_attachment(id, attachement_name, job_manager):
        log.debug('call_post_workorder_attachment')
        return False

    @staticmethod
    def call_get_active_workorder():
        log.debug('call_get_active_workorder')
        return ''

    #@staticmethod
    #def call_get_template_list():
    #    log.debug('call_get_template_list')
    #    return []

    @staticmethod
    def get_job_dir():
        jobs_dir = Bottle().config.get('rest-api.jobs_dir')
        return jobs_dir

    @staticmethod
    def get_job_extension():
        return '.job.json'

    @staticmethod
    def get_template_extension():
        return '.tpl.json'

    @staticmethod
    def verify_and_set_active_job(job_manager):
        pass

