"""Module containing base REST API configurations for Fiber products
"""
import logging
import threading
import traceback
import json

from rest_api.products import product_specific_base

from rest_api.products.mts_services import access_anywhere
from rest_api.products.mts_services import instrument_time
from rest_api.products.mts_services import instrument_location
from rest_api.products.mts_services import update_stratasync_info
from rest_api.products.mts_services import instrument_sync_files
from rest_api.products.mts_services import instrument_upgrade
from rest_api.products.mts_services import instrument_jobs
from rest_api.products.mts_services import update_asset_info

from rest_api.products.usc import asset_info
from rest_api.products.usc import file_sync
from rest_api.products.usc import cdm_sync
#from rest_api.products.usc import sync_files

from rest_api.products.config_schemas_fiber import ConfigSchema

from rest_api.api.job_manager.schemas_viavi import make_job_schema
from rest_api.api.job_manager.cdm_schemas_viavi import make_cdm_schema
from rest_api.api.job_manager.job_manager import JobManager

from rest_api.products import process_cdm_job_fiber


log = logging.getLogger(__name__)

timer_apply = True

WAIT_SECONDS = 30

class ProductSpecific(product_specific_base.ProductSpecific):
    """Class for all Fiber product specific functions
    """
    job_schema = make_job_schema(ConfigSchema)
    cdm_schema = make_cdm_schema(ConfigSchema)
    
    def start_timer_refresh():
        def myTimer():
            global timer_apply
            if timer_apply:
                instrument_sync_files.get_sync_list(False)
                
            timer_apply = True
            threading.Timer(WAIT_SECONDS, myTimer).start()
        
        timer = threading.Timer(WAIT_SECONDS, myTimer)
        timer.start()
        
    def cancel_next_refresh():
        global timer_apply
        timer_apply = False
        
    
    @classmethod
    def startup_side_effect(cls, db_file, report_dir, procedures_dir, jobs_dir):
        file_sync.register_report_path(report_dir)
        file_sync.register_procedures_path(procedures_dir)
        file_sync.register_job_files_path(jobs_dir)
        cdm_sync.register_cdm_filepath()
        
        ProductSpecific.start_timer_refresh()
        log.debug('## startup_side_effect (fiber_base): Jobs in %s', jobs_dir)


    @staticmethod
    def get_access_anywhere_code(servername):
        return access_anywhere.create_code(servername)

    @staticmethod
    def make_instrument_info():
        info_version = {'infoVersion':1}

        __version__ = '1.11.0' # Should we align this with latest Yocto project tag?
        __files_api_version__ = 1
        __job_manager_api_version__ = 1
        __location_api_version__ = 1
        __datetime_api_version__ = 1

        api_support = {
            'apiSupport': [
                {
                    'name': 'datetime',
                    'versions': [__datetime_api_version__]
                },
                {
                    'name': 'location',
                    'versions': [__location_api_version__]
                },
                {
                    'name': 'techinfo',
                    'versions': [1]
                },
                {
                    'name': 'filemgr',
                    'versions': [__files_api_version__]
                },
                {
                    'name': 'syncfiles',
                    'versions': [1]
                },
                {
                    'name': 'signature',
                    'versions': [1]
                },
                {
                    'name': 'testdata',
                    'versions': [__files_api_version__]
                },
#                 {
#                     'name': 'jobmanager',
#                     'versions': [__job_manager_api_version__]
#                 },
                {
                    'name': 'saa',
                    'versions': [1]
                },
                {
                    'name': 'instrumentlogin',
                    'versions': [1]
                },
                {
                    'name': 'firmware',
                    'versions': [1]
                #},
                #{
                    #'name': 'options',
                    #'versions': [1] 
                #},
                #{
                #    'name': 'workflow', #TBD vs job_manager (ATT???)
                #    'versions': [1]
                }
            ]
        }


        instrument_info = super(ProductSpecific, ProductSpecific).make_instrument_info()

        asset_info_usc = asset_info.make_asset_info()
        instrument_info['assetInfo'] = update_asset_info.update_asset_info(asset_info_usc)

        instrument_info.update(info_version)
        instrument_info.update(api_support)

        log.debug('## make_instrument_info: version                 = %s', __version__)
        log.debug('## make_instrument_info: files_api_version       = %s', __files_api_version__)
        log.debug('## make_instrument_info: location_api_version    = %s', __location_api_version__)
        log.debug('## make_instrument_info: datetime_api_version    = %s', __datetime_api_version__)
        log.debug('## make_instrument_info: api_support: %s', api_support)
        return instrument_info

    @staticmethod
    def update_time(time_value, location):
        return instrument_time.update_local_time(time_value, location)

    @staticmethod
    def update_location(geolocation):
        return instrument_location.update_location(geolocation)

    @staticmethod
    def update_stratasync_info(tech_first_name, tech_last_name, tech_id, account_id, url):
        return update_stratasync_info.update_stratasync_info(tech_id, account_id, url)
        
    @staticmethod
    def get_sync_list(sync_all):
        ProductSpecific.cancel_next_refresh()
        return instrument_sync_files.get_sync_list(sync_all)
        
    @staticmethod
    def set_sync_list_ack(full_path):
        instrument_sync_files.set_sync_list_ack(full_path)
        
    @staticmethod
    def call_process_cdm_job():
        process_cdm_job_fiber.main()
        
    @staticmethod
    def call_process_mta_workflow( cdm_json ):  ## difference relating to syncfiles or workflow usage - sync files, need notify, workflow without notify
        return(process_cdm_job_fiber.parse_cdm_job( cdm_json ))

    #@staticmethod
    #def call_get_workorder_id_list():
    #    return(instrument_jobs.get_workorder_id_list())

    #@staticmethod
    #def call_get_workorder(workorder_id_param):
    #    return(instrument_jobs.get_workorder(workorder_id_param))

    #@staticmethod
    #def call_put_active_workorder(active_workorder_json):
    #    return(instrument_jobs.put_active_workorder(active_workorder_json))

    @staticmethod
    def call_post_workorder_attachment(id, attachement_name, job_manager):
        return(instrument_jobs.post_workorder_attachment(id, attachement_name, job_manager))

    @staticmethod
    def call_get_active_workorder():
        return(instrument_jobs.get_active_workorder())
    
    #@staticmethod
    #def call_get_template_list():
    #    return(instrument_jobs.get_template_list())

    @staticmethod
    def call_notify_job_received(file_path, cdm_mta):
        
        try:
            top_schema = ProductSpecific.cdm_schema
            cdm = top_schema().loads(json.dumps({"cdm": [cdm_mta]})).data['cdm'][0]

            job_type = ''
        
            # Different actions are needed for ATT job notification.
            # We determine if a jobs is for ATT TRUE Fiber here:
            # if workorder_id's LABEL contains BAN/TN then deduce we have the very particular AT&T situation... 
            # No Job Mgr, but a Job to deal with !!!
            # Also expect 1st (& only) testType in the Test Plan == trueFiber
            try:
                workorderlabel = cdm['workflow']['workorder_label']
            except:
                workorderlabel = ""
     
            try:
                single_test_type = cdm['tests'][0]['type']
            except:
                single_test_type = ""
     
            try:
                workorder_type = cdm['workflow']['workflow_type']
            except:
                workorder_type = ""

            log.debug('## call_notify_job_received: [%s] [%s] [%s]', workorderlabel, single_test_type, workorder_type )
            if (workorderlabel == 'BAN' or workorderlabel == 'BAN/TN') and single_test_type == 'trueFiber' and workorder_type == 'viaviJob':
                # 'BAN' compatible with MTA4.2, 'BAN/TN' compatible with MTA 4.3
                log.debug('## call_notify_job_received -> (%s) Destined for ATT TRUE Fiber', file_path )
                job_type = 'ATT'
            else:
                log.debug('## call_notify_job_received -> (%s) Standard CDM Job heading for Job Manager', file_path )
                job_type = 'CDM'


            return( instrument_jobs.notify_job_received( file_path, job_type ) )

        except:
            print(traceback.format_exc())
            return False

    @staticmethod
    def get_sync_file_info(filepath, base_dir):
        return instrument_sync_files.get_sync_file_info(filepath, base_dir)
    
    @staticmethod
    def get_instrument_login_config():
        return update_stratasync_info.get_instrument_login_config()
    
    @staticmethod
    def get_instrument_login_credentials(loginConfig):
        asset_info_usc = asset_info.make_asset_info()
        
        fullLoginConfig = {
            'assetType': asset_info_usc.get('assetType', ''),
            'uniqueId': asset_info_usc.get('uniqueId', ''),
            'techId': loginConfig.get('techId', ''),
            'accountId': loginConfig.get('accountId', '')
        }
        return update_stratasync_info.get_instrument_login_credentials(fullLoginConfig)
        
    @staticmethod
    def reset_firmware_upgrade_status():
        instrument_upgrade.reset_firmware_upgrade_status()
        
    @staticmethod
    def process_firmware_upgrade_status():
        return instrument_upgrade.process_firmware_upgrade_status()
        
    @staticmethod
    def process_firmware_upgrade_post(request, Filename):
        return instrument_upgrade.process_firmware_upgrade_post(request, Filename)
        
    @staticmethod
    def process_firmware_upgrade_put(request):
        return instrument_upgrade.process_firmware_upgrade_put(request)

    @staticmethod
    def get_job_dir():
        return instrument_jobs.get_job_dir()

    @staticmethod
    def get_job_extension():
        return instrument_jobs.get_job_extension()

    @staticmethod
    def get_template_extension():
        return instrument_jobs.get_template_extension()
