"""
Module that contains schemas for the config information of different tests to
launch on the T-BERD/MTS-5800 product family
"""
from marshmallow import Schema, fields, validates_schema, ValidationError
from marshmallow.validate import OneOf

from .config_schema_common import ConfigBase, test_type_factory
from rest_api.products import config_schemas_caa as caa

#pylint: disable=no-self-use

def validate_ip_version(data):
    """
    Validates that a layer 3 or layer 4 test has a IP version
    """
    if (data['layer'] == 'L3') or (data['layer'] == 'L4'):
        if not data.get('ipversion'):
            raise ValidationError('Layer 3/4 test requires a IP version',
                                  field_names='ipVersion')

def validate_traffic_mode(data):
    """
    Validates that a layer 2 or layer 3 test has a traffic mode
    """
    if (data['layer'] == 'L2') or (data['layer'] == 'L3'):
        if not data.get('mode'):
            raise ValidationError('Layer 2/3 test requires a traffic mode',
                                  field_names='mode')


def config_factory(ui_path_hint):
    """Function to create a config file field containing

    Args:
        ui_path_hint (str): a path hint for the job manager web app to use as the base path
    """
    return fields.Str(
        required=False,
        load_from='configFile',
        dump_to='configFile',
        x_uiPathHint=ui_path_hint,
        description=(
            'the config or project file that needs to be loaded for the test at launch'
        )
    )


IP_VERSIONS = ('IPv4',
               'IPv6',
               None)

IP_VERSION_FIELD = fields.Str(required=False,
                              allow_none=True,
                              validate=OneOf(IP_VERSIONS,
                                             error="Not a valid IP version"),
                              load_from='ipVersion',
                              dump_to='ipVersion',
                              description='version of IP')


class CpriCheckConfigSchema(Schema):
    """
    Schema for the configuration information of a CPRI
    Check test
    """
    rates = ('Rates 1-7',
             '614.4M',
             '1228.8M',
             '2457.6M',
             '3072.0M',
             '4915.2M',
             '6144.4M',
             '9830.4M',
             '10137.6M')

    test_type = test_type_factory('CPRI Check')
    rate = fields.Str(required=True,
                      validate=OneOf(rates, error='Not a valid rate'),
                      dump_to='rate',
                      load_from='rate',
                      description='the rate of CPRI Check test to be launched')
    config_file = config_factory('/user/bert/tests/')

class CpriConfigSchema(Schema):
    """Schema for the configuration of a CPRI test"""
    rates = ("Rates 1-7", "Rates 8-9")
    modes = (
        "Layer 2 BERT Term",
        "Layer 2 BERT Mon/Thru",
        "Spectrum-High-Refresh Term",
        "Spectrum-High-Refresh Mon/Thru"
    )

    test_type = test_type_factory("CPRI")
    rate = fields.Str(
        required=True,
        validate=OneOf(rates, error="Not a valid rate"),
        dump_to="rate",
        load_from="rate",
        description="the rate of the CPRI test to be launched"
    )
    mode = fields.Str(
        required=True,
        validate=OneOf(modes, error="Not a valid mode"),
        dump_to="mode",
        load_from="mode",
        description="the mode of the CPRI test to be launched"
    )

class QuickCheckConfigSchema(Schema):
    """
    Schema for the configuration information of the QuickCheck test
    """
    interfaces = ('10/100/1000',
                  '100M',
                  '1GigE',
                  '10GigE',
                  '25GigE',
                  '40GigE',
                  '50GigE',
                  '100GigE',
                  '100GigE RS-FEC',
                  '400GigE')
    layers = ('L2',
              'L3')
    test_type = test_type_factory('QuickCheck')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces, error='Not a valid interface'),
                           load_from='interface',
                           dump_to='interface',
                           description='interface to launch the test')
    layer = fields.Str(required=True,
                       validate=OneOf(layers, error='Not a valid layer'),
                       load_from='layer',
                       dump_to='layer',
                       description='layer to launch the test')
    config_file = config_factory('/user/bert/tests/')

class PtpCheckConfigSchema(Schema):
    """
    Schema for the configuration information of the PTP Check test.
    """
    interfaces = ('1GigE', '10GigE', '25GigE')
    profiles = ('G.8275.1',
                'L4')
    test_type = test_type_factory('PTP Check')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces, error='Not a valid interface'),
                           dump_to='interface',
                           load_from='interface',
                           description='the interface of the PTP Check test to be launched')
    profile = fields.Str(required=True,
                         validate=OneOf(profiles, error='Not a valid profile'),
                         load_from='profile',
                         dump_to='profile',
                         missing='L4',
                         description='PTP Check profile',
                         config_file=config_factory('/user/bert/tests/'))

class OpticsSelfTestConfigSchema(Schema):
    """
    Schema for the configuration information of the Optics Self-test.
    """
    interfaces = ('OTU3',
                  'OTU4',
                  'OC-768',
                  'STM-256',
                  '10GigE',
                  '25GigE',
                  '40GigE',
                  '50GigE',
                  '100GigE',
                  '100GigE RS-FEC',
                  '400GigE')
    test_type = test_type_factory('Optics Self-Test')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces, error='Not a valid interface'),
                           dump_to='interface',
                           load_from='interface',
                           description='the interface of the Optics Self-Test to be launched')
    config_file = config_factory('/user/bert/tests/')

class CableTestConfigSchema(Schema):
    """
    Schema for the configuration information of Cable Test
    """
    interfaces = ('10GigE',
                  '25GigE',
                  '40GigE',
                  '100GigE',
                  '100GigE RS-FEC')
    test_type = test_type_factory('Cable Test')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces,
                                          error='Not a valid interface'),
                           dump_to='interface',
                           load_from='interface',
                           description='the interface of the Optics Self-Test to be launched')
    config_file = config_factory('/user/bert/tests/')

class TrueSpeedConfigSchema(Schema):
    """
    Schema for the configuration information of the Truespeed test.
    """
    interfaces = ('10/100/1000',
                  '1GigE',
                  '10GigE')
    test_type = test_type_factory('RFC 6349 TrueSpeed')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces,
                                          error='Not a valid interface'),
                           dump_to='interface',
                           load_from='interface',
                           description='the interface of the TrueSpeed test to be launched')
    config_file = config_factory('/user/bert/tests/')

class TrueSpeedVNFConfigSchema(Schema):
    """Schema for the configuration of the TrueSpeed VNF test
    """
    interfaces = ('10/100/1000',
                  '1GigE',
                 )
    test_type = test_type_factory('RFC 6349 TrueSpeed VNF')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces,
                                          error='Not a valid interface'),
                           dump_to='interface',
                           load_from='interface',
                           description=('the interface that the TrueSpeed VNF'
                                        'test will be launched on'),
                          )
    config_file = config_factory('/user/bert/tests/')

class OtnCheckConfigSchema(Schema):
    """
    Schema for the configuration information of the OTN Check test
    """
    rates = ('2.7G',
             '10.7G',
             '11.05G',
             '11.1G',
             '43.02G',
             '111.8G')
    test_type = test_type_factory('OTN Check')
    rate = fields.Str(required=True,
                      validate=OneOf(rates, error='Not a valid rate'),
                      dump_to='rate',
                      load_from='rate',
                      description='the rate of CPRI Check test to be launched')
    config_file = config_factory('/user/bert/tests/')

class RFC2544ConfigSchema(Schema):
    """
    Schema for the configuration information of the RFC 2544 Test.
    """
    interfaces = ('10/100/1000',
                  '100M',
                  '1GigE',
                  '10GigE',
                  '25GigE',
                  '40GigE',
                  '50GigE',
                  '100GigE',
                  '100GigE RS-FEC',
                  '400GigE')
    layers = ('L2',
              'L3',
              'L4')

    test_type = test_type_factory('RFC 2544')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces,
                                          error='Not a valid interface'),
                           dump_to='interface',
                           load_from='interface',
                           description='the interface of the RFC 2544 test to be launched')
    layer = fields.Str(required=True,
                       validate=OneOf(layers, error='Not a valid layer'),
                       load_from='layer',
                       dump_to='layer',
                       description='layer to launch the test')
    ipversion = IP_VERSION_FIELD
    config_file = config_factory('/user/bert/tests/')

    @validates_schema(skip_on_field_errors=True)
    def validate_ip_version(self, data):
        """
        Schema level validator that guarantees proper format and cotent of the fields specified
        """
        validate_ip_version(data)

class SAMCompleteConfigSchema(Schema):
    """
    Schema for the configuration information of the SAMComplete Test.
    """
    interfaces = ('10/100/1000',
                  '100M',
                  '1GigE',
                  '10GigE',
                  '25GigE',
                  '40GigE',
                  '50GigE',
                  '100GigE',
                  '100GigE RS-FEC',
                  '400GigE')
    layers = ('L2',
              'L3',
              'L4')
    modes = ('traffic',
             'streams',
             None)
    test_type = test_type_factory('Y.1564 SAMComplete')
    interface = fields.Str(required=True,
                           validate=OneOf(interfaces, error='Not a valid interface'),
                           dump_to='interface',
                           load_from='interface',
                           description='the interface of the SAMComplete test to be launched')
    layer = fields.Str(required=True,
                       validate=OneOf(layers, error='Not a valid layer'),
                       load_from='layer',
                       dump_to='layer',
                       description='layer to launch the test')
    mode = fields.Str(required=False,
                      allow_none=True,
                      validate=OneOf(modes, error='Not a valid mode'),
                      load_from='mode',
                      dump_to='mode',
                      description='traffic mode')
    ipversion = IP_VERSION_FIELD
    config_file = config_factory('/user/bert/tests/')

    @validates_schema(skip_on_field_errors=True)
    def validate_ip_and_traffic(self, data):
        """
        Schema level validator that guarantees proper format and cotent of the fields specified
        """
        validate_ip_version(data)
        validate_traffic_mode(data)


class EthernetConfigSchema(Schema):
    """Schema for the configuration of Ethernet BERT tests
    """
    interfaces = ('10/100/1000',
                  '100M',
                  '1GigE',
                  '10GigE',
                  '25GigE',
                  '40GigE',
                  '100GigE',
                  '100GigE RS-FEC')
    layers = ('Layer 2 Traffic',
              'Layer 2 Multiple Streams',
              'Layer 3 Traffic',
              'Layer 3 Multiple Streams',
              'Layer 4 Traffic,',
              'Layer 4 Multiple Streams')

    test_type = test_type_factory('Ethernet')
    interface = fields.Str(
        required=True,
        validate=OneOf(interfaces, error='Not a valid interface'),
        dump_to='interface',
        load_from='interface',
        description='the interface of the Ethernet test to be launched'
    )
    layer = fields.Str(
        required=True,
        validate=OneOf(layers, error='Not a valid layer'),
        dump_to='layer',
        load_from='layer',
        description='layer to launch the test'
    )
    ipversion = IP_VERSION_FIELD

    mode = fields.Str(
        requied=True,
        validate=OneOf(('Term', 'Loopback')),
        dump_to='mode',
        load_from='mode',
        description='the mode of the test to launch'
    )

    @validates_schema(skip_on_field_errors=True)
    def validate_ip_version(self, data):
        """
        Validates that a layer 3 or layer 4 test has a IP version
        """
        if ('3' in data['layer']) or ('4' in data['layer']):
            if not data.get('ipversion'):
                raise ValidationError('Layer 3/4 test requires a IP version',
                                      field_names='ipVersion')

class CableDiagnosticsConfigSchema(Schema):
    """
    Schema for the configuration information of the Cable Diagnostics Test.
    """
    test_type = test_type_factory('Cable Diagnostics')


class VoipConfigSchema(Schema):
    """Schema for the configuration information for VoIP tests
    """
    test_type = test_type_factory('VoIP')
    interfaces = (
        '10/100/1000',
        '100M',
        '1GigE',
        '10GigE',
        )
    interface = fields.Str(
        required=True,
        validate=OneOf(interfaces),
        load_from='interface',
        dump_to='interface',
        description='the interface of the VoIP test to be launched'
    )


class SDHConifigSchema(Schema):
    """Schema for the configuration of SDH tests
    """
    test_type = test_type_factory('SDH')
    interfaces = (
        'STM-1e',
        'STM-1e jitter',
        'STM-1',
        'STM-4',
        'STM-16',
        'STM-64',
    )
    interface = fields.Str(
        required=True,
        validate=OneOf(interfaces),
        load_from='interface',
        dump_to='interface',
        description='the interface of the SDH test to be launched'
    )
    mapping = fields.Str(
        load_from='mapping',
        dump_to='mapping',
        description='the payload mapping for the test'
    )
    mode = fields.Str(
        load_from='mode',
        dump_to='mode',
        description='the mode for the test'
    )


class SonetConfigSchema(Schema):
    """Schema for configuration of SONET tests
    """
    interfaces = (
        'STS-1',
        'STS-1 jitter',
        'OC-3',
        'OC-12',
        'OC-48',
        'OC-192',
    )
    test_type = test_type_factory('SONET')
    interface = fields.Str(
        required=True,
        validate=OneOf(interfaces),
        load_from='interface',
        dump_to='interface',
        description='the interface for the SONET test to be launched'
    )
    mapping = fields.Str(
        load_from='mapping',
        dump_to='mapping',
        description='the payload mapping for the test'
    )
    mode = fields.Str(
        load_from='mode',
        dump_to='mode',
        description='the mode for the test'
    )


class DS1ConfigSchema(Schema):
    """Schema for configuration of DS1 tests
    """
    test_type = test_type_factory('DS1')
    modes = (
        'BERT Term',
        'Wander Mon',
        'Signaling Term',
        'ISDN PRI Term',
        'VF Term',
    )
    mode = fields.Str(
        required=True,
        validate=OneOf(modes),
        load_from='mode',
        dump_to='mode',
        description='The test mode of the test to launch'
    )


class E1ConfigSchema(Schema):
    """Schema for configuration of E1 tests
    """
    test_type = test_type_factory('E1')
    modes = (
        'BERT Term',
        'Wander Mon',
        'ISDN PRI Term'
    )
    mode = fields.Str(
        required=True,
        validate=OneOf(modes),
        load_from='mode',
        dump_to='mode',
        description='The test mode for the test to launch'
    )


class OnePPSConfigSchema(Schema):
    """Schema for configuration of Timing tests
    """
    test_type = test_type_factory('1PPS')


class IEEE1588ConfigSchema(Schema):
    """Schema for configuration of IEEE 1588 PTP tests
    """
    test_type = test_type_factory('PTP/1588')
    interfaces = (
        '10/100/1000',
        '100M',
        '1GigE',
        '10GigE',
        '25GigE',
    )
    layers = (
        'Layer 4',
        'Layer 2',
    )
    interface = fields.Str(
        required=True,
        validate=OneOf(interfaces),
        load_from='interface',
        dump_to='interface',
        description='The interface for the IEEE 1588 test to be launched'
    )
    layer = fields.Str(
        required=True,
        validate=OneOf(layers),
        load_from='layer',
        dump_to='layer',
        description='The layer of encapsulation for the PTP traffic'
    )


class FiberInspectionSchema(Schema):
    """
    Schema for the configuration information of the Fiber Inspection Test.
    """
    profiles = (
        "SM UPC (IEC-61300-3-35 Ed. 2.0)",
        "SM APC (IEC-61300-3-35 Ed. 2.0)",
        "MM  (IEC-61300-3-35 Ed. 2.0)",
        "SM PC & Fiber Stub (IEC-61300-3-35 Ed. 2.0)",
    )

    test_type = test_type_factory('Fiber Inspection')
    profile = fields.Str(
        load_from='profile',
        dump_to='profile',
        description='The Inspection Probe profile to use for pass fail criteria',
        validate=OneOf(profiles)
    )


class OTDRSchema(Schema):
    """Schema for the configuration of an OTDR test
    """
    test_type = test_type_factory('OTDR')
    topologies = (
        'Expert-SM',
        'Expert-P2P',
        'Expert-PON',
        'Expert-MM',
        'SmartTest-SM',
        'SmartTest-MM',
        'FTTA-SM',
        'FTTA-MM',
        'FTTH',
        'Loopback',
        'Enterprise-SM',
        'Enterprise-MM',
    )
    otdr_topology =  fields.Str(
        validate=OneOf(topologies),
        load_from='otdrTopology',
        dump_to='otdrTopology',
        description='The type of OTDR test to be made for the network topology'
    )
    launch_cable = fields.Str(
        load_from='launchCable',
        dump_to='launchCable',
        description='The length (in km) of the launch cable used'
    )
    receive_cable = fields.Str(
        load_from='receiveCable',
        dump_to='receiveCable',
        description='The length (in km) of the receive cable used'
    )
    loop_cable = fields.Str(
        load_from='loopbackCable',
        dump_to='loopbackCable',
        description='The length (in km) of the loopback cable used'
    )
    alarms = fields.Str(
        load_from='alarms',
        dump_to='alarms',
        description='The name of the alarms standard to be used for pass fail criteria'
    )
    """TBD: Does this really need to exist as well as the more general .cfg/.prj file definition ?"""
    labelFile = fields.Str(
        load_from='labelFile',
        dump_to='labelFile',
        description='The path to the project (.prj) file containing Fiber labels'
    )

    config_file = config_factory('/user/disk/')

class PowerMeterConfigSchema(Schema):
    """Schema for configuration of Power Meter tests
    """
    test_type = test_type_factory('opticalPower')

class ManualSchema(Schema):
    """
    Schema for Generic Manual Steps in a Test Plan.
    """
    test_type = test_type_factory('manualStep')



# class CAAReflectionVSWR(Schema):
#     """
#     Schema for the configuration of a CAA ReflectionVSWR test
#     """
#     test_type = test_type_factory('CAA Reflection VSWR')
#     config_file = config_factory('/user/caa/internal/StrataSync/')
#
# class CAAReflectionReturnLoss(Schema):
#     """
#     Schema for the configuration of a CAA ReflectionReturnLoss test
#     """
#     test_type = test_type_factory('CAA Reflection Return Loss')
#     config_file = config_factory('/user/caa/internal/StrataSync/')
#
# class CAADTFVSWR(Schema):
#     """
#     Schema for the configuration of a CAA DTFVSWR test
#     """
#     test_type = test_type_factory('CAA DTF VSWR')
#     config_file = config_factory('/user/caa/internal/StrataSync/')
#
# class CAADTFReturnLoss(Schema):
#     """
#     Schema for the configuration of a CAA DTFReturnLoss test
#     """
#     test_type = test_type_factory('CAA DTF Return Loss')
#     config_file = config_factory('/user/caa/internal/StrataSync/')
#
# class CAA1PortCableLoss(Schema):
#     """
#     Schema for the configuration of a CAA 1 Port Cable Loss test
#     """
#     test_type = test_type_factory('CAA 1 Port Cable Loss')
#     config_file = config_factory('/user/caa/internal/StrataSync/')
#
# class CAACAA1PortPhase(Schema):
#     """
#     Schema for the configuration of a CAA 1 Port Phase test
#     """
#     test_type = test_type_factory('CAA 1 Port Phase')
#     config_file = config_factory('/user/caa/internal/StrataSync/')
#
# class CAASmithChart(Schema):
#     """
#     Schema for the configuration of a CAA SmithChart test
#     """
#     test_type = test_type_factory('CAA Smith Chart')
#     config_file = config_factory('/user/caa/internal/StrataSync/')
#
# class CAACheckConfigSchema(Schema):
#     """
#     Schema for the configuration information of a CAA
#     Check test
#     """
#
#     test_type = test_type_factory('CAA Check')
#     config_file = config_factory('/user/caa/internal/StrataSync/')
#
#
# class RASASweepTunedSpectrum(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RF Sweep Tuned Spectrum')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RASAChannelPower(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RF Channel Power')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RASAOccupiedBandwidth(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RF Occupied Bandwidth')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RASASpectrumEmissionMask(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RF Spectrum Emission Mask')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RASAAdjacentChannelPower(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RF Adjacent Channel Power')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RASAMultiAdjacentChannelPower(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RF Multi Adjacent Channel Power')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RASASpuriousEmissions(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RF Spurious Emissions')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RASATotalHarmonicDistortion(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RF Total Harmonic Distortion')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RASAFieldStrength(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RF Field Strength')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RASAAMFMAudioDemodulation(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RF AM/FM Demodulation')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RASAGatedSweep(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RF Gated Sweep')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RASARouteMap(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RF Route Map')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RAIASpectrum(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('IA Spectrum')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RAIASpectrogram(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('IA Spectrogram')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RAIARSSI(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('IA RSSI')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RAIAInterferenceFinder(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('IA Interference Finder')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RAIARadarChart(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('IA Radar Chart')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RAIAPIMDetectionSingleCarrier(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('IA PIM Detection Single Carrier')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RAIAPIMDetectionMultiCarrier(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('IA PIM Detection Multi Carrier')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RARtSAPersistentSpectrum(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RT Persistent Spectrum')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RARtSAPersistentSpectrogram(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RT Persistent Spectrogram')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RARtSAPersistentRSSI(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RT Persistent RSSI')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RARtSAPersistentInterferenceFinder(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RT Persistent Interference Finder')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RARtSAPersistentRadarChart(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RT Persistent Radar Chart')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RAChannelScanner(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RT Channel Scanner')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RAFrequencyScanner(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RT Frequency Scanner')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RACustomScanner(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RT Custom Scanner')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDSpectrum(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Spectrum')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDChannelPower(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Channel Power')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDOccupiedBandwidth(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Occupied Bandwidth')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDSpectrumEmissionMask(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Spectrum Emission Mask')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDAdjacentChannelPower(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Adjacent Channel Power')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDMultiAdjacentChannelPower(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Multi Adjacent Channel Power')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDSpuriousEmissions(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Spurious Emissions')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDPowerVsTimeFrame(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Power vs Time (Frame)')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDPowerVsTimeSlot(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Power vs Time (Slot)')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDDataChannel(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Data Channel')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDControlChannel(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Control Channel')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDSubframe(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Subframe')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDFrame(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Frame')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDTimeAlignmentError(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Time Alignment Error')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDDataAllocationMap(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Data Allocation Map')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDCarrierAggregation(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Carrier Aggregation')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDOTAChannelScanner(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD OTA Channel Scanner')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDOTAIDScanner(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD OTA ID Scanner')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDOTAMultipathProfile(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD OTA Multipath Profile')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDOTAControlChannel(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD OTA Control Channel')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDOTADatagram(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD OTA Datagram')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDOTARouteMap(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD OTA Route Map')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTETDDPowerStatisticsCCDF(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE TDD Power Statistics CCDF')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDSpectrum(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Spectrum')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDChannelPower(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Channel Power')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDOccupiedBandwidth(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Occupied Bandwidth')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDSpectrumEmissionMask(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Spectrum Emission Mask')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDAdjacentChannelPower(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Adjacent Channel Power')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDMultiAdjacentChannelPower(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Multi Adjacent Channel Power')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDSpuriousEmissions(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Spurious Emissions')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDPowerVsTimeFrame(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Power vs Time (Frame)')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDPowerVsTimeSlot(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Power vs Time (Slot)')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDDataChannel(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Data Channel')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDControlChannel(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Control Channel')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDSubframe(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Subframe')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDFrame(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Frame')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDTimeAlignmentError(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Time Alignment Error')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDDataAllocationMap(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Data Allocation Map')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDCarrierAggregation(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Carrier Aggregation')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDOTAChannelScanner(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD OTA Channel Scanner')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDOTAIDScanner(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD OTA ID Scanner')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDOTAMultipathProfile(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD OTA Multipath Profile')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDOTAControlChannel(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD OTA Control Channel')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDOTADatagram(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD OTA Datagram')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDOTARouteMap(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD OTA Route Map')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RALTEFDDPowerStatisticsCCDF(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('LTE FDD Power Statistics CCDF')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RA5GNRTriggerSpectrum(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('5G NR Trigger Spectrum')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RA5GNRChannelPower(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('5G NR Channel Power')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RA5GNROccupiedBandwidth(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('5G NR Occupied Bandwidth')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RA5GNRSpectrumEmissionMask(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('5G NR Spectrum Emission Mask')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RA5GNRAdjacentChannelPower(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('5G NR Adjacent Channel Power')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RA5GNRMultiAdjacentChannelPower(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('5G NR Multi Adjacent Channel Power')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RA5GNRSpuriousEmissions(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('5G NR Spurious Emissions')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RA5GNRBeamAnalyzer(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('5G NR Beam Analyzer')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RA5GNRCarrierScanner(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('5G NR Carrier Scanner')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RA5GNRRouteMap(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('5G NR Route Map')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RA5GNRBeamAvailabilityIndex(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('5G NR Beam Availability Index')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RA5GNRPowerVsTimeSymbol(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('5G NR Power vs Time (Symbol)')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RA5GNRPowerVsTimeFrame(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('5G NR Power vs Time (Frame)')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RARFInternalPowerMeter(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RFPM Internal RF Power Meter')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RARFExternalPowerMeter(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RFPM External RF Power Meter')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RARFoCPRISpectrum(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RFoCPRI Spectrum')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')
#
# class RARFoCPRISpectrogram(Schema):
#     """
#     Schema for the configuration of a Radio Analysis Spectrum Analyzer Sweep Tuned Spectrum test
#     """
#     test_type = test_type_factory('RFoCPRI Spectrogram')
#     config_file = config_factory('/user/cellAdvisor5G/internal/StrataSync/')


class CdmOTDRSchema(Schema):
    """Schema for the CDM otdrSettings: configuration of an OTDR test
    """
    topologies = (
        'Expert-SM',
        'Expert-P2P',
        'Expert-PON',
        'Expert-MM',
        'SmartTest-SM',
        'SmartTest-MM',
        'FTTA-SM',
        'FTTA-MM',
        'FTTH',
        'Loopback',
        'Enterprise-SM',
        'Enterprise-MM',
    )
    otdr_topology =  fields.Str(
        validate=OneOf(topologies),
        load_from='otdrTopology',
        dump_to='otdrTopology',
        description='The type of OTDR test to be made for the network topology'
    )
    launch_cable = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='launchCable',
        dump_to='launchCable',
        description='The length (in km) of the launch cable used'
    )
    receive_cable = fields.Str(
        required=False,
        allow_none=True,
        load_from='receiveCable',
        dump_to='receiveCable',
        description='The length (in km) of the receive cable used'
    )
    loop_cable = fields.Str(
        required=False,
        allow_none=True,
        load_from='loopbackCable',
        dump_to='loopbackCable',
        description='The length (in km) of the loopback cable used'
    )
    alarms = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='alarmThresholds',
        dump_to='alarmThresholds',
        description='The alarm thresholds'
    )
    config_file = config_factory('/user/disk/')
    schema_version = fields.Str(
        load_from='schemaVersion',
        dump_to='schemaVersion',
        description='version number'
    )
    setups = fields.Nested(
        Schema,
        missing=[],
        required=False,
        allow_none=True,
        load_from='setup',
        dump_to='setup',
        description='The setup(s)'
    )
    fiber = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='fiber',
        dump_to='fiber',
        description='The info on the fiber'
    )


CONFIG_SCHEMAS = {'CPRI Check': CpriCheckConfigSchema,
                  'CPRI': CpriConfigSchema,
                  'QuickCheck': QuickCheckConfigSchema,
                  'RFC 2544': RFC2544ConfigSchema,
                  'PTP Check': PtpCheckConfigSchema,
                  'Optics Self-Test': OpticsSelfTestConfigSchema,
                  'Cable Test': CableTestConfigSchema,
                  'OTN Check': OtnCheckConfigSchema,
                  'RFC 6349 TrueSpeed': TrueSpeedConfigSchema,
                  'Y.1564 SAMComplete': SAMCompleteConfigSchema,
                  'Fiber Inspection': FiberInspectionSchema,
                  'Cable Diagnostics': CableDiagnosticsConfigSchema,
                  'RFC 6349 TrueSpeed VNF': TrueSpeedVNFConfigSchema,
                  'VoIP': VoipConfigSchema,
                  'SDH': SDHConifigSchema,
                  'SONET': SonetConfigSchema,
                  'Ethernet': EthernetConfigSchema,
                  'DS1': DS1ConfigSchema,
                  'E1': E1ConfigSchema,
                  '1PPS': OnePPSConfigSchema,
                  'PTP/1588': IEEE1588ConfigSchema,
                  'OTDR': OTDRSchema,
                  'manualStep': ManualSchema,
                  'opticalPower': PowerMeterConfigSchema,
                  'CAA Reflection VSWR': caa.CAAReflectionVSWR,
                  'CAA Reflection Return Loss': caa.CAAReflectionReturnLoss,
                  'CAA DTF VSWR': caa.CAADTFVSWR,
                  'CAA DTF Return Loss': caa.CAADTFReturnLoss,
                  'CAA 1 Port Cable Loss': caa.CAA1PortCableLoss,
                  'CAA 1 Port Phase': caa.CAACAA1PortPhase,
                  'CAA Smith Chart': caa.CAASmithChart,
                  'CAA Check': caa.CAACheckConfigSchema,
                  'RF Sweep Tuned Spectrum': caa.RASASweepTunedSpectrum,
                  'RF Channel Power' : caa.RASAChannelPower,
                  'RF Occupied Bandwidth' : caa.RASAOccupiedBandwidth,
                  'RF Spectrum Emission Mask' : caa.RASASpectrumEmissionMask,
                  'RF ACLR' : caa.RASAACLR,
                  'RF Adjacent Channel Power' : caa.RASAAdjacentChannelPower,
                  'RF Multi-ACLR' : caa.RASAMultiACLR,
                  'RF Multi Adjacent Channel Power' : caa.RASAMultiAdjacentChannelPower,
                  'RF Spurious Emissions' : caa.RASASpuriousEmissions,
                  'RF Total Harmonic Distortion' : caa.RASATotalHarmonicDistortion,
                  'RF Field Strength' : caa.RASAFieldStrength,
                  'RF AM/FM Demodulation' : caa.RASAAMFMAudioDemodulation,
                  'RF Gated Sweep' : caa.RASAGatedSweep,
                  'RF Route Map' : caa.RASARouteMap,
                  'IA Spectrum' : caa.RAIASpectrum,
                  'IA Spectrogram' : caa.RAIASpectrogram,
                  'IA RSSI' : caa.RAIARSSI,
                  'IA Interference Finder' : caa.RAIAInterferenceFinder,
                  'IA Radar Chart' : caa.RAIARadarChart,
                  'IA PIM Detection Single Carrier' : caa.RAIAPIMDetectionSingleCarrier,
                  'IA PIM Detection Multi Carrier' : caa.RAIAPIMDetectionMultiCarrier,
                  'RT Persistent Spectrum' : caa.RARtSAPersistentSpectrum,
                  'RT Persistent Spectrogram' : caa.RARtSAPersistentSpectrogram,
                  'RT Persistent RSSI' : caa.RARtSAPersistentRSSI,
                  'RT Persistent Interference Finder' : caa.RARtSAPersistentInterferenceFinder,
                  'RT Persistent Radar Chart' : caa.RARtSAPersistentRadarChart,
                  'RT Channel Scanner' : caa.RAChannelScanner,
                  'RT Frequency Scanner' : caa.RAFrequencyScanner,
                  'RT Custom Scanner' : caa.RACustomScanner,
                  'LTE TDD Spectrum': caa.RALTETDDSpectrum,
                  'LTE TDD Channel Power' : caa.RALTETDDChannelPower,
                  'LTE TDD Occupied Bandwidth' : caa.RALTETDDOccupiedBandwidth,
                  'LTE TDD Spectrum Emission Mask' : caa.RALTETDDSpectrumEmissionMask,
                  'LTE TDD Adjacent Channel Power' : caa.RALTETDDAdjacentChannelPower,
                  'LTE TDD Multi Adjacent Channel Power' : caa.RALTETDDMultiAdjacentChannelPower,
                  'LTE TDD Spurious Emissions' : caa.RALTETDDSpuriousEmissions,
                  'LTE TDD Power vs Time (Frame)' : caa.RALTETDDPowerVsTimeFrame,
                  'LTE TDD Power vs Time (Slot)' : caa.RALTETDDPowerVsTimeSlot,
                  'LTE TDD Data Channel' : caa.RALTETDDDataChannel,
                  'LTE TDD Control Channel' : caa.RALTETDDControlChannel,
                  'LTE TDD Subframe' : caa.RALTETDDSubframe,
                  'LTE TDD Frame' : caa.RALTETDDFrame,
                  'LTE TDD Time Alignment Error' : caa.RALTETDDTimeAlignmentError,
                  'LTE TDD Data Allocation Map' : caa.RALTETDDDataAllocationMap,
                  'LTE TDD Carrier Aggregation' : caa.RALTETDDCarrierAggregation,
                  'LTE TDD OTA Channel Scanner' : caa.RALTETDDOTAChannelScanner,
                  'LTE TDD OTA ID Scanner' : caa.RALTETDDOTAIDScanner,
                  'LTE TDD OTA Multipath Profile' : caa.RALTETDDOTAMultipathProfile,
                  'LTE TDD OTA Control Channel' : caa.RALTETDDOTAControlChannel,
                  'LTE TDD OTA Datagram' : caa.RALTETDDOTADatagram,
                  'LTE TDD OTA Route Map' : caa.RALTETDDOTARouteMap,
                  'LTE TDD Freq / Time Error Variation' : caa.RALTETDDFreqTimeErrorVariation,
                  'LTE TDD Power Statistics CCDF' : caa.RALTETDDPowerStatisticsCCDF,
                  'LTE FDD Spectrum': caa.RALTEFDDSpectrum,
                  'LTE FDD Channel Power' : caa.RALTEFDDChannelPower,
                  'LTE FDD Occupied Bandwidth' : caa.RALTEFDDOccupiedBandwidth,
                  'LTE FDD Spectrum Emission Mask' : caa.RALTEFDDSpectrumEmissionMask,
                  'LTE FDD Adjacent Channel Power' : caa.RALTEFDDAdjacentChannelPower,
                  'LTE FDD Multi Adjacent Channel Power' : caa.RALTEFDDMultiAdjacentChannelPower,
                  'LTE FDD Spurious Emissions' : caa.RALTEFDDSpuriousEmissions,
                  'LTE FDD Power vs Time (Frame)' : caa.RALTEFDDPowerVsTimeFrame,
                  'LTE FDD Power vs Time (Slot)' : caa.RALTEFDDPowerVsTimeSlot,
                  'LTE FDD Data Channel' : caa.RALTEFDDDataChannel,
                  'LTE FDD Control Channel' : caa.RALTEFDDControlChannel,
                  'LTE FDD Subframe' : caa.RALTEFDDSubframe,
                  'LTE FDD Frame' : caa.RALTEFDDFrame,
                  'LTE FDD Time Alignment Error' : caa.RALTEFDDTimeAlignmentError,
                  'LTE FDD Data Allocation Map' : caa.RALTEFDDDataAllocationMap,
                  'LTE FDD Carrier Aggregation' : caa.RALTEFDDCarrierAggregation,
                  'LTE FDD OTA Channel Scanner' : caa.RALTEFDDOTAChannelScanner,
                  'LTE FDD OTA ID Scanner' : caa.RALTEFDDOTAIDScanner,
                  'LTE FDD OTA Multipath Profile' : caa.RALTEFDDOTAMultipathProfile,
                  'LTE FDD OTA Control Channel' : caa.RALTEFDDOTAControlChannel,
                  'LTE FDD OTA Datagram' : caa.RALTEFDDOTADatagram,
                  'LTE FDD OTA Route Map' : caa.RALTEFDDOTARouteMap,
                  'LTE FDD Freq / Time Error Variation' : caa.RALTEFDDFreqTimeErrorVariation,
                  'LTE FDD Power Statistics CCDF' : caa.RALTEFDDPowerStatisticsCCDF,
                  '5G NR Trigger Spectrum' : caa.RA5GNRTriggerSpectrum,
                  '5G NR Channel Power' : caa.RA5GNRChannelPower,
                  '5G NR Occupied Bandwidth' : caa.RA5GNROccupiedBandwidth,
                  '5G NR Spectrum Emission Mask' : caa.RA5GNRSpectrumEmissionMask,
                  '5G NR Adjacent Channel Power' : caa.RA5GNRAdjacentChannelPower,
                  '5G NR Multi Adjacent Channel Power' : caa.RA5GNRMultiAdjacentChannelPower,
                  '5G NR Spurious Emissions' : caa.RA5GNRSpuriousEmissions,
                  '5G NR Beam Analyzer' : caa.RA5GNRBeamAnalyzer,
                  '5G NR Carrier Scanner' : caa.RA5GNRCarrierScanner,
                  '5G NR Route Map' : caa.RA5GNRRouteMap,
                  '5G NR Beam Availability Index' : caa.RA5GNRBeamAvailabilityIndex,
                  '5G NR Freq / Time Error Variation' : caa.RA5GNRFreqTimeErrorVariation,
                  '5G NR Power vs Time (Symbol)' : caa.RA5GNRPowerVsTimeSymbol,
                  '5G NR Power vs Time (Frame)' : caa.RA5GNRPowerVsTimeFrame,
                  'EMF Spectrum' : caa.RAEMFSpectrum,
                  'EMF 5G NR Beam Analysis' : caa.RAEMF5GNRBeamAnalysis,
                  'DSS Spectrum' : caa.RADSSSpectrum,
                  'DSS Channel Power' : caa.RADSSChannelPower,
                  'DSS Occupied Bandwidth' : caa.RADSSOccupiedBandwidth,
                  'DSS Spectrum Emission Mask' : caa.RADSSSpectrumEmissionMask,
                  'DSS Adjacent Channel Power' : caa.RADSSAdjacentChannelPower,
                  'DSS Multi Adjacent Channel Power' : caa.RADSSMultiAdjacentChannelPower,
                  'DSS Spurious Emissions' : caa.RADSSSpuriousEmissions,
                  'DSS Power vs Time (Frame)' : caa.RADSSPowerVsTimeFrame,
                  'DSS Power vs Time (Slot)' : caa.RADSSPowerVsTimeSlot,
                  'DSS Constellation' : caa.RADSSConstellation,
                  'DSS Channel Mapper' : caa.RADSSChannelMapper,
                  'DSS Control Channel' : caa.RADSSControlChannel,
                  'DSS Subframe' : caa.RADSSSubframe,
                  'DSS Frame' : caa.RADSSFrame,
                  'DSS Time Alignment Error' : caa.RADSSTimeAlignmentError,
                  'DSS OTA Channel Scanner' : caa.RADSSOTAChannelScanner,
                  'DSS OTA ID Scanner' : caa.RADSSOTAIDScanner,
                  'DSS OTA Multipath Profile' : caa.RADSSOTAMultipathProfile,
                  'DSS OTA Control Channel' : caa.RADSSOTAControlChannel,
                  'DSS OTA Route Map' : caa.RADSSOTARouteMap,
                  'DSS Freq / Time Error Variation' : caa.RADSSFreqTimeErrorVariation,
                  'NSA Analyzer' : caa.RANSAAnalyzer,
                  'NSA Scanner' : caa.RANSAScanner,
                  'NSA Route Map' : caa.RANSARouteMap,
                  'RFPM Internal RF Power Meter' : caa.RARFInternalPowerMeter,
                  'RFPM External RF Power Meter' : caa.RARFExternalPowerMeter,
                  'RFoCPRI Spectrum' : caa.RARFoCPRISpectrum,
                  'RFoCPRI Spectrogram' : caa.RARFoCPRISpectrogram
                 }

CDM_CONFIG_SCHEMAS = {
                  'otdrSettings': CdmOTDRSchema,
                  'manualStep': Schema
                 }


class ConfigSchema(ConfigBase):
    """
    Polymorphic mapping for different ConfigSchemas
    """
    type_schemas = CONFIG_SCHEMAS
