'''Module containing a class for working with job information

Contains the Job class and associated helper functions
'''

import json
import logging

from .planned_test import PlannedTest
from .helper import html_format_test_data

log = logging.getLogger(__name__)

EMPTY_CDM = {
    'cdm': [
        {
        'tests': [],
        'workflow': {
            'workflowId': 0,
            'type': 'viaviJob',
            'workOrderId': '',
            'domainAttributes': {'logoInfo': {'path': ''}},
            'customerInfo': {'company': ''},
            'jobManagerAttributes': {'testLocation': {'value': '','label': 'Location'}},
            'techInfo': {'techId': ''}
          }
        }
    ],
    'cdmVersion': '2.1'
}

class Job(object): #pylint: disable=too-many-instance-attributes
    '''
    Class for transforming job information to and from JSON
    and for adding information to the job

    Attributes:
        job_number (str): the user's identifier for the job
        customer_name (str): the name of the user's customer
        technician_id (str): the user's identification number
        test_location (str): the location where the test is performed
        logo (str): the path to a logo for the test report
        test_data (list): a list of dictionaries representing test data
        test_plan (list): list of tests to be performed for the job
        wo_attribs (list): list of key-value pairs that may be specified by a particular customer, but not fundamental to Job Mgr operation
    '''
    def __init__(self, args):
        self.job_number = args.get('job_number',"")
        self.customer_name = args.get('customer_name',"")
        self.technician_id = args.get('technician_id',"")
        self.test_location = args.get('test_location',"")
        self.logo = args.get('logo',"")
        self.cable_id = args.get('cable_id', "")
        self.test_location_b = args.get('test_location_b', "")
        self.job_comments = args.get('job_comments', "")
        self.contractor_id = args.get('contractor_id', "")

        self.test_data = args.get('test_data', [])
        self.test_plan = [PlannedTest(planned_test) for planned_test in args.get('test_plan', [])]
        self.workflow_id = args.get('workflow_id', 0)
        self.wo_attribs = args.get('wo_attribs', {})

        self.cdm_string = args.get('cdm_string', json.dumps(EMPTY_CDM))

    @property
    def status(self):
        """Overall job status determined by the status of each planned test in
        the test plan

        Returns None if there is no test plan
        Returns "Incomplete" if any planned tests in the "To Do" state
        Returns "Fail" if any planned tests are in the "Fail" state
        Returns "Pass" otherwise
        """
        if not self.test_plan:
            return None
        todo_status = [planned_test.status == "To Do" for planned_test in self.test_plan]
        if any(todo_status):
            return 'Incomplete'
        fail_status = [planned_test.status == "Fail" for planned_test in self.test_plan]
        if any(fail_status):
            return "Fail"
        return "Pass"

    @property
    def status_ratio(self):
        """Ratio of the job items that are complete to the total number of job items
        in the job

        Returns a dictrionary: complete_count and total_count
        """
        status_ratio_dict = dict()
        status_ratio_dict['complete_count'] = 0
        status_ratio_dict['total_count']    = 0

        if not self.test_plan:
            return status_ratio_dict

        to_do_count = [planned_test.status for planned_test in self.test_plan].count("To Do")
        total_items_count = len(self.test_plan)

        status_ratio_dict['complete_count'] = total_items_count-to_do_count
        status_ratio_dict['total_count']    = total_items_count
        return status_ratio_dict


    def add_test_data(self, test_data):
        '''
        Method to add new test data to a job

        Args:
            test_data (dict): dictionary of the test data to add
        '''
        test_plan_index = find_test_plan_index(self.test_plan, test_data)

        if test_plan_index is not None:
            planned_test = self.test_plan[test_plan_index]
            planned_test.add_test_data(test_data)
            return test_plan_index
        else:
            self.test_data.append(test_data)
            return None

    def get_test_data_html_format(self):
        """
        Method to prepair the test_data list to be rendered as html in the
        creation of reports
        """
        test_data_html_format = [html_format_test_data(test_data)
                                 for test_data
                                 in self.test_data]
        return test_data_html_format

    def get_test_plan_html_format(self):
        """
        Method to prepair the test_plan list to be rendered as html in the
        creation of reports
        """
        test_plan_html_format = [planned_test.get_html_format()
                                 for planned_test
                                 in self.test_plan]
        return test_plan_html_format

    def get_cdm_workflow(self):
        '''
        Method to get the JSON CDM 'workflow' block

        Args:

        Returns:
            workflow (string): the json 'workflow' block
        '''
        cdm = self.cdm

        return { 'workflow' : cdm['cdm'][0]['workflow'] }

    @property
    def cdm(self):
        cdm = json.loads(self.cdm_string)
        workflow = cdm.get('cdm', [])[0].get('workflow')
        workflow['workOrderId'] = self.job_number
        workflow['techInfo']['techId'] = self.technician_id
        workflow['customerInfo'] = {'company':self.customer_name}
        
        workflow['domainAttributes']['logoInfo'] = {'path': self.logo}

        if self.cable_id:
            workflow['jobManagerAttributes']['cableId'] = {'value': self.cable_id, 'label': 'Cable Id'}

        if self.test_location_b:
            workflow['jobManagerAttributes']['testLocationB'] = {'value': self.test_location_b, 'label': 'Test Location B'}
            workflow['jobManagerAttributes']['testLocationA'] = {'value': self.test_location, 'label': 'Test Location A'}
        else:
            workflow['jobManagerAttributes']['testLocation'] = {'value': self.test_location, 'label': 'Test Location'}

        if self.job_comments:
            workflow['jobManagerAttributes']['jobComments'] = {'value': self.job_comments, 'label': 'Comments'}

        log.debug('get_cdm_workflow workflow object = %s', workflow )

        return cdm

def find_test_plan_index(test_plan, test_data):
    '''
    Function to determine if test data is an expected test in the test plan

    Args:
        test_plan (list): list of planned_test dictionaries
        test_data (dict): dictionay of the new test data

    Returns:
        index (int): the index in the test_plan of the matching planned_test or
            None if the test_data does not match any entries in the test plan
    '''
    for index, planned_test in enumerate(test_plan):
        if planned_test.matches(test_data):
            return index
    return None
