"""
Module that contains mainly Metro schema for job in CDM format
"""

from marshmallow import Schema, fields
from .cdm_schemas_viavi import TechInfoSchema, CustomerInfoSchema
from .cdm_schemas_viavi import empty_tech_info, empty_customer_info, empty_domain_attributes


def make_cdm_schema(config_schema=None):
    """CDM Schema factory function to create a cdm schema class based on the
    config schema for a particular product

    args:
        config_schema (cls): a OneOfSchema subclass for the particular product or None
    """
    class CdmSchema(Schema):
        """
        CDM Job Schema, contains all data related to a job
        """
        workflow = fields.Nested(
            'WorkflowSchema',
            required=True,
            load_from='workflow',
            dump_to='workflow',
            description='dictionary, contains all related to workflow')

        tests = fields.Nested(
            make_cdm_test_schema(config_schema),
            required=False,
            load_from='tests',
            dump_to='tests',
            many=True,
            description='dictionary, contains all related to tests')

    class TopSchema(Schema):
        """
        Top-level Schema for the CDM Job
        """
        cdm_version = fields.Str(required=True,
                                 load_from='cdmVersion',
                                 dump_to='cdmVersion',
                                 description='Version')
        cdm = fields.Nested(
            CdmSchema,
            required=True,
            load_from='cdm',
            dump_to='cdm',
            many=True,
            description='contains workflow, tests, etc.')

    return TopSchema

# def empty_customer_info():
#     """Function returns an empty customer info dict
#     Used to ease conversion if these fields are empty in the cdm structure
#     """
#     return {
#         "company": ""
#     }

# def empty_tech_info():
#     """Function returns an empty tech info dict
#     Used to ease conversion if these fields are empty in the cdm structure
#     """
#     return {
#         "tech_id": ""
#     }

def empty_workorder_attributes():
    """Function returns an empty attributes dictionary (as per Metro usage)
    Used to ease conversion if these fields are empty in the cdm structure
    """
    return {
        "testLocation": {
            "value": ""
        }
    }

# def empty_domain_attributes():
#     """Function returns an empty domain attributes dict (as per Metro usage)
#     Used to ease conversion if these fields are empty in the cdm structure
#     """
#     return {
#         "logo": ""
#     }

class WorkflowSchema(Schema):
    """
    Workflow Schema for the CDM Job
    """
    workflow_id = fields.Integer(
        missing=0,
        load_from="workflowId",
        dump_to="workflowId",
        description='integer, id of the workflow'
    )
    workorder_id = fields.Str(
        missing="",
        load_from="workOrderId",
        dump_to="workOrderId",
        description='string, id of the work order'
    )
    workorder_label = fields.Str(
        required=False,
        load_from="workOrderLabel",
        dump_to="workOrderLabel",
        description='string, label of the work order'
    )
    tech_info = fields.Nested(
        'TechInfoSchema',
        missing=empty_tech_info,
        load_from="techInfo",
        dump_to="techInfo",
        description="contains technician's description"
    )
    customer_info = fields.Nested(
        'CustomerInfoSchema',
        missing=empty_customer_info,
        load_from="customerInfo",
        dump_to="customerInfo",
        description="contains customer's description"
    )

    workorder_attributes = fields.Dict(
        values=fields.Nested('AttributeSchema'),
        keys=fields.Str(),
        missing=empty_workorder_attributes,
        load_from="workOrderAttributes",
        dump_to="workOrderAttributes",
        description='dictionary containing key-value pairs for work order'
    )

    domain_attributes = fields.Dict(
        values=fields.Str(),
        keys=fields.Str(),
        missing=empty_domain_attributes,
        load_from="domainAttributes",
        dump_to="domainAttributes",
        description='dictionary containing attributes for the domain'
    )

class AttributeSchema(Schema):
    """
    Schema for the attributes of the test
    """
    label = fields.Str(
        required=True,
        load_from="label",
        dump_to="label",
        description='attribute label'
    )
    value = fields.Str(
        required=True,
        load_from="value",
        dump_to="value",
        description='attribute key'
    )

# class TechInfoSchema(Schema):
#     """
#     Schema for the technician's information
#     """
#     tech_id = fields.Str(
#         required=True,
#         load_from="techId",
#         dump_to="techId",
#         description="the technician's identification number"
#     )
# 
# class CustomerInfoSchema(Schema):
#     """
#     Schema for the customer's information
#     """
#     company = fields.Str(
#         required=True,
#         load_from="company",
#         dump_to="company",
#         description="the customer's company name"
#     )


def make_cdm_test_schema(config_schema):
    """CDM Test Schema factory function to create a cdm test schema class based on the
    config schema for a particular product

    args:
        config_schema (cls): a OneOfSchema subclass for the particular product or None
    """
    def make_configuration(config_schema):
        if not config_schema:
            return fields.Dict(allow_none=True)

        return fields.Nested(
            config_schema,
            allow_none=True,
            required=False,
            load_from='configuration',
            dump_to='configuration',
            description='the configuration of the particular test to be launched'
        )

    class CdmTestSchema(Schema):
        """
        Schema for testing CDM Job
        """
        type = fields.Str(
            required=True,
            load_from="type",
            dump_to="type",
            description="type of the test"
        )
        label = fields.Str(
            required=False,
            load_from="label",
            dump_to="label",
            description="label name"
        )
        procedures = fields.Str(
            required=False,
            default="",
            load_from="procedures",
            dump_to="procedures",
            description="pathname for the procedures file"
        )
        configuration = make_configuration(config_schema)
        attributes = fields.Dict(
            values=fields.Nested('AttributeSchema'),
            keys=fields.Str(),
            required=False,
            load_from="attributes",
            dump_to="attributes",
            description='dictionary containing attribute for testing'
        )

    return CdmTestSchema
