<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\database;

use app\util\SMTLogger;

/**
 * Base Datatabase patch file
 *
 * @author Sylvain Desplat
 */
abstract class SMTSqlPatch
{
   /**
    * @var integer
    */    
   var $dbVersion;
   
   /**
    * @var integer
    */
   var $dbPatch;
   
   /**
    * Update patch type
    * @var string
    */
   const SQL_PATCH_TYPE_UPDATE = "update";

   /**
    * Upgrade patch type (migration from database version n to n + 1 )
    * 
    * @var string
    */     
   const SQL_PATCH_TYPE_UPGRADE = "upgrade";
   
   /**
    * update patch database with patch trace
    */
   const SQL_TRACE_DATABASE_PATCH = "INSERT INTO database_patch( patch_number, db_version, type, description, date_created ) VALUES( :patchNumber, :dbVersion, :type, :description, strftime('%s','now') )";      
   
    /**
     * Execute the database patch.
     * 
     * @param SMTSmartOtuDB $db
     */
    public abstract function execute( SMTSmartOtuDB $db );
    
    /**
     * Disable foreign key constraint
     *
     * @param \SQLite3 $db
     * return bool true if the query succeeded, false on failure.
     */    
    protected function disableForeignKeys( SMTSmartOtuDB $db )
    {
        $query = SMTSmartOtuDB::getEnableForeignKey( FALSE );
        //disable foreign key constraint
        return $db->execWithTrace($query, SMTLogger::INFO );
    }

    /**
     * Enable foreign key constraint
     * 
     * @param \SQLite3 $db
     * return bool true if the query succeeded, false on failure.
     */
    protected function enableForeignKeys( SMTSmartOtuDB $db )
    {
    	$query = SMTSmartOtuDB::getEnableForeignKey( TRUE );
    	//enable foreign key constraint
    	return $db->execWithTrace($query, SMTLogger::INFO );
    }
    
    
    /**
     * Add the patch trace in database_patch
     * 
     * @param \SQLite3 $db
     * @param string $description
     * 
     * @return SQLite3Result an SQLite3Result object on successful execution of the prepared
	 * statement, FALSE on failure.
     */
    function addPatchTrace( \SQLite3 $db, $description )
    {
        $patchType = self::SQL_PATCH_TYPE_UPDATE;
        SMTLogger::getInstance()->trace( sprintf( "Update patch trace; version: %d, patch number: %d, type: %s, description: %s ", $this->dbVersion, $this->dbPatch, $patchType, $description) , SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
        
        $stmt = $db->prepare( self::SQL_TRACE_DATABASE_PATCH);
        $stmt->bindValue(':patchNumber', $this->dbPatch, SQLITE3_INTEGER);
        $stmt->bindValue(':dbVersion', $this->dbVersion, SQLITE3_INTEGER);
        $stmt->bindValue(':description', $description, SQLITE3_TEXT);
        $stmt->bindValue(':type', $patchType, SQLITE3_TEXT);
        
        return $stmt->execute();        
    }
    
    /**
     * Handle SQL query result and trace error and throw exception on failure.
     * 
     * @param \SQLite3 $db
     * @param string $query
     * @param bool $success Whether statement was successful
     * @param string $file
     * @param string $method
     * @param string $line
     * 
     * @throws SMTDatabaseException
     */
    static function handleExecutePatchError( \SQLite3 $db, $query, $success, $file, $method, $line )
    {
        if ( $success === FALSE )
        {
         	SMTLogger::getInstance()->trace( "Query failed:: ".$query." SQL error: ".$db->lastErrorMsg() , SMTLogger::ERROR, $file, $method, $line);
            throw new SMTDatabaseException( SMTDatabaseException::CREATE_DATABASE_MODEL_FAILURE, $query );
        }
    } 
}
?>