"""Module containing the the ProductSpecificBase class

This class is the base class for providing product specific functions
to rest api endpionts
"""
import logging

from rest_api.products.access_anywhere import AccessAnywhere
from rest_api.api.job_manager.schemas_viavi import make_job_schema
from rest_api.api.job_manager.cdm_schemas_viavi import make_cdm_schema
from rest_api.api import __files_api_version__
from rest_api.api import __job_manager_api_version__
from rest_api.api import __location_api_version__
from rest_api.api import __datetime_api_version__

log = logging.getLogger(__name__)


class ProductSpecific(object):
    """Base class for implementing default functionality for endpoints on the
    rest api

    Default methods do not interact with the system and enable happy path
    functionality of api routes
    """
    job_schema = make_job_schema()

    cdm_schema = make_cdm_schema()

    @classmethod
    def startup_side_effect(cls, db_file, report_dir, procedures_dir, jobs_dir):
        """Default side effect executed on startup of the rest api

        Args:
            db_file: the path to the database file used by product manager
        """
        log.debug('db_file = %s', db_file)
        log.debug('report_dir = %s', report_dir)
        log.debug('procedures_dir = %s', procedures_dir)
        log.debug('jobs_dir = %s', jobs_dir)

    @staticmethod
    def update_job_side_effect(job_info):
        """Default side effect executed if job state change from inactive to
        active or when top level job information changes
        """
        #log.debug('job_info = %s', job_info)
        log.debug('job updated (default - no additional actions)')

    @staticmethod
    def get_access_anywhere_code(servername):
        """Default function executed when an access anywhere code is requested

        args:
            servername (str): host name of the syetem
        returns:
            AccessAnywere: namedtupple with the access anywhere information
        """
        log.debug('servername = %s', servername)
        return AccessAnywhere(
            accessCode="None",
            connectStatus=False,
            hasIpAddress=False,
            errorMessage='This product does not support Smart Access Anywhere'
        )

    @staticmethod
    def make_instrument_info():
        """Default function to create information for the info endpoints

        returns:
            Instrument info dictionary
        ---
        infoVersion:
            type: number
            description: version of the info JSON
            example: 1
        apiSupport:
            type: array
            description: API versions supported by this instrument
            items:
              type: object
              required:
                - name
                - versions
              properties:
                name:
                  type: string
                  description: API package name
                  example: fileMgr
                versions:
                  type: number
                  description: API version(s) supported
                  example:
                    - 1
                    - 2
                    - 3
        assetInfo:
          type: object
          properties:
            serialNo:
              type: string
              description: serial number of the instrument
            assetType:
              type: string
              description: instrument family
            uniqueId:
              type: string
              description: concatenation of platform name and serial number
            model:
              type: string
              description: model of the instrument
            hwVersion:
              type: string
              description: hardware version of the instrument
            swVersion:
              type: string
              description: current software version of the instrument
            calibrationDate:
              type: string
              description: date of last calibration
            macAddress:
              type: string
              description: mac address of the instrument
            mfgDate:
              type: string
              description: manufacturing date of the instrument
            swOptions:
              items:
                type: object
                properties:
                  description:
                    type: string
                  name:
                    type: string
                  optionLicenseType:
                    type: string
                  expirationDate:
                    type: string
              type: array
              description: currently installed software options
            hwOptions:
              type: array of strings
              description: hardware options on the mainframe
            customFields:
              type: object
              properties:
                Challenge ID:
                  type: string
                Board Info:
                  type: string
        """
        info_version = {'infoVersion':1}

        api_support = {
            'apiSupport': [
#                 {
#                     'name': 'datetime',
#                     'versions': [__datetime_api_version__]
#                 },
#                 {
#                     'name': 'location',
#                     'versions': __location_api_version__[]
#                 },
#                 {
#                     'name': 'techinfo',
#                     'versions': [1]
#                 },
#                 {
#                     'name': 'filemgr',
#                     'versions': [__files_api_version__]
#                 },
#                 {
#                     'name': 'signature',
#                     'versions': [__files_api_version__]
#                 },
#                 {
#                     'name': 'testdata',
#                     'versions': [__files_api_version__]
#                 },
#                 {
#                     'name': 'jobmanager',
#                     'versions': [__job_manager_api_version__]
#                 },
#                 {
#                     'name': 'saa',
#                     'versions': [__job_manager_api_version__]
#                 }
            ]
        }

        asset_info = {
            'assetInfo': {
                'serialNo': '12345',
                'assetType': 'Off Target',
                'uniqueId': 'Off Target_12345',
                'model': 'Off Target 1',
                'hwVersion': '1.0',
                'swVersion': '1.0',
                'calibrationDate': 'N/A',
                'macAddress': 'N/A',
                'mfgDate': 'N/A',
                'swOptions': [],
                'hwOptions': [],
                'customFields': {},
            }
        }
        instrument_info = info_version.copy()
        instrument_info.update(api_support)
        instrument_info.update(asset_info)
        return instrument_info

    @staticmethod
    def update_time(time_value, location):
        """Default function to update the instrument time
        """
        print ('product_specific_base : update_time(time_value, location)')
        log.debug('time_value = %s', time_value)
        log.debug('location = %s', location)
        return True

    @staticmethod
    def update_location(geolocation):
        """Default function to update the instrument geolocation
        """
        log.debug('latitude = %s', geolocation['latitude'])
        log.debug('longitude = %s', geolocation['longitude'])
        log.debug('timestamp = %s', geolocation['timestamp'])
        log.debug('altitude = %s', geolocation['altitude'])
        log.debug('accuracy = %s', geolocation['accuracy'])
        log.debug('altitude_accuracy = %s', geolocation['altitude_accuracy'])
        return True

    @staticmethod
    def update_stratasync_info(tech_first_name, tech_last_name, tech_id, account_id, url):
        """Default function for updating stratasync information on the instrument
        """
        log.debug('tech_first_name = %s', tech_first_name)
        log.debug('test_last_name = %s', tech_last_name)
        log.debug('tech_id = %s', tech_id)
        log.debug('account_id = %s', account_id)
        log.debug('url = %s', url)
        return True

    @staticmethod
    def update_test_data_side_effect(test_plan_index):
        """Default function executed as a result of test data being updated
        """
        log.debug('test_plan_index = %s', test_plan_index)
