"""Module containing base configurations for metro products
"""

from rest_api.products import product_specific_base

from rest_api.products.cpservices import access_anywhere
from rest_api.products.cpservices import instrument_time
from rest_api.products.cpservices import instrument_location
from rest_api.products.cpservices import update_stratasync_info

from rest_api.products.usc import asset_info
from rest_api.products.usc import file_sync
from rest_api.products.usc import cdm_sync

from rest_api.products.config_schemas_5800 import ConfigSchema

from rest_api.api.job_manager.schemas_viavi import make_job_schema
from rest_api.api.job_manager.cdm_schemas_viavi import make_cdm_schema

from clientmgr import utils


class ProductSpecific(product_specific_base.ProductSpecific):
    """Class for SmartClass 4800 product specific functions
    """
    job_schema = make_job_schema(ConfigSchema)

    cdm_schema = make_cdm_schema(ConfigSchema)

    @classmethod
    def startup_side_effect(cls, db_file, report_dir, procedures_dir, jobs_dir):
        file_sync.register_report_path(report_dir)
        file_sync.register_procedures_path(procedures_dir)
        file_sync.register_job_files_path(jobs_dir)
        cdm_sync.register_cdm_filepath()

    @staticmethod
    def get_access_anywhere_code(servername):
        return access_anywhere.create_code(servername)

    @staticmethod
    def make_instrument_info():
        info_version = {'infoVersion':1}

        __version__ = '1.3.0'
        __files_api_version__ = 1
        __job_manager_api_version__ = 1
        __location_api_version__ = 1
        __datetime_api_version__ = 1

        api_support = {
            'apiSupport': [
                {
                    'name': 'datetime',
                    'versions': [__location_api_version__]
                },
                {
                    'name': 'location',
                    'versions': [__datetime_api_version__]
                },
                {
                    'name': 'techinfo',
                    'versions': [1]
                },
                {
                    'name': 'filemgr',
                    'versions': [__files_api_version__]
                },
                {
                    'name': 'signature',
                    'versions': [__files_api_version__]
                },
                {
                    'name': 'testdata',
                    'versions': [__files_api_version__]
                },
                {
                    'name': 'jobmanager',
                    'versions': [__job_manager_api_version__]
                },
                {
                    'name': 'saa',
                    'versions': [__job_manager_api_version__]
                }
            ]
        }


        instrument_info = super(ProductSpecific, ProductSpecific).make_instrument_info()

        instrument_info['assetInfo'] = asset_info.make_asset_info()

        instrument_info.update(info_version)
        instrument_info.update(api_support)

        return instrument_info

    @staticmethod
    def update_time(time_value, location):
        return instrument_time.update_local_time(time_value)

    @staticmethod
    def update_location(geolocation):
        latitude = geolocation['latitude']
        longitude = geolocation['longitude']
        altitude = geolocation['altitude']
        return instrument_location.update_location(latitude, longitude, altitude)

    @staticmethod
    def update_stratasync_info(tech_first_name, tech_last_name, tech_id, account_id, url):
        return update_stratasync_info.update_stratasync_info(tech_id, account_id, url)

    @staticmethod
    def update_test_data_side_effect(test_plan_index):
        try:
            utils.cmsend('127.0.0.1', 'jobstatus', 'showPrompt', test_plan_index, 10)
        except ClientFactory.ConnectionError:
            pass
