"""Module containing base REST API configurations for Fiber products
"""
import logging

from rest_api.products import product_specific_base

from rest_api.products.mts_services import access_anywhere
from rest_api.products.mts_services import instrument_time
from rest_api.products.mts_services import instrument_location
from rest_api.products.mts_services import update_stratasync_info
from rest_api.products.mts_services import instrument_sync_files

from rest_api.products.mts_services import update_asset_info

from rest_api.products.usc import asset_info
from rest_api.products.usc import file_sync
from rest_api.products.usc import cdm_sync
#from rest_api.products.usc import sync_files

from rest_api.products.config_schemas_fiber import ConfigSchema

from rest_api.api.job_manager.schemas_viavi import make_job_schema
from rest_api.api.job_manager.cdm_schemas_viavi import make_cdm_schema

from rest_api.api.job_manager.job_manager import JobManager

from rest_api.products import process_cdm_job_fiber

log = logging.getLogger(__name__)

class ProductSpecific(product_specific_base.ProductSpecific):
    """Class for all Fiber product specific functions
    """
    job_schema = make_job_schema(ConfigSchema)

    cdm_schema = make_cdm_schema(ConfigSchema)

    @classmethod
    def startup_side_effect(cls, db_file, report_dir, procedures_dir, jobs_dir):
        file_sync.register_report_path(report_dir)
        file_sync.register_procedures_path(procedures_dir)
        file_sync.register_job_files_path(jobs_dir)
        cdm_sync.register_cdm_filepath()

    @staticmethod
    def get_access_anywhere_code(servername):
        return access_anywhere.create_code(servername)

    @staticmethod
    def make_instrument_info():
        info_version = {'infoVersion':1}

        __version__ = '1.11.0' # Should we align this with latest Yocto project tag?
        __files_api_version__ = 1
        __job_manager_api_version__ = 1
        __location_api_version__ = 1
        __datetime_api_version__ = 1

        api_support = {
            'apiSupport': [
                {
                    'name': 'datetime',
                    'versions': [__datetime_api_version__]
                },
                {
                    'name': 'location',
                    'versions': [__location_api_version__]
                },
                {
                    'name': 'techinfo',
                    'versions': [1]
                },
                {
                    'name': 'filemgr',
                    'versions': [__files_api_version__]
                },
                {
                    'name': 'syncfiles',
                    'versions': [1]
                },
                {
                    'name': 'signature',
                    'versions': [1]
                },
                {
                    'name': 'testdata',
                    'versions': [__files_api_version__]
                },
#                 {
#                     'name': 'jobmanager',
#                     'versions': [__job_manager_api_version__]
#                 },
                {
                    'name': 'saa',
                    'versions': [1]
                }
                #{
                    #'name': 'options',
                    #'versions': [1] 
                #}
            ]
        }


        instrument_info = super(ProductSpecific, ProductSpecific).make_instrument_info()

        asset_info_usc = asset_info.make_asset_info()
        instrument_info['assetInfo'] = update_asset_info.update_asset_info(asset_info_usc)

        instrument_info.update(info_version)
        instrument_info.update(api_support)

        log.debug('## make_instrument_info: version                 = %s', __version__)
        log.debug('## make_instrument_info: files_api_version       = %s', __files_api_version__)
        log.debug('## make_instrument_info: location_api_version    = %s', __location_api_version__)
        log.debug('## make_instrument_info: datetime_api_version    = %s', __datetime_api_version__)
        log.debug('## make_instrument_info: api_support: %s', api_support)
        return instrument_info

    @staticmethod
    def update_time(time_value, location):
        return instrument_time.update_local_time(time_value, location)

    @staticmethod
    def update_location(geolocation):
        return instrument_location.update_location(geolocation)

    @staticmethod
    def update_stratasync_info(tech_first_name, tech_last_name, tech_id, account_id, url):
        return update_stratasync_info.update_stratasync_info(tech_id, account_id, url)
        
    @staticmethod
    def get_sync_list(sync_all):
        return instrument_sync_files.get_sync_list(sync_all)
        
    @staticmethod
    def set_sync_list_ack(full_path):
        instrument_sync_files.set_sync_list_ack(full_path)
        
    @staticmethod
    def call_process_cdm_job():
        process_cdm_job_fiber.main()
        
    @staticmethod
    def get_sync_file_info(filepath, base_dir):
        return instrument_sync_files.get_sync_file_info(filepath, base_dir)
