"""
Module that contains schemas for the config information of different tests to
launch on the T-BERD/MTS-4000v2/2000v2/SmartOTDR product family
"""
from marshmallow import Schema, fields, validates_schema, ValidationError
from marshmallow.validate import OneOf
from marshmallow_oneofschema import OneOfSchema

from .config_schema_common import ConfigBase, test_type_factory, config_factory

#pylint: disable=no-self-use


class FiberInspectionSchema(Schema):
    """
    Schema for the configuration information of the Fiber Inspection Test.
    """
    profiles = (
        "SM UPC (IEC 61300-3-35 2.0)",
        "SM APC (IEC 61300-3-35 2.0)",
        "SM PC&FiberStub(IEC 61300-3-35 2.0)",
        "MM (IEC 61300-3-35 2.0)",
        "Ribbon, SM APC (IEC 61300-3-35 2.0)",
        ""
    )

    test_type = test_type_factory('Fiber Inspection')
    profile = fields.Str(
        load_from='profile',
        dump_to='profile',
        description='The Inspection Probe profile to use for pass fail criteria',
        validate=OneOf(profiles)
    )


class OTDRSchema(Schema):
    """Schema for the configuration of an OTDR test
    """
    test_type = test_type_factory('OTDR')
    topologies = (
        'Expert-SM',
        'Expert-P2P',
        'Expert-PON',
        'Expert-MM',
        'SmartTest-SM',
        'SmartTest-MM',
        'FTTA-SM',
        'FTTA-MM',
        'FTTH',
        'Loopback',
        'Enterprise-SM',
        'Enterprise-MM',
    )
    otdr_topology =  fields.Str(
        validate=OneOf(topologies),
        load_from='otdrTopology',
        dump_to='otdrTopology',
        description='The type of OTDR test to be made for the network topology'
    )
    launch_cable = fields.Str(
        load_from='launchCable',
        dump_to='launchCable',
        description='The length (in km) of the launch cable used'
    )
    receive_cable = fields.Str(
        load_from='receiveCable',
        dump_to='receiveCable',
        description='The length (in km) of the receive cable used'
    )
    loop_cable = fields.Str(
        load_from='loopbackCable',
        dump_to='loopbackCable',
        description='The length (in km) of the loopback cable used'
    )
    alarms = fields.Str(
        load_from='alarms',
        dump_to='alarms',
        description='The name of the alarms standard to be used for pass fail criteria'
    )
    """TBD: Does this really need to exist as well as the more general .cfg/.prj file definition ?"""
    labelFile = fields.Str(
        load_from='labelFile',
        dump_to='labelFile',
        description='The path to the project (.prj) file containing Fiber labels'
    )

    config_file = config_factory('/user/disk/')


class ManualSchema(Schema):
    """
    Schema for Generic Manual Steps in a Test Plan.
    """
    test_type = test_type_factory('manualStep')



class CdmOTDRSchema(Schema):
    """Schema for the CDM otdrSettings: configuration of an OTDR test
    """
    topologies = (
        'Expert-SM',
        'Expert-P2P',
        'Expert-PON',
        'Expert-MM',
        'SmartTest-SM',
        'SmartTest-MM',
        'FTTA-SM',
        'FTTA-MM',
        'FTTH',
        'Loopback',
        'Enterprise-SM',
        'Enterprise-MM',
    )
    otdr_topology =  fields.Str(
        validate=OneOf(topologies),
        load_from='otdrTopology',
        dump_to='otdrTopology',
        description='The type of OTDR test to be made for the network topology'
    )
    launch_cable = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='launchCable',
        dump_to='launchCable',
        description='The length (in km) of the launch cable used'
    )
    receive_cable = fields.Str(
        required=False,
        allow_none=True,
        load_from='receiveCable',
        dump_to='receiveCable',
        description='The length (in km) of the receive cable used'
    )
    loop_cable = fields.Str(
        required=False,
        allow_none=True,
        load_from='loopbackCable',
        dump_to='loopbackCable',
        description='The length (in km) of the loopback cable used'
    )
    alarms = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='alarmThresholds',
        dump_to='alarmThresholds',
        description='The alarm thresholds'
    )
    config_file = config_factory('/user/disk/')
    setups = fields.Nested(
        Schema,
        missing=[],
        required=False,
        allow_none=True,
        load_from='setup',
        dump_to='setup',
        description='The setup(s)'
    )
    fiber = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='fiber',
        dump_to='fiber',
        description='The info on the fiber'
    )


class OTDRSettingsSchema(Schema):#CdmOTDRSchema):
    """Schema for the CDM otdrSettings: configuration of an OTDR test
    """
    test_type = 'OTDR'

    otdr_settings = fields.Nested(
        CdmOTDRSchema,
        required=True,
        missing={},
        load_from='otdrSettings',
        dump_to='otdrSettings',
        description='The CDM configuration structure for OTDRs'
    )

class OpenSchema(Schema):
    """
    Schema to leave any config schema checks outside of this scope
    """
    test_type = test_type_factory('fromCdm')

class TestproSchema(Schema):
    """
    Cdm Schema for TestPRO / FCOMP / measurements
    """
    test_type = test_type_factory('bidirIlOrl')
    fiber = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='fiber',
        dump_to='fiber',
        description='The info on the fiber'
    )

class PwmSchema(Schema):
    """
    Cdm Schema for Powermeter
    """
    test_type = test_type_factory('powermeter')
    fiber = fields.Dict(
        required=False,
        missing={},
        allow_none=True,
        load_from='fiber',
        dump_to='fiber',
        description='The info on the fiber'
    )

CONFIG_SCHEMAS = {
                 #'otdrSettings': CdmOTDRSchema, # CDM for OTDRs
                 'CDMOTDR': OTDRSettingsSchema, #CdmOTDRSchema, # CDM for OTDRs
                 'OTDR': OTDRSchema, # old, for native job manager backwards compatibility
                 'manualStep': Schema,  ## ManualSchema for native
                 'Fiber Inspection': FiberInspectionSchema, # Allows something to happen if generated from Web tool rather than rejecting the Job
                 'fiberInspection': FiberInspectionSchema, # FIT schema for Northbound CDM ?any differences to require this? CdmFiberInspectionSchema,
                 'fromCdm': OpenSchema,
                 'powermeter': PwmSchema,
                 'bidirIlOrl': TestproSchema
                 }

class ConfigSchema(ConfigBase):
    """
    Polymorphic mapping for different CDM ConfigSchemas
    """

    # Need to override the _load method of OneOfSchema
    # because for some CDM schemas we need to get the config test type expected by Native Job Manager schemas
    # from the test.type itself
    # e.g. OTDR
    def _load(self, data, partial=None):
        if 'otdrSettings' in data:
            data['testType'] = 'CDMOTDR' # 'otdrSettings' #'CDMOTDR'
        elif 'testType' not in data: # need the CDM test type to feed in here somehow TODO
            data['testType'] = 'fromCdm'

        return super(ConfigSchema, self)._load(data)

    type_schemas = CONFIG_SCHEMAS

