"""
Module containing functions for user files rest api
"""
import os
import json
import datetime
import xml.etree.ElementTree as ET
try:
    from urllib import quote, unquote  # Python 2.X
except ImportError:
    from urllib.parse import quote, unquote  # Python 3+

from webargs.bottleparser import use_kwargs

from bottle import request, response, static_file, Bottle

from .schemas import UploadArgs

user_files = Bottle() # pylint: disable=C0103

@user_files.get('/api/syncfiles/v1/pendingfiles/all')
def syncfiles_get_all_pendingfiles():
    """Endpoint to list the files to sync and already sync
    ---
    get:
      tags:
        - user files
      description: Endpoint to list the files to sync and and already sync
      responses:
        200:
          description: success
          schema: {'$ref': '#/definitions/UserDirectory'}
    """
    base_dir = request.app.config.get('user_files.base_dir') #pylint: disable=no-member
    product_specific = user_files.config['rest_api.product_specific']

    files = product_specific.get_sync_list(True)

    res = []
    for f in files:
        res.append( get_file_info(f, base_dir) )

    response.content_type = 'application/json'
    return json.dumps(res)

@user_files.get('/api/syncfiles/v1/pendingfiles')
def syncfiles_get_pendingfiles():
    """Endpoint to list the files to sync
    ---
    get:
      tags:
        - user filesto sync
      responses:
        200:
          description: success
          schema: {'$ref': '#/definitions/UserDirectory'}
    """
    base_dir = request.app.config.get('user_files.base_dir') #pylint: disable=no-member
    product_specific = user_files.config['rest_api.product_specific']

    files = product_specific.get_sync_list(False)

    res = []
    for f in files:
        res.append( get_file_info(f, base_dir) )

    response.content_type = 'application/json'
    return json.dumps(res)

@user_files.get('/api/syncfiles/v1/userfiles<file_path:path>')
def syncfiles_get_files(file_path):
    """
    Endpoint to either list the files and directories in the file_path or to
    return the file to the client
    ---
    get:
      tags:
        - user files
      description: >
                     Endpoint to either list the files and directories in the
                     file_path or to return the file to the client
      parameters:
        - in: path
          name: file_path
          type: string
      responses:
        200:
          description: success
        400:
          description: file or directory not found
    """
    base_dir = request.app.config.get('user_files.base_dir')

    unquoted_file_name = unquote(file_path)
    rel_path = strip_basepath(unquoted_file_name, base_dir)
    full_path = os.path.join(base_dir, rel_path)

    if os.path.isfile(full_path):
        static_file_response = static_file(unquoted_file_name, root=base_dir)
        return static_file_response

    response.status = 400
    return {}

@user_files.get('/api/syncfiles/v1/metadata/<file_path:path>')
def syncfiles_get_metadata(file_path):
    """Endpoint for getting the metadata for a file from the instrument
    ---
    post:
      tags:
        - user files
      description: Endpoint for getting the metadata for a file from the instrument
      parameters:
        - in: path
          name: file_path
          type: string
    """

    unquoted_file_name = unquote(file_path)

    base_dir = request.app.config.get('user_files.base_dir')

    product_specific = user_files.config['rest_api.product_specific']
    res = product_specific.get_sync_file_info( unquoted_file_name, base_dir )
    
    return res

@user_files.put('/api/syncfiles/v1/userfiles/<file_path:path>')
def syncfiles_put_acknowledge(file_path):
    """Endpoint for uploading a file to the instrument
    ---
    post:
      tags:
        - user files
      description: Endpoint to upload a file to the instrument
      parameters:
        - in: path
          name: file_path
          type: string
      responses:
        200:
          description: success
    """

    unquoted_file_name = unquote(file_path)

    request.body.read() #pylint: disable=no-member

    base_dir = request.app.config.get('user_files.base_dir') #pylint: disable=no-member

    rel_path = strip_basepath(unquoted_file_name, base_dir)
    full_path = os.path.join(base_dir, rel_path)

    product_specific = user_files.config['rest_api.product_specific']
    product_specific.set_sync_list_ack(full_path)

    response.status = 200

    return 'Ack'



@user_files.post('/api/syncfiles/v1/userfiles')
def syncfiles_save_file():
    """Endpoint for uploading a file to the instrument
    ---
    post:
      tags:
        - user files
      description: Endpoint to upload a file to the instrument
      consumes:
        - multipart/form-data
      parameters:
        - in: formData
          schema: UploadArgs
      responses:
        200:
          description: success
    """ 
    try:
        metadata = request.files.get('metadata')
        metadata_data = json.load(metadata.file)
        
        filename = metadata_data['name']
        fileUri = metadata_data['fileUri']
        
        filedata = request.files.get(filename)
        
        filedata.save(fileUri, overwrite=True)
    
        if 'workflow.cdm.json' in filedata.filename:
            product_specific = user_files.config['rest_api.product_specific']
            product_specific.call_process_cdm_job()

        response.status = 200
    except:
        response.status = 400

    return {}

def get_file_info(filepath, base_dir):
    """Function to create a dictionary representing the file path
    the swagger schema is below
    ---
    type:
      enum:
        - F
      type: string
      description: the type of this object for files this will be F
    name:
      type: string
      description: the name of the file
    path:
      type: string
      description: the relative path to the file from the base directory
    mtime:
      type: string
      description: time of most recent content modification fo rthis file
    size:
      type: number
      description: 'the size of the file, in bytes'
    """
    fstats = os.stat(filepath)
    file_name = os.path.basename(filepath)
    file_path_rel = os.path.relpath(filepath, base_dir)

    file_info = {
            "name": file_name,
            "fileUri": "/" + file_path_rel,
        }

    return file_info

def strip_basepath(file_path, base_path):
    """Strip any directories matching the base directories from the file path

    This funciton is so that if the user supplies an absolute path that matches
    the base path the api will return the file
    """
    file_path_dirs = file_path.split('/')
    base_path_dirs = base_path.split('/')

    for dir_name in base_path_dirs:
        if dir_name in file_path_dirs and file_path_dirs.index(dir_name) == 0:
            file_path_dirs.remove(dir_name)

    return '/'.join(file_path_dirs)
