"""
Module that contains Fiber schema for jobs in CDM format
TODO Gradually merge Metro & Fiber together and prepare for future 'ViaviJob'
"""

from marshmallow import Schema, fields
#from marshmallow.validate import OneOf, Length

from .cdm_schemas_viavi import TechInfoSchema, CustomerInfoSchema
from .cdm_schemas_viavi import empty_tech_info, empty_customer_info, empty_domain_attributes
from rest_api.products.config_schemas_fiber import CdmConfigSchema


def make_cdm_schema(config_schema=None):
    """CDM Schema factory function to create a cdm schema class based on the
    config schema for a particular product

    args:
        config_schema (cls): a OneOfSchema subclass for the particular product or None
    """

    class CdmSchema2(Schema):
        """
        CDM Job Schema, contains all data related to a job
        """
        workflow = fields.Nested(
            'WorkflowSchema2',
            required=True,
            load_from='workflow',
            dump_to='workflow',
            description='dictionary, contains all related to workflow')

        tests = fields.Nested(
            make_cdm_test_schema(config_schema),
            required=False,
            load_from='tests',
            dump_to='tests',
            many=True,
            description='dictionary, contains all related to tests')

    class TopSchema(Schema):
        """
        Top-level Schema for the CDM Job
        """
        cdm_version = fields.Str(required=True,
                                 load_from='cdmVersion',
                                 dump_to='cdmVersion',
                                 description='Version')
        cdm = fields.Nested(
            CdmSchema2,
            required=True,
            load_from='cdm',
            dump_to='cdm',
            many=True,
            description='contains workflow, tests, etc.')


    return TopSchema


def empty_attributes():
    """
    Function returns an (almost)empty attributes dictionary (as per Fiber usage)
    Used to ease conversion if these fields are empty in the cdm structure
    (TODO: will change with upcoming jobMgrAttributes section) 
    """
    return {
        "cableId": ""
    }


class WorkflowSchema2(Schema):
    """
    Workflow Schema for the CDM 2.0 Job (Fiber)
    """
    workflow_id=fields.Integer(
                            missing=0,
                            load_from="workflowId",
                            dump_to="workflowId",
                            description='integer, id of the workflow')
    workflow_type=fields.Str(missing="fiberJob",
                            load_from="type",
                            dump_to="type",
                            description='string, type of the work order')
    workorder_id=fields.Str(missing="",
                            load_from="workOrderId",
                            dump_to="workOrderId",
                            description='string, id of the work order')
    workorder_label=fields.Str(required=False,
                            load_from="workOrderLabel",
                            dump_to="workOrderLabel",
                            description='string, label of the work order')
    tech_info=fields.Nested('TechInfoSchema',
                            missing=empty_tech_info,
                            load_from="techInfo",
                            dump_to="techInfo",
                            description="contains technician's description")

    customer_info=fields.Nested('CustomerInfoSchema',
                            missing=empty_customer_info,
                            load_from="customerInfo",
                            dump_to="customerInfo",
                            description="contains customer's description")

    contractor_id=fields.Str(missing="",
                            load_from="contractorId",
                            dump_to="contractorId",
                            description='contains contractor or sub-contractor Id')
    
    workorder_source=fields.Str(missing="",
                            load_from="workOrderSource",
                            dump_to="workOrderSource",
                            description='string, source of any customer-specific workOrderAttributes')

    workorder_attributes=fields.Dict(
                            ordered = True,
                            missing="{}",
                            values=fields.Nested('Schema'),
                            keys=fields.Str(),
                            load_from="workOrderAttributes",
                            dump_to="workOrderAttributes",
                            description='dictionary containing customer-specific set of key-value pairs')

    domain_attributes=fields.Dict(values=fields.Str(),
                            keys=fields.Str(),
                            missing=empty_domain_attributes,
                            load_from="domainAttributes",
                            dump_to="domainAttributes",
                            description='dictionary containing attributes for the domain')

    attributes=fields.Dict( values=fields.Nested('Schema'),
                            keys=fields.Str(),
                            missing=empty_attributes,
                            load_from="attributes",
                            dump_to="attributes",
                            description='dictionary containing key-value pairs for general workflow attributes')


class CdmTestLocAttribsSchema(Schema):
    """
    Schema for testLocation(s) of a test
    """
    ref_info=fields.Dict(
                   values=fields.Nested('Schema'),
                   keys=fields.Str(),
                   required=False,
                   load_from="referenceInfo",
                   dump_to="referenceInfo",
                   description='contains Job Manager referenceInfo for a test location')
    sub_type_info=fields.Dict(
                   values=fields.Nested('Schema'),
                   keys=fields.Str(),
                   required=False,
                   load_from="subTypeInfo",
                   dump_to="subTypeInfo",
                   description='contains Job Manager subTypeInfo for a test location')

    
class CdmTestLocationsSchema(Schema):
    """
    Schema for testLocation(s) of a test
    """
    workflow_id=fields.Integer(required=False,
                            default=0,
                            load_from="workflowId",
                            dump_to="workflowId",
                            description='integer, id of the workflow')
    label=fields.Str(required=False,
                   load_from="label",
                   dump_to="label",
                   description="label name")
    attributes=fields.Nested( CdmTestLocAttribsSchema,
                   required=False,
                   load_from="attributes",
                   dump_to="attributes",
                   description='contains attributes for a test location')

def make_cdm_test_schema(config_schema):
    """CDM Test Schema factory function to create a cdm test schema class based on the
    config schema for a particular product

    args:
        config_schema (cls): a OneOfSchema subclass for the particular product or None
    """
    def make_configuration(config_schema):
        if not config_schema:
            return fields.Dict(allow_none=True)

        return fields.Dict(
            values=fields.Nested('CdmConfigSchema'),
            keys=fields.Str(),
            #config_schema, # CdmConfigSchema
            allow_none=True,
            required=False,
            load_from='configuration',
            dump_to='configuration',
            description='the configuration of the particular test to be launched'
        )

    
    class CdmTestSchema2(Schema):
        """
        Schema for testing CDM 2.0 Job Test entries
        """
        type=fields.Str(required=True,
                       load_from="type",
                       dump_to="type",
                       description="type of the test")
        label=fields.Str(required=False,
                       load_from="label",
                       dump_to="label",
                       description="label name")
        workflow_id=fields.Integer(required=False,
                                default=0,
                                load_from="workflowId",
                                dump_to="workflowId",
                                description='integer, id of the workflow')
        procedures=fields.Str(required=False,
                       default="",
                       load_from="procedures",
                       dump_to="procedures",
                       description="pathname for the procedures file")
        configuration = make_configuration(config_schema)
        test_locations=fields.Nested(
                       CdmTestLocationsSchema,
                       keys=fields.Str(),
                       allow_none=True,
                       required=False,
                       many=True,
                       load_from="testLocations",
                       dump_to="testLocations",
                       description='testLocations for a test')
    

    return CdmTestSchema2
