"""
Module containing endpoints for the job manager api
"""
from webargs.bottleparser import use_kwargs, use_args
from marshmallow import Schema, fields
from bottle import request, response, Bottle

from rest_api.api.info_schemas import DateTimeSchema, LocationSchema

import collections

app = Bottle()  # pylint: disable=invalid-name

@app.get('/info')
def instrument_info():
    """
    Endpoint to get information about the instrument for StrataSync
    ---
    get:
        tags:
          - info
        description: gets information specific to this instrument
        responses:
            200:
                description: Success.
                'schema': {'$ref': '#/definitions/AssetInfo'}
    """
    product_specific = app.config['rest_api.product_specific']
    info = product_specific.make_instrument_info()
    return info


@app.put('/api/v1/datetime')
@use_kwargs(DateTimeSchema(strict=True))
def set_time(time_value, location):
    """
    Endpoint to set the instrument's date and time

    Attributes:
        time_value(DateTime): Python DateTime object representation of the iso 8601
            compliant date/time string provided when a request is made to this endpoint.

        location(string): A string representation of the timezone (e.g. Americas/Los Angeles)
    ---
    put:
        tags:
          - settings
        description: Currently just dummy endpoint to facilitate mobile tech connectivity
        responses:
            200:
                description: Success
                'schema':
                    title: Instrument Time
                    type: object
            400:
                description: Bad Request
    """

    request.body.read() #pylint: disable=no-member

    # print('Debug /api/v1/datetime [{0}] [{1}]'.format(time_value, location))

    product_specific = app.config['rest_api.product_specific']
    success = product_specific.update_time(time_value, location)

    if success:
        response.status = 200
    else:
        response.status = 400

    return 'Hello'




@app.get('/userfiles/list')
def user_files_list():
    """
    Endpoint to get flattened list of user files
    ---
    post:
        tags:
          - info
        description: Currently just dummy endpoint to facilitate mobile tech connectivity
        responses:
            200:
                description: Success
                'schema':
                  type: object
                  properties:
                    UserFilesList:
                      items:
                        type: object
                      type: array
    """
    return {'UserFilesList': []}

@app.put('/api/v1/location')
@use_args(LocationSchema(strict=True))
def set_geolocation(geolocation):
    """
    Endpoint to update geolocation data on the instrument

    Attributes:
        latitude(decimal): A decimal representation of the floating point latitude

        longitude(decimal): A decimal representation of the floating point longitude

        timestamp(integer): An integer representation of the timestamp from when location
            was gathered.

        accuracy(decimal): A decimal representing how accurate location data is

        altitude(decimal): A decimal representation of the floating point altitude

        altitude_accuracy(decimal): A decimal representing how accurate altitude data is
    ---
    put:
        tags:
          - settings
        description: Set location information from mobile tech app
        responses:
            200:
                description: Success
                'schema':
                    title: Instrument Time
                    type: object
            400:
                description: Bad Request
    """

    request.body.read() #pylint: disable=no-member
    product_specific = app.config['rest_api.product_specific']
    success = product_specific.update_location(geolocation)
    if success:
        response.status = 200
    else:
        response.status = 400

    return 'Hello'

class StrataSyncSchema(Schema):
    """Schema for the StrataSync information used to sync and instrument
    """
    tech_first_name = fields.Str(
        load_from="techFirstName",
        dump_to="techFirstName",
        required=True,
    )
    tech_last_name = fields.Str(
        load_from="techLastName",
        dump_to="techLastName",
        required=True,
    )
    tech_id = fields.Str(
        load_from="techId",
        dump_to="techId",
        required=True,
    )
    url = fields.Str(
        load_from="url",
        dump_to="url",
        missing="",
    )
    account_id = fields.Str(
        load_from="accountId",
        dump_to="accountId",
        missing="",
    )

@app.post('/api/v1/techinfo')
@use_args(StrataSyncSchema(strict=True))
def update_ss_info(args):
    """Endpoint for updating StrataSync information on an instrument
    This endpoint receives StrataSync tech info from the Mobile Tech app and
    updates that information on the instrument
    ---
    post:
      summary: Assign techId on instrument
      description: >
        Assigns the techId on the instrument to match the techId in StrataSync
        if the StrataSync field 'techUpdatedOnSS' is true. Updates the tech
        first name and tech last name, as well.
      tags:
        - tech info
      parameters:
        - in: body
          name: body
          required: true
          schema: StrataSyncSchema
      responses:
        200:
          description: Success
        404:
          description: No response from instrument

    """
    request.body.read()

    product_specific = app.config['rest_api.product_specific']
    success = product_specific.update_stratasync_info(**args)

    if success:
        response.status = 200
    else:
        response.status = 400

    return {}
