

from .cdm_object import CdmObject
from copy import deepcopy
import jdsu.qrcodegen.olst_config as olts
import jdsu.qrcodegen.otdr_config as otdr


ACRONYMS = [
    ('""', "*"),
    ("},{", "#"),
    (":{", "+"),
    ("},", "$"),
    (":[{", "!"),
    (":", "|"),
    (",", "?"),
    ("}]}", "/")
]


class JsonLite:
    def __init__(self, cdm_data):
        assert isinstance(cdm_data, CdmObject)
        self.__cdm = cdm_data

        self.__compress_format = ACRONYMS
        self.__content = None

        if self.__cdm.test_type == "bidirIlOrl":
            self.__compress_format += deepcopy(olts.ABRIDGMENTS)
            self.__content = self.__produce_content(**olts.SCHEMAS)

        if self.__cdm.test_type == "OTDR":
            self.__compress_format += deepcopy(otdr.ABRIDGMENTS)
            self.__content = self.__produce_content(**otdr.SCHEMAS)

    @property
    def content(self):
        return self.__content

    @property
    def compressed(self):
        if self.content is None:
            return ""

        str_content = str(self.content).replace("'", '"')\
                                       .replace(" ", "")\
                                       .replace("\n", "")

        for old, new in self.__compress_format:
            str_content = str_content.replace(old, new)

        return str_content

    def __get_cdm_content(self, sub_data, *indexes):
        if isinstance(sub_data, str) and ("$." in sub_data):  # sub_data must be a jsonpath
            required_indexes = indexes[:sub_data.count("{}")]
            path = sub_data.format(*required_indexes) if (len(required_indexes) > 0) else sub_data

            return self.__cdm.get_content(path)

        return sub_data

    def __unpack_entries(self, data, *indexes):
        """
        :param data: can be a dict or list
        :param indexes: indexes for cdm jsonpath
        :return: None
        """
        if isinstance(data, list):
            for idx, sub_data in enumerate(data):
                if isinstance(sub_data, (dict, list)):
                    self.__unpack_entries(sub_data, *indexes)
                else:
                    data[idx] = self.__get_cdm_content(sub_data, *indexes)

        if isinstance(data, dict):
            for key, sub_data in data.items():
                if isinstance(sub_data, (dict, list)):
                    self.__unpack_entries(sub_data, *indexes)
                else:
                    data[key] = self.__get_cdm_content(sub_data, *indexes)

    def __produce_content(self, main_schema, **schemas):
        """

        :param main_schema:
        :param schemas:
        :return: content
        """
        assert isinstance(main_schema, dict)

        content = deepcopy(main_schema)

        self.__unpack_entries(content)

        if "test_schema" in schemas:
            self.__add_tests(content, **schemas)

        return content

    def __add_tests(self, content, test_schema, **schemas):
        """

        :param content:
        :param test_schema:
        :param schemas:
        :return: None
        """
        for test_idx in range(self.__cdm.tests_number):
            test = deepcopy(test_schema)

            self.__unpack_entries(test, test_idx)

            test["Identifiers"] = "{}_{}_{}".format(self.__cdm.get_cable_id(test_idx),
                                                    self.__cdm.get_fiber_id(test_idx),
                                                    self.__cdm.get_fiber_number(test_idx))

            if "result_schema" in schemas:
                self.__add_results(test, test_idx, **schemas)

            content["Measurements" + str(test_idx + 1)] = test

    def __add_results(self, test, test_idx, result_schema, **schemas):
        """

        :param test:
        :param test_idx:
        :param result_schema:
        :param schemas:
        :return: None
        """
        test["Results"] = []

        for result_idx, _ in enumerate(self.__cdm.get_mesured_results(test_idx)):
            result = deepcopy(result_schema)

            self.__unpack_entries(result, test_idx, result_idx)

            if "event_schema" in schemas:
                self.__add_events(result, test_idx, result_idx, **schemas)

            test["Results"].append(result)

    def __add_events(self, result, test_idx, result_idx, event_schema):
        """

        :param result:
        :param test_idx:
        :param result_idx:
        :param event_schema:
        :return:
        """
        result["Events"] = []

        for event_idx, _ in enumerate(self.__cdm.get_events(test_idx, result_idx)):
            event = deepcopy(event_schema)

            self.__unpack_entries(event, test_idx, result_idx, event_idx)

            result["Events"].append(event)




