#!/bin/bash
#
# This script will monitor memory for running applications
#
#

function usage() 
{
    echo "$0 usage :"
    echo "--start : start the monitoring, if no database exists, the command will create a new one. If a database exists, it will be used."
    echo "--stop : stop the monitoring (do not remove the database)."
    echo "--reset : remove monitoring data, if monitoring is running create an empty database for the monitoring to continue."
    echo "--graph time : generate a png graph for the monitoring data from time to now. time is in date command format and quoted"
    echo "               Example of valid time expression :"
    echo "                       '-2 hours' : the graph will show monitoring data for the last two hours"
    echo "                       '-2 months -2 days' : the graph will show monitoring data for the last 2 months plus 2 days"
    echo "                       '-5 weeks 5 days' : the graph will show monitoring data for the last 5 weeks minus 5 days"
}

function create_db() 
{
    DS=""
    for APP in ${APP_TO_MONITOR[@]}; do            
        DS="$DS DS:${APP}:GAUGE:120:0:1000"
    done        
    
    rrdtool create $DATABASE --step 60 ${DS} RRA:MAX:0.5:1:1500 RRA:MAX:0.5:15:672 RRA:MAX:0.5:120:4372
    if [ "$?" != "0" ] ; then
        echo "Cannot create rrdtool database."
        exit -1
    fi 
}

START=0
STOP=0
RESET=0
POPULATE=0
GRAPH_ORIGIN=0
GRAPH=0
APP_TO_MONITOR=(Otu Fiber_Optic Isu Xvfb)

DATABASE="/var/lib/monitor_db.rrd"
TMP_FILE="/var/tmp/ps_mem_result.tmp"
CRON_FILE="/etc/cron.d/memory_monitor.cron"
COLOR_TABLE=("#C5F19A" "#FFD8A0" "#B9D0E8" "#D6BFD4" "#F5DDB7" "#F79494" "#D3D7CF" "#FFFFCC" "#FBF4D4" "#FFFF80" "#F3E078" "#B3B300" "#BFCFFF" "#CCCCFF" "#F3E0FF" "#DCFFDC")

if [ $# -eq 0 ] ; then 
    usage
    exit 0
fi

while [ $# -gt 0 ]
do
    case "$1" in
    "--start") START=1;;
    "--reset")  RESET=1;;
    "--populate") POPULATE=1;;
    "--stop")  STOP=1;;
    "--graph") 
        if [ $# -lt 2 ] ; then echo "Bad Usage"; exit 0; fi
        GRAPH=1
        GRAPH_ORIGIN=$2
        shift
        ;;
    "-h")
        usage
        ;;
    esac
    shift
done

if [ $START -eq 1 ] ; then
    if [ -f $DATABASE ] ; then
        echo "Continuing monitoring"
    else
        echo "Starting a new monitoring"
        create_db
    fi    
    
    if ! [ -f $CRON_FILE ] ; then 
        echo "*   *   *   *   *   root   /etc/init.d/memory_monitor.sh --populate" > $CRON_FILE
        chmod 444 $CRON_FILE
    fi
fi

if [ $RESET -eq 1 ] ; then 
    if [ -f $DATABASE ] ; then
        echo "Reseting monitoring"
        rm $DATABASE 
    fi
    [ -f $CRON_FILE ] && create_db    
fi

if [ $POPULATE -eq 1 ] ; then
    PID=$(echo ${APP_TO_MONITOR[@]} | xargs -n1 pgrep)
    ps_mem.py -p $(echo $PID | tr ' ' ',') > $TMP_FILE
    MEM_CONSO=()
    
    for APP in ${APP_TO_MONITOR[@]}; do            
        MEM_CONSO+=( $(cat $TMP_FILE | grep -i $APP | cut -d '=' -f2 | tr -s ' ' | cut -d ' ' -f2) )
    done        
    
    rm $TMP_FILE
    
    rrdtool update $DATABASE N:$(echo ${MEM_CONSO[@]} | tr ' ' ':')    
fi

if [ $STOP -eq 1 ] ; then
    [ -f $CRON_FILE ] && rm $CRON_FILE    
fi

if [ $GRAPH -eq 1 ] ; then
    DEF=""
    LINES=""
    COLOR_INDEX=0    
    # TIME_PERIOD=$(( ( $(date +"%s") - $(date -d "$GRAPH_ORIGIN" +"%s") ) ))
    TIME_PERIOD=$(( ( $(date -d "$GRAPH_ORIGIN" +"%s") ) ))
    for APP in ${APP_TO_MONITOR[@]}; do                        
        DEF="$DEF DEF:${APP}=${DATABASE}:${APP}:MAX"
        LINES="$LINES LINE2:${APP}${COLOR_TABLE[$COLOR_INDEX]}:\"${APP^}\""
        COLOR_INDEX=$(($COLOR_INDEX + 1))        
    done     
    rrdtool graph /var/tmp/app_memory.png -w 1024 -h 800 --start ${TIME_PERIOD} --end now --lower-limit 0 --vertical-label "Memory Occupation (MB)" --font DEFAULT:8: --title "Memory usage" $DEF $LINES
    echo "File generated : /var/tmp/app_memory.png"     
fi

exit 0
