<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\serviceshelper\maintenance;

use app\util\SMTConnectionException;

use app\parser\SMTOtuSocketManager;

use app\parser\SMTOtuSocket;

use app\services\maintenance\SMTSmartAccessAnywhereDto;

use app\parser\SMTOtuApi;

use app\util\SMTConcurrentAccessException;

use app\serviceshelper\SMTServiceHelper;

use app\util\SMTLogger;


/**
 * Handle SmartAccessAnywhere setup
 *
 * @author Sylvain Desplat
*/
class SMTSmartAccessAnywhere extends SMTServiceHelper
{   
    /**
     * Test setup file exclusive lock
     *
     * @var string
     */
    private static $saaSetupLockFile = NULL;
    
    const LOCK_SAA_SETUP_FILE = "/../../../tmp/saaSetup.lock";
    
    const MAX_SAA_QUERIES = 15;
    
    /**
     * Retrieve current access code: NULL if SAA is not connected
     * @return \app\services\maintenance\SMTSmartAccessAnywhereDto
     */
    public function fetchSaaSetup()
    {
    	$dto = new SMTSmartAccessAnywhereDto();
    	try
    	{
    		//retrieve connection from the socket manager
    	    $socketManager = new SMTOtuSocketManager( $this->getContext() );
    		
    	    $socket = $socketManager->openSocket( SMTOtuSocket::ISU_PORT );
    		$accessCode = $socket->sendReceive(SMTOtuApi::getSaaAccessCode());
    		$socket->close();
    		
    		$dto->setAccessCode( $accessCode );
    	}
    	catch ( \Exception $e )
    	{
    	    if ( $socket != NULL )
    	    {
    	    	$socket->close();
    	    }
    		//When saa is not enabled, the function call returns ESR 16: catch error and do nothing
    		$dto->setAccessCode( NULL );
    	}
    	return $dto;
    }
    
    /**
     * Connect Smart Access Anywhere
     * @throws Exception
     * @throws SMTConcurrentAccessException
     * @return \app\services\maintenance\SMTSmartAccessAnywhereDto
     */
    public function connectSaa()
    {
        if ( self::acquireSaaSetupLock() )
        {
            $dto = new SMTSmartAccessAnywhereDto();
            try
            {
                //retrieve connection from the socket manager
                $socketManager = new SMTOtuSocketManager( $this->getContext() );                
                $socket = $socketManager->openSocket( SMTOtuSocket::ISU_PORT );
            	$socket->send(SMTOtuApi::getSaaConnect());
            	$dto->setConnectionUnderModification( TRUE );
            }
            catch( \Exception $e)
            { 
                //release lock before closing socket which can fail
                self::releaseSaaSetupLock();
                
                if ( $socket != NULL )
                {
                	$socket->close();
                }
            	throw $e;
            }
             
            //loop during 30s to wait for SAA connection
            $loopCount = 0;
            do 
            {
                try
                {
                    sleep( 2 );
                	$accessCode = $socket->sendReceive(SMTOtuApi::getSaaAccessCode());
                	$dto->setAccessCode( $accessCode );
                	$dto->setConnectionUnderModification( FALSE );
                }
                catch ( \Exception $e )
                {
                	//When saa is not enabled, the function call returns ESR 16: catch error and do nothing
                	$dto->setAccessCode( NULL );
                }
            }
            while ( $dto->getAccessCode() == NULL && $loopCount++ < self::MAX_SAA_QUERIES );    
            
            //release lock before closing socket which can fail
            self::releaseSaaSetupLock();
            
            if ( $socket != NULL )
            {
            	$socket->close();
            }
            
            if ( $loopCount >= self::MAX_SAA_QUERIES )
            {
            	$dto->setConnectionFailure( TRUE );
            	throw new SMTConnectionException( SMTConnectionException::SAA_CONNECTION_FAILED );
            }
        }
        else
        {
        	$this->getContext()->getLogger()->trace("Couldn't acquire lock to setup SAA: ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTConcurrentAccessException( SMTConcurrentAccessException::SAA_ALLREADY_RUNNING );
        }
        return $dto;
    }
    
    /**
     * Disconnect Smart Access Anywhere
     * @throws Exception
     * @return \app\services\maintenance\SMTSmartAccessAnywhereDto
     */
    public function disconnectSaa()
    {
    	if ( self::acquireSaaSetupLock() )
    	{
    		$dto = new SMTSmartAccessAnywhereDto();
    		try
    		{
    		    //retrieve connection from the socket manager
    		    $socketManager = new SMTOtuSocketManager( $this->getContext() );
    		    $socket = $socketManager->openSocket( SMTOtuSocket::ISU_PORT );
    			$socket->send(SMTOtuApi::getSaaDisconnect());
    			
    			if ( $socket != NULL )
    			{
    				$socket->close();
    			}

    			self::releaseSaaSetupLock();
    			$dto->setAccessCode( NULL );    			
    		}
    		catch( \Exception $e)
    		{
    		    //release lock before closing socket which can fail
    		    self::releaseSaaSetupLock();
    		    
    		    if ( $socket != NULL )
    		    {
    		    	$socket->close();
    		    }

    			throw $e;
    		}
    	}
    	return $dto;
    }    
    
    private static function getSaaSetupFile()
    {
    	if ( self::$saaSetupLockFile == NULL )
    	{
    		$handle = fopen( __DIR__.self::LOCK_SAA_SETUP_FILE, "w+");
    		 
    		self::$saaSetupLockFile = ( $handle != FALSE)? $handle : NULL;
    	}
    	return self::$saaSetupLockFile;
    }
    
    /**
     * Acquire SAA exclusive lock
     *
     * @return boolean
     */
    private static function acquireSaaSetupLock()
    {
    	$retries = 0;
    	$max_retries = 10;
    
    	//SMTLogger::getInstance()->trace( "acquireSaaSetupLock" );
    
    	// keep trying to get a lock as long as possible ( max about 4s n(n+1)(2n+1)/6)
    	do
    	{
    		if ($retries > 0)
    		{
    			usleep( 10000 * ( $retries*$retries ) );
    		}
    		$retries += 1;
    	} while ( ( !flock(self::getSaaSetupFile(), LOCK_EX, $eWouldBlock) || $eWouldBlock ) && $retries <= $max_retries);
    
    	return ($retries < $max_retries);
    }
    
    /**
     * Releases SAA exclusive lock
     *
     * @return boolean
     */
    private static function releaseSaaSetupLock()
    {
    	//SMTLogger::getInstance()->trace( "releaseSaaSetupLock" );
    	return isset(self::$saaSetupLockFile) && self::$saaSetupLockFile != NULL? flock( self::$saaSetupLockFile, LOCK_UN ) : TRUE;
    }    
}
?>