<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\alarm;

use app\util\SMTUtil;

use app\serviceshelper\alarm\SMTAlarmSeverity;

use app\database\SMTIExecuteInTransaction;

use app\database\SMTSmartOtuDB;

use app\serviceshelper\alarm\SMTAlarmSpecificProblemCode;

/**
 * System alarm Dto 
 * 
 * 
 * @author Sylvain Desplat
 * @SMTClassTableNameAnnotation(tablename='system_alarm', tablekey='id') 
 */
class SMTSystemAlarmDto extends SMTAlarmDto
{      
    /**
     * @SMTAttributeInfosAnnotation(classname='app\services\alarm\SMTSystemAlarmEventDto', id='id', tablename='system_alarm_event')
     * @var SMTSystemAlarmEventDto
     */    
    protected $currentAlarmEvent;

    /**
     * @var string
     */
    protected $additionalText;
    
    /**
     * Specific problem Id in OTU ( tOtu_Alarm_Label; for example cOtu_Alarm_Local_Mode )
     *
     * @var integer
     */
    protected $specificProblemId;
    
    /**
     * OTU additional info Id string
     *
     * @var string
     */
    protected $otuAdditionalInfoId;
    
    
    /**
     * OTU additional info string info for compatibility. EPT #6693
     *
     * @var string
     */
    protected $otuAdditionalInfo;
    
    
    const FETCH_SYSTEM_ALARM_BY_DISCRIMINATOR = "SELECT * FROM system_alarm WHERE discriminator=:alarmDiscriminator";
    
    const FETCH_SYSTEM_ALARM_BY_ID = "SELECT * FROM system_alarm WHERE id=:id";    
    
    const FETCH_ALL_SYSTEM_ALARM = "SELECT * FROM system_alarm";
    
    const FETCH_ACTIVE_SYSTEM_ALARM = "SELECT * FROM system_alarm WHERE clear_status =:clearStatus ";
    
    //delete events with cascade
    const DELETE_ALL_SYSTEM_ALARM = "DELETE FROM system_alarm;";
    
    /**
     * Fetch the system alarm for the given alarm descriptor
     *
     * @param \SQLite3 $dbConnection
     * @param string $alarmDiscriminator
     * @param boolean $joinFetch FALSE by default: TRUE to force the retrieval of joined properties (system alarm last event)
     *
     * @return SMTSystemAlarmDto
     */
    public static function fetch( \SQLite3 $dbConnection, $alarmDiscriminator, $joinFetch = FALSE )
    {
        $query = self::FETCH_SYSTEM_ALARM_BY_DISCRIMINATOR;
    	$statement = $dbConnection->prepare( $query);
    	
    	self::handleStatementError( $dbConnection, $statement, "Prepare select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    	
    	$bindParameters = ':alarmDiscriminator=>'.$alarmDiscriminator;
    	$statement->bindValue(':alarmDiscriminator', $alarmDiscriminator, SQLITE3_TEXT );
    	$resultSet = $statement->execute();
    	
    	self::handleStatementError( $dbConnection, $resultSet, "Select", $query, $bindParameters, __FILE__,__METHOD__,__LINE__ );
    
    	$systemAlarm = parent::getSingleInstanceFromResultSet($resultSet, get_class() );
    	
    	//translates alarm specific problem
    	if ( $systemAlarm != NULL )
    	{
    		$systemAlarm->setSpecificProblemText( SMTAlarmSpecificProblemCode::decodeSystemAlarmSpecificProblem( $systemAlarm->getSpecificProblemCode(), $systemAlarm->isCleared() ) );
    	}
    	
        if ( $joinFetch && $systemAlarm != NULL)
    	{
    	    $systemAlarmEvent = SMTSystemAlarmEventDto::fetch($dbConnection, $systemAlarm->getId());
    	    $systemAlarm->setCurrentAlarmEvent( $systemAlarmEvent );
    	}
    	
    	return $systemAlarm;
    }    
    
    /**
     * Fetch the system alarm for the given alarm id
     *
     * @param \SQLite3 $dbConnection
     * @param string $id
     * @param boolean $joinFetch FALSE by default: TRUE to force the retrieval of joined properties (system alarm last event)
     *
     * @return SMTSystemAlarmDto
     */
    public static function fetchFromId( \SQLite3 $dbConnection, $id, $joinFetch = FALSE )
    {
    	$query = self::FETCH_SYSTEM_ALARM_BY_ID;
    	$statement = $dbConnection->prepare( $query);
    	 
    	self::handleStatementError( $dbConnection, $statement, "Prepare select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    	 
    	$bindParameters = ':id=>'.$id;
    	$statement->bindValue(':id', $id, SQLITE3_TEXT );
    	$resultSet = $statement->execute();
    	 
    	self::handleStatementError( $dbConnection, $resultSet, "Select", $query, $bindParameters, __FILE__,__METHOD__,__LINE__ );
    
    	$systemAlarm = parent::getSingleInstanceFromResultSet($resultSet, get_class() );
    	 
    	//translates alarm specific problem
    	if ( $systemAlarm != NULL )
    	{
    		$systemAlarm->setSpecificProblemText( SMTAlarmSpecificProblemCode::decodeSystemAlarmSpecificProblem( $systemAlarm->getSpecificProblemCode(), $systemAlarm->isCleared() ) );
    	}
    	 
    	if ( $joinFetch && $systemAlarm != NULL)
    	{
    		$systemAlarmEvent = SMTSystemAlarmEventDto::fetch($dbConnection, $systemAlarm->getId());
    		$systemAlarm->setCurrentAlarmEvent( $systemAlarmEvent );
    	}
    	 
    	return $systemAlarm;
    }

    /**
     * Fetch all the system alarms
     *
     * @param \SQLite3 $dbConnection
     * @param boolean $joinFetch FALSE by default: TRUE to force the retrieval of joined properties (system alarm last event)
     *
     * @return array SMTSystemAlarmDto
     */
    public static function fetchAll( \SQLite3 $dbConnection, $joinFetch = FALSE )
    {
        $systemAlarms = array();
        
    	$query = self::FETCH_ALL_SYSTEM_ALARM;
    	$statement = $dbConnection->prepare( $query);
    	 
    	self::handleStatementError( $dbConnection, $statement, "Prepare select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    	 
    	$result = $statement->execute();
    	 
    	self::handleStatementError( $dbConnection, $result, "Select", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    
    	$systemAlarms = parent::getListFromResultSet($result, get_class() );    	  	

		foreach ( $systemAlarms as &$alarm )
		{
		    //translates alarm specific problem
		    if ( $alarm != NULL )
		    {
		    	$alarm->setSpecificProblemText( SMTAlarmSpecificProblemCode::decodeSystemAlarmSpecificProblem( $alarm->getSpecificProblemCode(), $alarm->isCleared() ) );
		    	
		    	//fetch joined system alarm event
		    	if ( $joinFetch )
		    	{
		    		$systemAlarmEvent = SMTSystemAlarmEventDto::fetch($dbConnection, $alarm->getId());
		    		$alarm->setCurrentAlarmEvent( $systemAlarmEvent );
		    	}			    	
		    }			    
		}    	
    	
    	return $systemAlarms;
    }    
    
    /**
     * Fetch all the system alarms
     *
     * @param \SQLite3 $dbConnection
     * @param boolean $joinFetch FALSE by default: TRUE to force the retrieval of joined properties (system alarm last event)
     *
     * @return array SMTSystemAlarmDto
     */
    public static function fetchActiveAlarms( \SQLite3 $dbConnection, $joinFetch = FALSE )
    {
    	$systemAlarms = array();
    
    	$query = self::FETCH_ACTIVE_SYSTEM_ALARM;
    	$statement = $dbConnection->prepare( $query);
    	
    	$clearStatus = SMTAlarmSeverity::UNCLEARED_STATUS;
    	$bindParameters = ':clearStatus=>'.$clearStatus;
    	$statement->bindValue(':clearStatus', $clearStatus, SQLITE3_TEXT );
    	
    	self::handleStatementError( $dbConnection, $statement, "Prepare select", $query, $bindParameters, __FILE__,__METHOD__,__LINE__ );
    	
    	$result = $statement->execute();
    
    	self::handleStatementError( $dbConnection, $result, "Select", $query, $bindParameters, __FILE__,__METHOD__,__LINE__ );
    
    	$systemAlarms = parent::getListFromResultSet($result, get_class() );
    
    	foreach ( $systemAlarms as &$alarm )
    	{
    		//translates alarm specific problem
    		if ( $alarm != NULL )
    		{
    			$alarm->setSpecificProblemText( SMTAlarmSpecificProblemCode::decodeSystemAlarmSpecificProblem( $alarm->getSpecificProblemCode(), $alarm->isCleared() ) );
    		  
    			//fetch joined system alarm event
    			if ( $joinFetch )
    			{
    				$systemAlarmEvent = SMTSystemAlarmEventDto::fetch($dbConnection, $alarm->getId());
    				$alarm->setCurrentAlarmEvent( $systemAlarmEvent );
    			}
    		}
    	}
    	 
    	return $systemAlarms;
    }
    
    /**
     * Save dto in database and save joined dtos (alarm event)
     * 
     * @param SMTSmartOtuDB $dbConnection
     */
    public function save( SMTSmartOtuDB $dbConnection)
    {
        $dbConnection->runInTransaction( new SMTSaveSystemAlarm( $this ) );
    }
        
    /**
     * Save dto in database and save joined dtos (alarm event)
     *
     * @param SMTSmartOtuDB $dbConnection
     */
    function performSaveSystemAlarm( SMTSmartOtuDB $dbConnection)
    {
		//create new alarm
		parent::save($dbConnection);
		//update alarm event if it exists
		if ( $this->getCurrentAlarmEvent() != NULL )
		{
			$this->getCurrentAlarmEvent()->setAlarmId( $this->id );
			$this->getCurrentAlarmEvent()->save($dbConnection);

			//update last alarm event in alarm
			$this->updateSingleProperty( $dbConnection, "currentAlarmEvent" );
		}    
    }
    

    /**
     * Delete all system alarms
     * 
     * @param SMTSmartOtuDB $dbConnection
     */
    static function performDeleteAll( SMTSmartOtuDB $dbConnection )
    {
    	$query = self::DELETE_ALL_SYSTEM_ALARM;
    	$statement = $dbConnection->prepare( $query);
    	self::handleStatementError( $dbConnection, $statement, "Prepare delete", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    	 
    	$result = $statement->execute();
    	 
    	self::handleStatementError( $dbConnection, $result, "Delete", $query, NULL, __FILE__,__METHOD__,__LINE__ );
    }
    
    /**
     *
     * @param SMTSystemAlarmEventDto $currentAlarmEvent
     */
    public function setCurrentAlarmEvent( SMTSystemAlarmEventDto $currentAlarmEvent )
    {
    	$this->currentAlarmEvent = $currentAlarmEvent;
    }
    /**
     *
     * @return SMTSystemAlarmEventDto
     */
    public function getCurrentAlarmEvent()
    {
    	return $this->currentAlarmEvent;
    }
    
    /**
     * Set the system alarm description.
     * 
     * @param string $alarmDescription
     */
    public function setAdditionText( $additionalText )
    {
    	$this->additionalText = $additionalText;
    }
    /**
     * Returns the system alarm description.
     *
     * @return string
     */
    public function getAdditionalText()
    {
    	return $this->additionalText;
    }    
    
    /**
     * Set the system alarm description.
     *
     * @param string $otuAdditionalInfoId
     */
    public function setOtuAdditionalInfoId( $otuAdditionalInfoId )
    {
    	$this->otuAdditionalInfoId = ( $otuAdditionalInfoId == NULL)? "" : $otuAdditionalInfoId;
    }
    
    /**
     * Returns the system alarm otu additional info Id.
     *
     * @return string
     */
    public function getOtuAdditionalInfoId()
    {
    	return $this->otuAdditionalInfoId;
    }
    
    /**
     * Set the system alarm otu additional info for compatibility. EPT #6693
     *
     * @param string $otuAdditionalInfo
     */
    public function setOtuAdditionalInfo( $otuAdditionalInfo )
    {
    	$this->otuAdditionalInfo = ( $otuAdditionalInfo == NULL)? "" : $otuAdditionalInfo;
    }
    
    /**
     * Returns the system alarm otu additional info for compatibility. EPT #6693
     *
     * @return string
     */
    public function getOtuAdditionalInfo()
    {
    	return (($this->otuAdditionalInfo != NULL) && ( strlen($this->otuAdditionalInfo) > 0 ) )? $this->otuAdditionalInfo : $this->otuAdditionalInfoId;
    }
    
    /**
     * Set the specific problem Id in OTU
     *
     * @return number
     */
    public function setSpecificProblemId( $specificProblemId )
    {
    	$this->specificProblemId = SMTUtil::convertToInt( $specificProblemId );
    }
    
    /**
     * Returns the specific problem Id in OTU
     *
     * @return number
     */
    public function getSpecificProblemId()
    {
    	return SMTUtil::convertToInt( $this->specificProblemId );
    }
    
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }
}

/**
 * Save system alarm in a transaction
 *
 * @author Sylvain Desplat
 */
class SMTSaveSystemAlarm implements SMTIExecuteInTransaction
{
	/**
	 *
	 * @var SMTSystemAlarmDto
	 */
	var $alarm;

	/**
	 *
	 * @param SMTAlarmDto $alarm
	 */
	function __construct( SMTSystemAlarmDto $alarm )
	{
		$this->alarm = $alarm;
	}

	/**
	 * 
	 */
	public function run( SMTSmartOtuDB $dbConnection )
	{
		$this->alarm->performSaveSystemAlarm($dbConnection);		
	}

	/**
	 * @return SMTResultSetDto
	 */
	public function getResultSetDto()
	{
		return $this->alarm;
	}
}

?>