<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\http;

use Luracast\Restler\Format\JsonFormat;

use app\util\SMTLogger;

use app\serviceshelper\monitoring\SMTTestOnDemandEventDto;
use app\serviceshelper\monitoring\SMTMeasureOnDemandEventDto;
use app\parser\SMTSocketException;
use app\settings\SMTSmartOtuSettings;

/**
 * Monitor a test on demand or a measure running on OTU.
 * Poll the PHP server up to the end of the test or measure operation:
 *  - get the measure or test status by querying the PHP web server REST API.
 *  - PHP Server side: when the request, sent by that polling task, is received and processed by the PHP server, a message with the status of the operation is sent to message consumers (navigator).
 * 
 * @author Sylvain Desplat
 */
class SMTAsyncOperationPolling
{
    /**
     * Maximum time that process can wait before exiting
     * 
     * @var int 10 minutes
     */
    const PROCESS_DURATION_MAX_TIME = 600;
    /**
     * Polling period
     * @var int 1 second
     */
    const POLLING_PERIOD = 2;
    
    /**
     * The file pointer corresponding to http socket stream opened on the PHP web server. 
     * @var var
     */
    private $socket;
    /**
     * The http REST url
     * @var string
     */
    private $url;
    
    /**
     * The type of operation currently monitored
     * @var string
     */
    private $operationClassType;
    
    /**
     * Operation identifier (generated when starting a measurement or a test)
     * @var string
     */
    private $operationId;
    
    /**
     * JSON start TAG in HTML response
     * @var string
     */
    const JSON_START_TAG = "{"; 
    
    /**
     * Http query template string with 2 parameters:
     *  - query url
     *  - server address
     * @var string 
     */
    const HTTP_QUERY_STATUS = "POST %s HTTP/1.1\r\nHost: %s\r\nConnection: Close\r\n\r\n";
    
    /**
     * REST measure on link url 
     *  - resource Id
     *  - operation Id
     * @var string
     */
    const HTTP_MEASURE_ON_LINK_URL = "/tests/%s/measureOnDemand/%s/status";
    
    /**
     * REST measure on port url
     *  - operation Id
     * @var string
     */
    const HTTP_MEASURE_URL = "/acquisitions/measure/%s/status";    

    /**
     * REST test url 
     *  - resource Id
     *  - opeartion Id
     * @var string
     */
    const HTTP_TEST_URL = "/tests/%s/testOnDemand/%s/status";
    
    /**
     * Max retry count when a call fails
     * 
     * @var int
     */
    const MAX_QUERY_RETRY_COUNT = 3;
    
    /**
     * 
     * @param string $operationClassType
     * @param string $resourceId Test or measure Identifier or NULL in the case of measurement on undeclared link
     * @param string $operationId operation identifier (generated when starting a measurement or a test)
     * 
     * @throws \InvalidArgumentException
     */
    function __construct( $operationClassType, $resourceId, $operationId )
    {
        $errno = '';
        $errstr = '';
        
        if ( $operationId == "" || 
           ( $operationClassType != SMTMeasureOnDemandEventDto::getClass() && $operationClassType != SMTTestOnDemandEventDto::getClass() ) )
        {
            $message = sprintf( "Class: %s should be of type %s or a %s.", $operationClassType, SMTMeasureOnDemandEventDto::getClass(), SMTTestOnDemandEventDto::getClass());
            self::traceException($message);
            throw new \InvalidArgumentException( $message );    
        }
        $this->operationId = $operationId;
        $this->operationClassType = $operationClassType;
        
        if ( $operationClassType == SMTMeasureOnDemandEventDto::getClass() )
        {
            //measurement on demand on link
            if ( $resourceId != "" )
            {
                $this->url = sprintf(self::HTTP_MEASURE_ON_LINK_URL, $resourceId, $operationId );//"/tests/".$resourceId."/measureOnDemand/".$operationId."/status";
            }
            //measurement on undeclared link
            else
            {
                $this->url = sprintf(self::HTTP_MEASURE_URL, $operationId );//"/acquisitions/measure/".$operationId."/status";
            }
        }        
        else if ( $operationClassType == SMTTestOnDemandEventDto::getClass() )
        {
        	$this->url = sprintf(self::HTTP_TEST_URL, $resourceId, $operationId );//"/tests/".$resourceId."/testOnDemand/".$operationId."/status";
        }        
        else
        {
            $message = sprintf( "Class: %s should be of type %s or %s.", $operationClassType, SMTMeasureOnDemandEventDto::getClass(), SMTTestOnDemandEventDto::getClass());
            self::traceException($message);
            throw new \InvalidArgumentException( $message );
        }
        
        set_time_limit( self::PROCESS_DURATION_MAX_TIME );     
    }
    
    /**
     * Poll the smartOTU http server to monitor the execution of an operation (request a measurement status or a test on demand status)
     * 
     * @throws SMTSocketException if a reading or writing error occured
     */
    function poll( $startTime = 0, $retryCount = 0 )
    {
        $result = "";
        if ( $startTime <= 0 )
        {
            $startTime = microtime( true );
        }        
        
        try 
        {
            do            
            {
                sleep( self::POLLING_PERIOD );
                $result = $this->pollServer();
            }
            while ( ( $this->isOperationRunning( $result ) ) && ( microtime( true) - $startTime ) < self::PROCESS_DURATION_MAX_TIME  );
        }
        catch ( \Exception $e )
        {
            $retryCount++;            
            //in case of failure (whatever the origin, retry)
            if ( $retryCount > self::MAX_QUERY_RETRY_COUNT )
            {             
                self::traceException( "Failed to write or read from socket on http server in SMTAsyncPolling:\r\n ". $e->getMessage() );
                throw $e;
            }
            else 
            {
                self::trace( "Retry after failure. Failed to write or read from socket on http server in SMTAsyncPolling:\r\n ". $e->getMessage());
                $this->poll( $startTime, $retryCount );                
            }
        }        
    }
    
    /**
     * Query server to get operation result
     * 
     * @throws SMTSocketException
     * @return boolean|string
     */
    private function pollServer()
    {
        $response = NULL;
        $this->socket = fsockopen( SMTSmartOtuSettings::DEFAULT_OTU_ADDRESS, SMTSmartOtuSettings::DEFAULT_HTTP_PORT, $errno, $errstr, 30 );
        
        if (!$this->socket)
        {
        	self::trace( "Failed to create socket on http server in SMTAsyncPolling: ".$errstr."(".$errno.")");
        	throw new SMTSocketException( SMTSocketException::ERROR_OPENING );
        }
        
        $query = sprintf( self::HTTP_QUERY_STATUS, $this->url, SMTSmartOtuSettings::DEFAULT_OTU_ADDRESS );
        
//         $query = "GET ".$this->url." HTTP/1.1\r\nHost: ".SMTSmartOtuSettings::DEFAULT_OTU_ADDRESS."\r\nConnection: Close\r\n\r\n";       
        //error_log(SMTLogger::INFO."SMTAsyncOperationPolling request from PHP server: ".$out );
        
        try 
        {
            stream_set_blocking($this->socket, true);
            stream_set_timeout($this->socket, self::PROCESS_DURATION_MAX_TIME);
            
            if ( fwrite($this->socket, $query) === FALSE )
            {
            	throw new SMTSocketException( SMTSocketException::ERROR_WRITTING );
            }
                            
            $response = fread($this->socket, 10000);        
            if ( $response == false )
            {
            	throw new SMTSocketException( SMTSocketException::ERROR_READING );
            }
            //self::trace( "SMTAsyncOperationPolling response from PHP server: \r\n". strstr( $response, self::JSON_START_TAG ) );
            
            fclose($this->socket);
            $this->socket = false;
        }
        catch ( \Exception $e )
        {
            if ( $this->socket != false )
            {
        		fclose($this->socket);
        		$this->socket = false;
            }
    		throw $e;
        }                
        
        return $response;
    }    
    
    /**
     * Whether the operation status encoded in the http response has an operation status "running".
     *
     * @param string $stringStatus operation status
     *
     * @return boolean True if the given status is an operation status which indicates that the operation is running.
     * @throws Exception if json response processing fails (server can send )
     */
    private function isOperationRunning( $httpResponse )
    {
    	$running = FALSE;   	
    	
    	//in the case of an empty response, the operation no longer exists
    	if ( $httpResponse !== NULL && $httpResponse != "" )
    	{    	
    	    $json_array = self::retrieveJsonData( $httpResponse );
            if ( $json_array !== NULL && $json_array != "" )
            {	    
                try 
                {
                    //forge dtos
                    if ( $this->operationClassType == SMTMeasureOnDemandEventDto::getClass() )
                    {
                        
                        $operation = SMTMeasureOnDemandEventDto::getInstance( $json_array );
                    }
                    else
                    {
            
                        $operation = SMTTestOnDemandEventDto::getInstance( $json_array );
                    }
                    $running = $operation->isOperationRunning();
                }
                catch(\Exception $e)
                {                    
                    self::traceException( $e->getMessage() );
                    throw $e;
                }
            }
            else
            {
            	$running = FALSE;
            }            
        }
        else
        {
            $running = FALSE;
        }        
                     
        return $running;
    }
    
    /**
     * Retrieve JSON data from http response.
     * 
     * @param string $httpResponse HTTP response with http header and with JSON encoded data
     * @return Json array
     */
    private static function retrieveJsonData( $httpResponse )
    {
        //remove http header        
        $httpResponseWithoutHeader = strstr( $httpResponse, self::JSON_START_TAG );
        
//         self::trace( $httpResponseWithoutHeader );
        
        //decode JSON data
        $jsonDecode = new JsonFormat();
        $json_array =  $jsonDecode->decode( $httpResponseWithoutHeader );
        return $json_array;
    }
    
    public static function traceException( $message )
    {
        error_log( SMTLogger::formatMessage($message, SMTLogger::ERROR) );
    }
    
    public static function trace( $message )
    {
    	error_log( SMTLogger::formatMessage($message, SMTLogger::DEBUG) );
    }    
}
?>