<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\monitoring;

use app\services\setup\SMTOtuCurrentPortDto;

use app\admin\SMTOtuType;

use app\serviceshelper\alarm\SMTAlarm;

use app\sharedmemory\SMTMemoryManager;

use app\serviceshelper\system\SMTSystemConfigurationDto;
use app\serviceshelper\otdr\SMTOtdrTraceFileUtil;
use app\services\monitoring\SMTTestAcquisitionParametersDto;
use app\util\SMTLogger;
use app\serviceshelper\otau\SMTSwitchManager;
use app\settings\SMTSmartOtuSettings;
use app\serviceshelper\module\SMTModule;
use app\services\monitoring\SMTTestDetailDto;
use app\parser\SMTOtuApi;
use app\serviceshelper\SMTServiceHelper;
use app\util\SMTUtil;
use app\serviceshelper\otau\SMTSwitchBufferEncoderDecoder;
use app\services\monitoring\SMTLinkTestDto;
use app\services\monitoring\SMTMonitoringPeakDto;
use app\settings\SMTSmartOtuThresholds;
use app\serviceshelper\maintenance\SMTLicence;
use app\serviceshelper\otdr\SMTHighsensOtdrTraceFileUtil;
use app\services\monitoring\SMTOtuTestThresholdHysteresisListDto;
use app\parser\SMTOtuSocket;

include_once( "phputf8/ascii.php" );

/**
 * 
 * @author Sylvain Desplat
 */
class SMTLinkTest extends SMTServiceHelper
{
    const LINK_UTF8_NAME_INDEX = 0;
    const LINK_NAME_INDEX = 1;
    const LINK_AVAILABLE_INDEX = 2;
    const LINK_MODULE_INDEX = 4;
    const SWITCH_BUFFER_INDEX = 7;
    const GPS_FILE_NAME_INDEX = 8;
    const LINK_UTF8_NAME_MAX_SIZE = 80; //80 chars=> 320 bytes max
    const TIME_SLOT_NONE = "NONE";
    const STOP = "STOP";
    const SEQUENTIAL = "SEQU";
    const RANDOM = "RAND";
    const SLEEP_TIME_FOR_TEST_REMOVAL_S = 5;//sleep 5s
    const LOCK_TEST_SETUP_FILE = "/../../../tmp/testSetup.lock";
    const PEAK_NAME_SEPARATOR = "\t";
    
    const TEST_MODE_INDEX = 0;
    const TEST_AVAILABLE_INDEX = 1;
    
    /**
     * Keeps configured switch dto to avoid to retrieve it several times:
     *  
     * @var app\serviceshelper\otau\SMTMacroSwitch
     */
    private $configuredSwitch = NULL;
    
    /**
     * Keeps configured module dto to avoid to retrieve it several times:
     *  
     * @var app\serviceshelper\module\SMTModule
     */
    private $moduleDto = NULL;
        
    /**
     * Keeps time zone offset, to avoid to call it for each link retrieval
     * @var unknown
     */
    private $timezoneOffset = NULL;
    
    /**
     * Test setup file exclusive lock
     *
     * @var string
     */
    private static $testSetupLockFile = NULL;
    
    /**
     * Check link name: #charaters > 0  and #charaters > 80 and does not contain ". Throw exception if the test fails.
     * 
     * @param string $name Link name
     * 
     * @throws SMTLinkTestException
     */
    static function checkLinkName( $name )
    {
        //strlen count bytes and not characters
        if ( (strlen( $name ) <= 0) || (mb_strlen( $name, 'UTF-8' ) > self::LINK_UTF8_NAME_MAX_SIZE) || (strpos($name, '"') !== false) )
        {
            throw new SMTLinkTestException( SMTLinkTestException::ERROR_INVALID_LINK_NAME );
        }
    }
    
    /**
     * Turn the given utf-8 string into an ascii string
     * (with accent converted to unaccented characters and remaining non ascii characters removed).
     * Append the port of the link to the name.
     *
     * @param string $utf8Name
     * @param string $portNumber
     * @return string Link name in ascii with port number append
     */
    static function formatLinkNameInAscii( $utf8Name, $portNumber )
    {
        $asciiName = $utf8Name;
        $portName = MSG_PORT." ".$portNumber;
    
        if ( !utf8_is_ascii( $utf8Name ) )
        {
            //turn latin characters with accents to ascii equivalent
            $asciiName = utf8_accents_to_ascii( $utf8Name );

            //if asciiName is not an ascii string, use Link + port number for the name
            //Link musn't be translated: OTU alerts are only in one language and text must be ascii 
            if ( !utf8_is_ascii( $asciiName ) )
            {
                $asciiName = "Link ".$portNumber;
            }
        }
    
        //check string length
        $asciiName = substr( $asciiName, 0, (self::LINK_UTF8_NAME_MAX_SIZE - mb_strlen( $portName, 'UTF-8' ) - 3 ) );
        $asciiName.= " - ".$portName;
    
        return $asciiName;
    }
    
    /**
     * Check link-test id: integer >=0
     * Throws exception if test fails and if $throwException is not FALSE.
     * 
     * @param number $id link identifier
     * @param boolean $throwException True by default: whether an exception is thrown if the check fails
     *
     * @throws SMTLinkTestException
     * @return whether the link id has a valid value
     */
    static function checkLinkTestId( $id, $throwException = TRUE )
    {
        if ( $id === NULL || !SMTUtil::isInt( $id ) || $id < 0 )
        {        
            if ( $throwException )
            {
                throw new SMTLinkTestException( SMTLinkTestException::ERROR_INVALID_LINK_ID );
            }
            else
            {
                return FALSE;
            }            
        }
        return TRUE;
    }     
    
    /**
     * Check test id validity: integer >=0.
     * Throws exception if test fails and if $throwException is not FALSE.
     *
     * @param number $id test identifier
     * @param boolean $throwException True by default: whether an exception is thrown if the check fails
     *
     * @throws SMTLinkTestException if tests fails and if $throwException is not FALSE
     * 
     * @return whether the test id has a valid value
     */
    static function checkTestId( $id, $throwException = TRUE )
    {
        if ( $id === NULL || !SMTUtil::isInt( $id ) || $id < 0 )
        {
            if ( $throwException )
            {
                throw new SMTLinkTestException( SMTLinkTestException::ERROR_INVALID_TEST_ID );
            }
            else 
            {
                return FALSE;
            } 
        }
        return TRUE;
    }
    
    /**
     * Check port out number: > 0. 
     * Throws exception if test fails and if $throwException is not FALSE.
     *      
     * @param number $portOutNumber port out number
     * @param boolean $throwException True by default: whether an exception is thrown if the check fails
     * 
     * @throws SMTLinkTestException
     * @return whether the port number has a valid value
     */
    static function checkPortOutNumber( $portOutNumber, $throwException = TRUE )
    {
        if (  $portOutNumber === NULL || $portOutNumber <= 0 )
        {
            if ( $throwException )
            {
                throw new SMTLinkTestException( SMTLinkTestException::ERROR_INVALID_PORT_OUT );
            }
            else 
            {
                return FALSE;
            } 
        }
        return TRUE;
    }    
    
    /**
     * Create a new link and updates the given DTO with the new identifier of the link
     * 
     * @param SMTLinkTestDto $linkTestDto
     * @return number the created link identifier
     * @throws SMTLinkTestException, SMTSocketException
     */
    function createLink( SMTLinkTestDto $linkTestDto  )
    {        
        self::checkLinkName( $linkTestDto->getName() );
        SMTModule::checkModuleRunningNumber( $linkTestDto->getModuleRunningNumber() );
        
        //Don't check port out: if port out is NUll, no switch is selected. 
        //self::checkPortOutNumber($linkTestDto->getPortNumber() );
        
        //create switch buffer
        //case of no switch
        if ( $linkTestDto->getPortNumber() === NULL || !( $this->hasSwitches() ) )
        {
            $switchCount = 0;
            $switchBuffer = SMTOtuApi::EMPTY_BUFFER;
        }
        else
        {
            $switchCount = SMTSwitchBufferEncoderDecoder::DEFAULT_SWITCH_COUNT; // NO ROTAU
            $switchBuffer = SMTSwitchBufferEncoderDecoder::encodeSwitch( SMTSwitchBufferEncoderDecoder::DEFAULT_SWITCH_INPUT, $linkTestDto->getPortNumber() );
        }        
        
        SMTLogger::getInstance()->trace( sprintf( "Create link  %s on module %s, on switch %s, with switch buffer %s", $linkTestDto->getName(), $linkTestDto->getModuleRunningNumber(), $switchCount, $switchBuffer ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        
        $linkId = $this->sendReceive( SMTOtuApi::addLinkUtf8Command( $linkTestDto->getName(), self::formatLinkNameInAscii( $linkTestDto->getName(), $linkTestDto->getPortNumber() ), $linkTestDto->getModuleRunningNumber(), $switchCount, $switchBuffer) );
        $linkTestDto->setId( $linkId );
        
        return $linkId;
    }
    
    /**
     * Check that the link exists with the given linkId.
     * If linkId is not set, check that a link exists on the given port.
     * Warning: If the linkId wasn't set in $linkTestDto parameter, and a link was found on the corresponding port,
     * set it.
     * 
     * @param SMTLinkTestDto $linkTestDto
     * 
     * @return boolean Whether a link exists on the given port, or, if no switch is present, 
     * whether the link exists with the given linkId
     */
    function checkLinkAndUpdateDto( SMTLinkTestDto $linkTestDto )
    {
        $linkFound = FALSE;
        $linkId = $linkTestDto->getId() ;
        $portNumber = $linkTestDto->getPortNumber();
        
        //if the linkId is given, check that it exists
        if ( SMTUtil::isInt( $linkId ) && $linkId >= 0 )
        {                
            try 
            {
                //retrieve link info
                $this->parseLinkInfo( $linkId );
                $linkFound = TRUE;
            }
            catch(\Exception $e)
            {
                $linkFound = FALSE;
                $this->getContext()->getLogger()->traceException($e);
            }            
        }
        else if ( $portNumber !== NULL && $portNumber > 0 )
        {
            //retrieve links ids
            try 
            {
        		$linkId = $this->sendReceive( SMTOtuApi::getLinkListPerPortCommand( intval($portNumber), intval($portNumber) ) );
        		if ( self::checkLinkTestId( $linkId, FALSE) )
        		{
        			$linkTestDto->setId($linkId);
	        		$linkFound = TRUE;
        		}
        	}
        	catch(\Exception $e)
        	{
        		$linkFound = FALSE;
        	}
        }
        
        return $linkFound;
    }
    
    /**
     * Check whether the given link has a test
     * 
     * @param string $linkId
     * 
     * @return boolean
     */
    function hasTest( $linkId )
    {
        //retrieve test identifier:
        $linkTestIds = $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $linkId ) );
        $linkIdsArray = array();
        if ( $linkTestIds != NULL && $linkTestIds != "" )
        { 
            $linkIdsArray = explode (",", $linkTestIds );
        }
        return ( count( $linkIdsArray ) > 0 );
    }    
    
    /**
     * @param number $linkId
     * @throws \Exception if the deletion fails
     */
    function deleteLink( $linkId )
    {
        try 
        {
            $this->send( SMTOtuApi::deleteLinkCommand($linkId) );
            
            //delete link-test from APC memory cache to force its retrieval
            SMTMemoryManager::delete( SMTLinkTestDto::getClass(), $linkId );            
        }
        catch(\Exception $e)
        {
            $this->getContext()->getLogger()->traceException($e);
            throw new SMTLinkTestException( SMTLinkTestException::ERROR_DELETE_LINK );
        }
    }
    
    /**
     * Add a highsens test from the given DTO and update that DTO with values of the monitoring test added on OTU.
     *
     * @param SMTLinkTestDto $linkTestDto
     * @param string $peaksPositionsBuffer
     * @param string $peakNames
     * @param int    $peakCount
     *
     * @return the identifier of the new test created
     * @throws SMTLinkTestException, SMTSocketException
     */
    private function addHighsensMonitoringTest( SMTLinkTestDto $linkTestDto, $peaksPositionsBuffer, $peakNames, $peakCount)
    {
        $testId = NULL;
        
        $thresholdsBuffer = SMTThresholdBufferEncoderDecoder::encodeHighsenMonitoringThreshold( $linkTestDto->getTestDetail()->getFirstMarkerThresholdsDb() );
        $peakThresholdsBuffer = SMTThresholdBufferEncoderDecoder::encodeHighsenPeaksMonitoringThreshold( $linkTestDto->getTestDetail()->getMonitoredPeaks(), $peakCount);
        
        SMTLogger::getInstance()->trace( sprintf( "Start creating test with thresholds buffer %s ", $thresholdsBuffer ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        SMTLogger::getInstance()->trace( sprintf( "Start creating test with peaks thresholds buffer %s ", $peakThresholdsBuffer), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        SMTLogger::getInstance()->trace( sprintf( "Start creating test with peak position buffer %s ", $peaksPositionsBuffer), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        
        try
        {
            //copy acquisition file from session directory to otu detection and localisation directories
            $referenceTraceFiles = SMTHighsensOtdrTraceFileUtil::copyMeasurementAsReferenceInOTU( $linkTestDto->getPortNumber(), $linkTestDto->getId(), $linkTestDto->getTestId(), $linkTestDto->getTestDetail()->getOtdrTraceName(), $linkTestDto->getTestDetail()->getOtdrTraceType() );
            
            //update OTDR trace name set as a reference in test
            $linkTestDto->getTestDetail()->setOtdrTraceName( $referenceTraceFiles->getOtdrFileName() );
        }
        catch ( \Exception $e )
        {
            $this->getContext()->getLogger()->traceException($e);
            throw new SMTLinkTestException( SMTLinkTestException::ERROR_MISSING_OTDR_TRACE );
        }
        
        try
        {
            //set attenuation critical thresholds (applied on OTU only if they have changed)
            $this->send( SMTOtuApi::setAttenuationCriticalThresholds( $linkTestDto->getTestDetail()->getAttenuationCriticalThresholdDbDefault() ) );
            
            //create test and its detection
            $peakMonitoringType= self::getOtuMonitoringType($linkTestDto->getTestDetail()->getPeakShiftMode(), ($peakCount > 0));
            
            $testId = $this->sendReceive( SMTOtuApi::addHighsensTestCommand( $linkTestDto->getId(), $linkTestDto->getTestDetail()->getTestMode() ,$referenceTraceFiles->getOtdrFileNameWithoutExtension(), $linkTestDto->getTestDetail()->getFirstMarkerDistanceM(), $linkTestDto->getTestDetail()->getLastMarkerDistanceM(), $linkTestDto->getTestDetail()->getMissingPeakThresholdPt(), $peakMonitoringType, $peakCount, $thresholdsBuffer, $peakThresholdsBuffer, $peaksPositionsBuffer, $peakNames) );
            
            //When creating a new test, ensure there are no old data in SmartOTU referencing a previous test of same Id
            $this->deleteTestDependenciesOnSmartOTU( $testId );
        }
        catch ( \Exception $e )
        {
            //get test addition error:
            $status = $this->sendReceive( SMTOtuApi::getLastTestAddStatusCommand() );
            
            if ( $status != SMTLinkTestException::OTU_ERROR_TEST_OK )
            {
                //remove OTDR trace files copied
                SMTOtdrTraceFileUtil::removeMeasurementReferenceTracesFromOTU( $referenceTraceFiles );
                
                throw new SMTLinkTestException( SMTLinkTestException::decodeOtuErrorCode( $status ) );
            }
        }
        
        //check test addition status:
        $status = $this->sendReceive( SMTOtuApi::getLastTestAddStatusCommand() );
        if ( $status != SMTLinkTestException::OTU_ERROR_TEST_OK )
        {
            //remove OTDR trace files copied
            SMTOtdrTraceFileUtil::removeMeasurementReferenceTracesFromOTU( $referenceTraceFiles );
            
            throw new SMTLinkTestException( SMTLinkTestException::decodeOtuErrorCode( $status ) );
        }
        
        return $testId;
    }
    
    /**
     * Add a monitoring test from the given DTO and update that DTO with values of the monitoring test added on OTU.
     *
     * @param SMTLinkTestDto $linkTestDto
     * @param string $peaksPositionsBuffer
     * @param string $peakNames
     * @param int    $peakCount 
     *
     * @return the identifier of the new test created
     * @throws SMTLinkTestException, SMTSocketException
     */
    private function addOtdrMonitoringTest( SMTLinkTestDto $linkTestDto, $peaksPositionsBuffer, $peakNames, $peakCount)
    {
        $testId = NULL;
        
        $peaksThresholdsBuffer = SMTOtuApi::EMPTY_BUFFER;
        
        //5 mandory thresholds: first marker, delta first-last marker, new peak, fiber extension, ORL
        $thresholdsCount = SMTThresholdBufferEncoderDecoder::TOPAZ_THRESHOLD_COUNT + $peakCount;
        $thresholdsBuffer = SMTThresholdBufferEncoderDecoder::encodeThresholds( $linkTestDto->getTestDetail()->getFirstMarkerThresholdsDb(), $linkTestDto->getTestDetail()->getDeltaFirstLastMarkersThresholdsDb(), $linkTestDto->getTestDetail()->getMonitoredPeaks(), $peakCount);        
        
        SMTLogger::getInstance()->trace( sprintf( "Start creating test with buffer %s ", $thresholdsBuffer ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        SMTLogger::getInstance()->trace( sprintf( "Start creating test with peak position buffer %s ", $peaksPositionsBuffer), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        
        try
        {
            //copy acquisition file from session directory to otu detection and localisation directories
            $referenceTraceFiles = SMTOtdrTraceFileUtil::copyMeasurementAsReferenceInOTU( $linkTestDto->getPortNumber(), $linkTestDto->getId(), $linkTestDto->getTestId(), $linkTestDto->getTestDetail()->getOtdrTraceName(), $linkTestDto->getTestDetail()->getOtdrTraceType() );
            
            //update OTDR trace name set as a reference in test
            $linkTestDto->getTestDetail()->setOtdrTraceName( $referenceTraceFiles->getOtdrFileName() );
        }
        catch ( \Exception $e )
        {
            $this->getContext()->getLogger()->traceException($e);
            throw new SMTLinkTestException( SMTLinkTestException::ERROR_MISSING_OTDR_TRACE );
        }
        
        try
        {
            //set attenuation critical thresholds (applied on OTU only if they have changed)
            $this->send( SMTOtuApi::setAttenuationCriticalThresholds( $linkTestDto->getTestDetail()->getAttenuationCriticalThresholdDbDefault() ) );
            
            //create test and its detection
            $monitoringMode = self::getOtuMonitoringType($linkTestDto->getTestDetail()->getPeakShiftMode(), ($peakCount > 0));
            $testId = $this->sendReceive( SMTOtuApi::addTestCommand( $linkTestDto->getId(), $referenceTraceFiles->getOtdrFileNameWithoutExtension(), $linkTestDto->getTestDetail()->getFirstMarkerDistanceM(), $linkTestDto->getTestDetail()->getLastMarkerDistanceM(), $linkTestDto->getTestDetail()->getMissingPeakThresholdPt(), $monitoringMode, $thresholdsCount, $thresholdsBuffer, $peaksPositionsBuffer, $peakNames) );
            
            //When creating a new test, ensure there are no old data in SmartOTU referencing a previous test of same Id
            $this->deleteTestDependenciesOnSmartOTU( $testId );
        }
        catch ( \Exception $e )
        {
            //get test addition error:
            $status = $this->sendReceive( SMTOtuApi::getLastTestAddStatusCommand() );
            
            if ( $status != SMTLinkTestException::OTU_ERROR_TEST_OK )
            {
                //remove OTDR trace files copied
                SMTOtdrTraceFileUtil::removeMeasurementReferenceTracesFromOTU( $referenceTraceFiles );
                
                throw new SMTLinkTestException( SMTLinkTestException::decodeOtuErrorCode( $status ) );
            }
        }
        
        //check test addition status:
        $status = $this->sendReceive( SMTOtuApi::getLastTestAddStatusCommand() );
        if ( $status != SMTLinkTestException::OTU_ERROR_TEST_OK )
        {
            //remove OTDR trace files copied
            SMTOtdrTraceFileUtil::removeMeasurementReferenceTracesFromOTU( $referenceTraceFiles );
            
            throw new SMTLinkTestException( SMTLinkTestException::decodeOtuErrorCode( $status ) );
        }
        
        //try to add localisation test
        try
        {
            //add localisation test , le monitoring de peaks s'effectue avec la detection => 0 peaks, empty buffer 
            $this->send( SMTOtuApi::addLocalisationTestCommand($testId, $linkTestDto->getTestDetail()->getFirstMarkerDistanceM(), $linkTestDto->getTestDetail()->getMissingPeakThresholdPt(), $referenceTraceFiles->getOtdrFileNameWithoutExtension(), 0, $peaksThresholdsBuffer, $peaksPositionsBuffer) );
        }
        catch ( \Exception $e )
        {
            //First, delete test created when adding detection:
            $this->deleteTest( $testId, TRUE );
            
            //remove OTDR trace files copied
            SMTOtdrTraceFileUtil::removeMeasurementReferenceTracesFromOTU( $referenceTraceFiles );
            
            throw new SMTLinkTestException( SMTLinkTestException::ERROR_TEST_ADD_LOC );
        }
        
        //check test addition status:
        $status = $this->sendReceive( SMTOtuApi::getLastTestAddStatusCommand() );
        if ( $status != SMTLinkTestException::OTU_ERROR_TEST_OK )
        {
            //remove OTDR trace files copied
            SMTOtdrTraceFileUtil::removeMeasurementReferenceTracesFromOTU( $referenceTraceFiles );
            
            throw new SMTLinkTestException( SMTLinkTestException::decodeOtuErrorCode( $status ) );
        }
        return $testId;
    }
    
    /**
     * Create a monitoring test from the given DTO and update that DTO with values of the monitoring test added on OTU.
     * 
     * @param SMTLinkTestDto $linkTestDto
     *
     * @return the identifier of the new test created
     * @throws SMTLinkTestException, SMTSocketException 
     */
    function createTest( SMTLinkTestDto $linkTestDto )
    {
        $testId = NULL;
        $peakNames = "";
        
        if ( self::acquireTestSetupLock() )
        {                    
            //no peak monitoring: empty buffer for peaks position by default
            $peaksPositionsBuffer = SMTOtuApi::EMPTY_BUFFER;
            
            $peakCount = 0;
            
            try 
            {
                if ( $linkTestDto->getTestDetail() == NULL || 
                     $linkTestDto->getTestDetail()->getFirstMarkerThresholdsDb() === NULL || 
                     $linkTestDto->getTestDetail()->getDeltaFirstLastMarkersThresholdsDb() === NULL )
                {
                    throw new SMTLinkTestException( SMTLinkTestException::ERROR_MISSING_TEST_THRESHOLD_PARAMETERS );
                }        
                
                if ( $linkTestDto->getTestDetail()->getFirstMarkerDistanceM() == $linkTestDto->getTestDetail()->getLastMarkerDistanceM() )
                {
                    throw new SMTLinkTestException( SMTLinkTestException::ERROR_LAST_MARKER_BADLY_SET );
                }
                
                if ( ($linkTestDto->getTestDetail()->getMonitoredPeaks()) != NULL && count($linkTestDto->getTestDetail()->getMonitoredPeaks()) > 0 )
                {
                    $peakCount = count($linkTestDto->getTestDetail()->getMonitoredPeaks());
                    if (!$linkTestDto->getTestDetail()->getExistingPeaksMonitored())
                    {
                        $peakCount = 0;
                    }
                }
                
                if ($peakCount > 0)
                {
                    $peaksPositionsBuffer = SMTPositionBufferEncoderDecoder::encodePeakPositions($linkTestDto->getTestDetail()->getMonitoredPeaks(), $peakCount);
                    
                    //peak names
                    foreach ($linkTestDto->getTestDetail()->getMonitoredPeaks() as $peak)
                    {
                        $peakNames.=$peak->getPeakName().self::PEAK_NAME_SEPARATOR;
                    }
                    $peakNames = rtrim($peakNames,self::PEAK_NAME_SEPARATOR);
                }
                
                if ( $linkTestDto->getTestDetail()->isHighsensTest() )
                {
                    $testId = $this->addHighsensMonitoringTest( $linkTestDto, $peaksPositionsBuffer, $peakNames, $peakCount);
                }
                else
                {
                    $testId = $this->addOtdrMonitoringTest( $linkTestDto, $peaksPositionsBuffer, $peakNames, $peakCount);
                }
                
                //update new testId in DTO
                $linkTestDto->setTestId( $testId );
            }
            catch( \Exception $e)
            {
                self::releaseTestSetupLock();
                throw $e;
            }
            self::releaseTestSetupLock();
        }
        else
        {
            $this->getContext()->getLogger()->trace("Couldn't acquire lock to create test: ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            throw new SMTLinkTestException( SMTLinkTestException::ERROR_ONLY_ONE_TEST_SETUP );
        }
        
        return $testId;
    }       
    
    /**
     * Delete the monitoring test on OTU and SmartOTU
     * @param number $testId
     * @param boolean $noLock FALSE by default. Don't acquire the lock to delete the test when the lock is already taken
     */
    function deleteTest( $testId, $noLock = FALSE )
    {
        if ( $noLock || self::acquireTestSetupLock() )
        {
            try
            {                
                $this->deleteTestDependenciesOnSmartOTU( $testId );
                SMTLogger::getInstance()->trace( sprintf( "Delete test %s ", $testId ), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
                $this->send( SMTOtuApi::deleteTestCommand( $testId ) );                
                if ( !$noLock )
                {
                    self::releaseTestSetupLock();
                }
            }
            catch(\Exception $e)
            {
                if ( !$noLock )
                {
                    self::releaseTestSetupLock();
                }
                $this->getContext()->getLogger()->traceException($e);
                throw new SMTLinkTestException( SMTLinkTestException::ERROR_DELETE_TEST );
            }
        }    
        else
        {
            $this->getContext()->getLogger()->trace("Couldn't acquire lock to delete test: ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            throw new SMTLinkTestException( SMTLinkTestException::ERROR_ONLY_ONE_TEST_SETUP );
        }
    }        
    
    /**
     * Delete test dependencies if needed: SmartOTU alarms and APC cache
     * 
     * @param integer $testId
     */
    private function deleteTestDependenciesOnSmartOTU( $testId )
    {
        try 
        {
            //delete test alarms on SmartOTU database and on SmartOTU directories
            $alarmHelper = new SMTAlarm();
            $alarmHelper->setContext( $this->getContext() );
            $alarmHelper->deleteOpticalAlarmOnSmartOtu( $testId );
            
            //cleanup APC memory cache
            SMTMemoryManager::delete( SMTTestAcquisitionParametersDto::getClass(), $testId );
        }
        catch( \Exception $e )
        {
            SMTLogger::getInstance()->traceException($e);
        }        
    }
    
    /**
     * Set link and test availability
     * 
     * @param boolean $linkId Link identifier
     * @param boolean $measureAllowed
     * @param boolean $monitoringAllowed
     */
    function setLinkTestAvailable( $linkId, $measureAllowed, $monitoringAllowed )
    {
        self::checkLinkTestId($linkId);
        
        SMTLogger::getInstance()->trace( sprintf( "Set link availability: measureAllowed %b monitoringAllowed %b ", $measureAllowed, $monitoringAllowed ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);

        $status = $measureAllowed? SMTOtuApi::RES_OK : SMTOtuApi::RES_KO;
        $this->send( SMTOtuApi::linkAvailableCommand( $linkId, $status ) );
         
        //retrieve test identifier:
        $linkTestIds = $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $linkId ) );
        $testId = self::parseLinkTestId( $linkTestIds );
        if ( $testId !== NULL && $testId >= 0 )
        {
            $status = $monitoringAllowed? SMTOtuApi::RES_OK : SMTOtuApi::RES_KO;
            $this->send( SMTOtuApi::testAvailableCommand( $testId, $status ) );
        }
    }
    

    /**
     * Rename link-test
     * 
     * @param string $id link Id
     * @param string $name new link name
     * @param int $portNumber (optional). If NULL, retrieve it
     */
    function renameLinkTest( $id, $name, $portNumber = NULL )
    {
        self::checkLinkName( $name );
        self::checkLinkTestId($id);

        SMTLogger::getInstance()->trace( sprintf( "Rename link to: %s ", $name ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);        

        if ( $portNumber == NULL )
        {
            $linkTestDto = $this->retrieveLinkInfo( $id );
            $portNumber = $linkTestDto->getPortNumber();
        }
        
        $asciiName = self::formatLinkNameInAscii($name, $portNumber);
        
        //update UTF-8 name
        $this->send( SMTOtuApi::linkUtf8RenameCommand( $id, $name ) );
        
        //update ascii link name (used for SMS)
        $this->send( SMTOtuApi::linkRenameCommand( $id, $asciiName ) );
        
        //delete link-test from APC memory cache to force its retrieval
        SMTMemoryManager::delete( SMTLinkTestDto::getClass(), $id );
    }    
    
    /**
     * Update parameters of the monitoring test on OTU: (classic monitoring : first marker, last marker and OTDR trace if needed / 
     *                                                   highsens monitoring:  attenuations and OTDR traces)
     * Update the reference localisation and detection traces if they have changed.
     *
     * No control of the change can be performed: the OTU will apply the changes synchronized with sequensor;
     * the check of the update will be performed with the execution of a new test.
     *
     * Update the given test DTO with values of monitoring test added on OTU.
     *
     * @param SMTLinkTestDto $linkTestDto
     *
     * @return Whether the OTDR trace was replaced
     * @throws SMTLinkTestException if the update fails
     */
    function updateTest( SMTLinkTestDto $linkTestDto )
    {           
    	$otdrTraceFileReplaced = FALSE;
    	$peakNames = "";
    	
        //retrieve test identifier of the test to update
        $testId = $linkTestDto->getTestId();
            
        if ( $linkTestDto->getTestDetail() == NULL )
        {
            throw new SMTLinkTestException( SMTLinkTestException::ERROR_MISSING_TEST_THRESHOLD_PARAMETERS );
        }
        else
        {
            if ( self::acquireTestSetupLock() )
            {
             	try
               	{
               		//no peak monitoring: empty buffer for peaks position by default
               		$peaksPositionsBuffer = SMTOtuApi::EMPTY_BUFFER;          		
               		$peakCount = 0;
               		
               		if ( $linkTestDto->getTestDetail() == NULL ||
               			 $linkTestDto->getTestDetail()->getFirstMarkerThresholdsDb() === NULL ||
               			 $linkTestDto->getTestDetail()->getDeltaFirstLastMarkersThresholdsDb() === NULL )
               		{
               			throw new SMTLinkTestException( SMTLinkTestException::ERROR_MISSING_TEST_THRESHOLD_PARAMETERS );
               		}
               		
               		if ( $linkTestDto->getTestDetail()->getFirstMarkerDistanceM() == $linkTestDto->getTestDetail()->getLastMarkerDistanceM() )
               		{
               			throw new SMTLinkTestException( SMTLinkTestException::ERROR_LAST_MARKER_BADLY_SET );
               		}
               		
               		if ( ($linkTestDto->getTestDetail()->getMonitoredPeaks()) != NULL && count($linkTestDto->getTestDetail()->getMonitoredPeaks()) > 0 )
               		{
               			$peakCount = count($linkTestDto->getTestDetail()->getMonitoredPeaks());
               			if (!$linkTestDto->getTestDetail()->getExistingPeaksMonitored())
               			{
               				$peakCount = 0;
               			}
               		}
               		
               		if ($peakCount > 0)
               		{
               			$peaksPositionsBuffer = SMTPositionBufferEncoderDecoder::encodePeakPositions($linkTestDto->getTestDetail()->getMonitoredPeaks(), $peakCount);
               			
               			//peak names
               			foreach ($linkTestDto->getTestDetail()->getMonitoredPeaks() as $peak)
               			{
               				$peakNames.=$peak->getPeakName().self::PEAK_NAME_SEPARATOR;
               			}
               			$peakNames = rtrim($peakNames,self::PEAK_NAME_SEPARATOR);
               		}
               		
                    //HIGHSENS
                    if ( $linkTestDto->getTestDetail()->isHighsensTest() )
                    {
                    	$otdrTraceFileReplaced = $this->updateHighsensTestParameters( $linkTestDto, $peaksPositionsBuffer, $peakNames, $peakCount);
                    }
                    //LEGACY
                    else
                    {
                    	$otdrTraceFileReplaced = $this->updateTestParameters( $linkTestDto, $peaksPositionsBuffer, $peakNames, $peakCount);
                    }
                }
                catch (\Exception $e )
                {
                	self::releaseTestSetupLock();
                	throw $e;
                }
                //delete test parameters in cache to force their retrieval
                SMTMemoryManager::delete( SMTTestAcquisitionParametersDto::getClass(), $testId );
                self::releaseTestSetupLock();
            }
            else
            {
                $this->getContext()->getLogger()->trace("Couldn't acquire lock to update test: ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
                throw new SMTLinkTestException( SMTLinkTestException::ERROR_ONLY_ONE_TEST_SETUP );
            }
        }

        return $otdrTraceFileReplaced;
    }
    
    /**
     * Update parameters of the monitoring test on OTU: (firs marker, last marker and OTDR trace if needed)
     * Update the reference localisation and detection traces if they have changed.
     * 
     * No control of the change can be performed: the OTU will apply the changes synchronized with sequensor;
     * the check of the update will be performed with the execution of a new test.
     * 
     * Update the given test DTO with values of monitoring test added on OTU.
     * 
     * @param SMTLinkTestDto $linkTestDto
     *
     * @return Whether the OTDR trace was replaced
     * @throws SMTLinkTestException if the update fails
     */
    function updateTestParameters( SMTLinkTestDto $linkTestDto, $peaksPositionsBuffer, $peakNames, $peakCount)
    {        
        $otdrTraceFileReplaced = FALSE;
            
            //retrieve test identifier of the test to update
            $testId = $linkTestDto->getTestId();
            
            //5 mandory thresholds: first marker, delta first-last marker, new peak, fiber extension, ORL
            $thresholdsCount = SMTThresholdBufferEncoderDecoder::TOPAZ_THRESHOLD_COUNT + $peakCount;
                
                $thresholdsBuffer = SMTThresholdBufferEncoderDecoder::encodeThresholds( $linkTestDto->getTestDetail()->getFirstMarkerThresholdsDb(), $linkTestDto->getTestDetail()->getDeltaFirstLastMarkersThresholdsDb(), $linkTestDto->getTestDetail()->getMonitoredPeaks(), $peakCount);
                
                //check if the OTDR references traces must be updated        
                $otdrFileName = $linkTestDto->getTestDetail()->getOtdrTraceName();
                
                SMTLogger::getInstance()->trace( sprintf( "Start updating test parameters; buffer: %s ", $thresholdsBuffer ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
                SMTLogger::getInstance()->trace( sprintf( "Start updating test with peak position buffer %s ", $peaksPositionsBuffer), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
                
                $otdrTraceFileReplaced = $this->isOtdrTraceFileToBeReplaced($testId, $otdrFileName );
                if ( $otdrTraceFileReplaced )
                {
                    try
                    {            
                        SMTLogger::getInstance()->trace( sprintf( "Start updating otdr trace: %s ", $otdrFileName ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
                        
                        //copy acquisition file from session directory or last acquisition directory to otu detection and localisation directories
                        $referenceTraceFiles = SMTOtdrTraceFileUtil::copyMeasurementAsReferenceInOTU( $linkTestDto->getPortNumber(), $linkTestDto->getId(), $linkTestDto->getTestId(), $otdrFileName, $linkTestDto->getTestDetail()->getOtdrTraceType(), FALSE );
                        
                        //update new trace name
                        $referenceTraceNameWithoutExtension = $referenceTraceFiles->getOtdrFileNameWithoutExtension();
                        $linkTestDto->getTestDetail()->setOtdrTraceName( $referenceTraceFiles->getOtdrFileName() );
                    }
                    catch ( \Exception $e )
                    {
                        $this->getContext()->getLogger()->traceException($e);
                        throw new SMTLinkTestException( SMTLinkTestException::ERROR_MISSING_OTDR_TRACE );
                    }
                }
                else
                {
                    //fetch current trace name:
                    $testDetailDto = $this->retrieveLinkTestDetail( $testId );
                    $referenceTraceNameWithoutExtension = $testDetailDto->getOtdrFileNameWithoutExtension();
                }
                
                try
                {
                    //set attenuation critical thresholds (applied on OTU only if they have changed)
                    $this->send( SMTOtuApi::setAttenuationCriticalThresholds( $linkTestDto->getTestDetail()->getAttenuationCriticalThresholdDbDefault() ) );            
                    
                    //update test detection and localization
                    $monitoringMode = self::getOtuMonitoringType($linkTestDto->getTestDetail()->getPeakShiftMode(), ($peakCount > 0));
                    $this->send( SMTOtuApi::updateTestCommand( $linkTestDto->getTestId(), $linkTestDto->getId(), $referenceTraceNameWithoutExtension, $linkTestDto->getTestDetail()->getFirstMarkerDistanceM(), $linkTestDto->getTestDetail()->getLastMarkerDistanceM(), $linkTestDto->getTestDetail()->getMissingPeakThresholdPt(), $monitoringMode, $thresholdsCount, $thresholdsBuffer, $peaksPositionsBuffer, $peakNames, $linkTestDto->isMonitoringAllowed() ) );
                }
                catch ( \Exception $e )
                {
                    //no control of the change can be performed here: the OTU will apply the changes synchronized with sequensor
                    //the check of the update will be performed through the execution of a new test
                                
                    //remove OTDR trace files copied (if new traces have been copied)
                    if ( isset( $referenceTraceFiles ) && $referenceTraceFiles != NULL )
                    {
                        //non, il faudrait supprimer les fichiers memorises ( _copy ), mais cela ne peut pas etre fait ici car la modif est synchronisee sur le sequenceur
                        //SMTOtdrTraceFileUtil::removeMeasurementReferenceTracesFromOTU( $referenceTraceFiles );
                    }
            
                    throw new SMTLinkTestException( SMTLinkTestException::ERROR_TEST_UPDATE_FAILED );
                }                
        
        return $otdrTraceFileReplaced;
    }
    
    /**
     * Update parameters of the monitoring test on OTU: (attenuation and OTDR trace if needed)
     * Update the reference detection traces if they have changed.
     *
     * No control of the change can be performed: the OTU will apply the changes synchronized with sequensor;
     * the check of the update will be performed with the execution of a new test.
     *
     * Update the given test DTO with values of monitoring test added on OTU.
     *
     * @param SMTLinkTestDto $linkTestDto
     *
     * @return Whether the OTDR trace was replaced
     * @throws SMTLinkTestException if the update fails
     */
    function updateHighsensTestParameters( SMTLinkTestDto $linkTestDto, $peaksPositionsBuffer, $peakNames, $peakCount)
    {
    	$otdrTraceFileReplaced = FALSE;
    	$peakThresholdsBuffer= SMTOtuApi::EMPTY_BUFFER; 
    	
    	//retrieve test identifier of the test to update
    	$testId = $linkTestDto->getTestId();
    	   	
    	$thresholdsBuffer = SMTThresholdBufferEncoderDecoder::encodeHighsenMonitoringThreshold( $linkTestDto->getTestDetail()->getFirstMarkerThresholdsDb() );
    	if ($peakCount > 0)
    	{
    		$peakThresholdsBuffer = SMTThresholdBufferEncoderDecoder::encodeHighsenPeaksMonitoringThreshold( $linkTestDto->getTestDetail()->getMonitoredPeaks(), $peakCount);
    	}
    	
    	//check if the OTDR references traces must be updated
    	$otdrFileName = $linkTestDto->getTestDetail()->getOtdrTraceName();
    	
    	SMTLogger::getInstance()->trace( sprintf( "Start updating highsens test parameters; buffer: %s ", $thresholdsBuffer ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    	SMTLogger::getInstance()->trace( sprintf( "Start updating highsens test with peak position buffer %s ", $peaksPositionsBuffer), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    	
    	$otdrTraceFileReplaced = $this->isOtdrTraceFileToBeReplaced($testId, $otdrFileName );
    	if ( $otdrTraceFileReplaced )
    	{
    		try
    		{
    			SMTLogger::getInstance()->trace( sprintf( "Start updating otdr trace: %s ", $otdrFileName ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    			
    			//copy acquisition file from session directory or last acquisition directory to otu detection and localisation directories
    			$referenceTraceFiles = SMTHighsensOtdrTraceFileUtil::copyMeasurementAsReferenceInOTU( $linkTestDto->getPortNumber(), $linkTestDto->getId(), $linkTestDto->getTestId(), $otdrFileName, $linkTestDto->getTestDetail()->getOtdrTraceType(), FALSE );
    			
    			//update new trace name
    			$referenceTraceNameWithoutExtension = $referenceTraceFiles->getOtdrFileNameWithoutExtension();
    			$linkTestDto->getTestDetail()->setOtdrTraceName( $referenceTraceFiles->getOtdrFileName() );
    		}
    		catch ( \Exception $e )
    		{
    			$this->getContext()->getLogger()->traceException($e);
    			throw new SMTLinkTestException( SMTLinkTestException::ERROR_MISSING_OTDR_TRACE );
    		}
    	}
    	else
    	{
    		//fetch current trace name:
    		$testDetailDto = $this->retrieveLinkTestDetail( $testId );
    		$referenceTraceNameWithoutExtension = $testDetailDto->getOtdrFileNameWithoutExtension();
    	}
    	
    	try
    	{
    		//set attenuation critical thresholds (applied on OTU only if they have changed)
    		$this->send( SMTOtuApi::setAttenuationCriticalThresholds( $linkTestDto->getTestDetail()->getAttenuationCriticalThresholdDbDefault() ) );
    		
    		//update test detection and localization
    		$peakMonitoringType= self::getOtuMonitoringType($linkTestDto->getTestDetail()->getPeakShiftMode(), ($peakCount > 0));
    		$this->send( SMTOtuApi::updateHighsensTestCommand( $linkTestDto->getTestId(), $linkTestDto->getId(), $referenceTraceNameWithoutExtension, $linkTestDto->getTestDetail()->getFirstMarkerDistanceM(), $linkTestDto->getTestDetail()->getLastMarkerDistanceM(), $linkTestDto->getTestDetail()->getMissingPeakThresholdPt(), $peakMonitoringType, $peakCount, $thresholdsBuffer, $peakThresholdsBuffer, $peaksPositionsBuffer, $peakNames, $linkTestDto->isMonitoringAllowed()) );
    	}
    	catch ( \Exception $e )
    	{
    		//no control of the change can be performed here: the OTU will apply the changes synchronized with sequensor
    		//the check of the update will be performed through the execution of a new test
    		
    		//remove OTDR trace files copied (if new traces have been copied)
    		if ( isset( $referenceTraceFiles ) && $referenceTraceFiles != NULL )
    		{
    			//non, il faudrait supprimer les fichiers memorises ( _copy ), mais cela ne peut pas etre fait ici car la modif est synchronisee sur le sequenceur
    			//SMTOtdrTraceFileUtil::removeMeasurementReferenceTracesFromOTU( $referenceTraceFiles );
    		}
    		
    		throw new SMTLinkTestException( SMTLinkTestException::ERROR_TEST_UPDATE_FAILED );
    	}
    	
    	return $otdrTraceFileReplaced;
    }
    
    /**
     * Translate SmartOTU peak monitoring mode into OTU mode
     * 
     * @param string $monitoringType SmartOTU peak monitoring mode
     * @param boolean $hasPeakMonitoring Whether monitoring peak is enabled
     * @return OTU monitoring type
     */
    static function getOtuMonitoringType( $monitoringType, $hasPeakMonitoring )
    {
        return $hasPeakMonitoring? (strcasecmp(SMTPeakShiftMode::REFERENCE_TRACE , $monitoringType) == 0? SMTPeakShiftMode::OTU_REFERENCE_TRACE : SMTPeakShiftMode::OTU_TOLERANCE) : SMTPeakShiftMode::OTU_DEFAULT_MODE_WITHOUT_PEAK;
    }
    
    /**
     * Translate SmartOTU peak monitoring mode into OTU mode
     *
     * @param string $monitoringType OTU monitoring type
     * @param boolean $hasPeakMonitoring Whether monitoring peak is enabled
     * 
     * @return SmartOTU peak monitoring mode
     */
    static function getSmartOtuMonitoringType( $monitoringType, $hasPeakMonitoring)
    {
        return $hasPeakMonitoring? (strcasecmp(SMTPeakShiftMode::OTU_REFERENCE_TRACE , $monitoringType) == 0? SMTPeakShiftMode::REFERENCE_TRACE : SMTPeakShiftMode::TOLERANCE) : SMTPeakShiftMode::getDefaultPeakShiftMode();
    }
    
    /**
     * Check whether the otdr trace file must be updated: the trace must be updated if its name has changed.
     * 
     * @param string $testId
     * @param string $otdrFileName
     * 
     * @return boolean whether the otdr trace file must be updated.
     */
    function isOtdrTraceFileToBeReplaced( $testId, $otdrFileName )
    {
        //fetch current trace name:
        $testDetailDto = $this->retrieveLinkTestDetail( $testId );
        $currentTraceName = SMTHighsensOtdrTraceFileUtil::removeExtension( basename($testDetailDto->getOtdrTraceName()) );
        $newFileName = SMTHighsensOtdrTraceFileUtil::removeExtension(basename( $otdrFileName ) );
        
        //SMTLogger::getInstance()->trace("testId ".$testId );
        //SMTLogger::getInstance()->trace($testDetailDto->getOtdrTraceName() );
        //SMTLogger::getInstance()->trace($otdrFileName );
        return strcmp( $currentTraceName, $newFileName) != 0;
    }
    
    /**
     * Retrieve main link and test info.
     * 
     * @param $id number The link id
     * @param $sequensorOn boolean Whether OTU sequencer is on; NULL by default to force retrieval of otu sequencer
     * @param $portCount Number of port available. NULL by default 
     * 
     * @return \app\services\monitoring\SMTLinkTestDto
     * @throws \Exception If link is not found
     */
    function retrieveLinkTest( $id, $sequensorOn = NULL, $portCount = NULL )
    {            
        //retrieve link info
        try 
        {
            $linkDto = $this->parseLinkInfo( $id );
        }
        catch( \Exception $e)
        {
            throw new SMTLinkTestException( SMTLinkTestException::ERROR_LINK_NOT_FOUND, sprintf("Link not found %s!", $id ) );
        }

        //retrieve main test info
        $testId = self::parseLinkTestId( $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $id ) ) );
        
        if ( self::checkTestId( $testId, FALSE ) )
        {
            $linkDto->setTestId( $testId );
        }
        
        //sequencer status 
        if ( $sequensorOn === NULL )
        {
            $linkDto->setSequensorOn( $this->isSequensorOn() );
        }
        else
        {
            $linkDto->setSequensorOn( $sequensorOn );
        }
        
        //don't retrieve last monitoring date, measurement allowed... if the link is not connected to a port
        if ( $portCount === NULL || ( $portCount !== NULL && $linkDto->getPortNumber() <= $portCount ) )
        {
            if( !isset ($this->timezoneOffset) || $this->timezoneOffset === NULL)
            {
                //retrieve time zone offset and store it, to avoid to call it for each link retrieval
                $dto = SMTSystemConfigurationDto::fetch( $this->getContext()->getDatabase() );
                $this->timezoneOffset = $dto->getTimezoneSec();
            }
            
            $dateString = $this->sendReceive( SMTOtuApi::getLinkLastMonitoringDateCommand( $id ) );
            $linkDto->setLastMonitoringTimeStamp( SMTUtil::getTimestampUTC( $dateString, $this->getContext()->getDatabase(), $this->timezoneOffset ) );
            
            //retrieve both link availability and test availability, because if link is not available, it forbids the execution of a measurement on the link
//             $linkDto->setMeasurementAllowed( $this->sendReceive( SMTOtuApi::getLinkAvailableCommand( $id ) ) == SMTOtuApi::RES_OK );
            if ( self::checkTestId( $testId, FALSE ) )
            {
            	$this->parseTestInfo($testId, $linkDto );
            }    
        }
    
        return $linkDto;
    }    
    

    /**
     * Retrieve main link info: link name, link availability, testId (no monitoring status,...)
     *
     * @param $id number The link id
     * @param $monitoringInfo boolean Whether link additional monitoring info (monitoring time stamp, monitoring status) must be retrieved. FALSE by default.
     *
     * @return \app\services\monitoring\SMTLinkTestDto
     * @throws \Exception if the link is not found
     */
    function retrieveLinkInfo( $id, $monitoringInfo = FALSE )
    {
        //retrieve link info
        try 
        {
            $linkDto = $this->parseLinkInfo( $id );
        }
        catch ( \Exception $e )
        {
            throw new SMTLinkTestException( SMTLinkTestException::ERROR_LINK_NOT_FOUND, sprintf("Link not found %s!", $id ) );
        }
    
        //retrieve test Id
        $testId = self::parseLinkTestId( $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $id ) ) );
//         $linkDto->setMeasurementAllowed( $this->sendReceive( SMTOtuApi::getLinkAvailableCommand( $id ) ) == SMTOtuApi::RES_OK );
         
        if ( self::checkTestId( $testId, FALSE ) )
        {
            $linkDto->setTestId( $testId );
            $this->parseTestInfo($testId, $linkDto );
        }
        
        if ( $monitoringInfo )
        {
            if( !isset ($this->timezoneOffset) || $this->timezoneOffset === NULL)
            {
                //retrieve time zone offset and store it, to avoid to call it for each link retrieval
                $dto = SMTSystemConfigurationDto::fetch( $this->getContext()->getDatabase() );
                $this->timezoneOffset = $dto->getTimezoneSec();
            }
            
            $dateString = $this->sendReceive( SMTOtuApi::getLinkLastMonitoringDateCommand( $id ) );
            $linkDto->setLastMonitoringTimeStamp( SMTUtil::getTimestampUTC( $dateString, $this->getContext()->getDatabase(), $this->timezoneOffset ) );
        }         
        
        return $linkDto;        
    }    
    
    /**
     * Whether monitoring test is activated
     * 
     * @param String $testId
     * @return boolean Whether monitoring test is activated 
     */
    public function isMonitoringAllowed( $testId )
    {
        return $this->sendReceive( SMTOtuApi::getTestAvailableCommand( $testId ) ) == SMTOtuApi::RES_OK;
    } 
    
    /**
     * Whether there is no monitoring test scheduled.
     * 
     * @param array $testsIds Monitoring test ids
     * @return boolean True if there is no monitoring test scheduled.
     */
    public function isNoTestRunning( $testsIds )
    {
        $noTestRunning = TRUE;
        
        if ( is_array( $testsIds ) )
        {
            foreach ( $testsIds as $testId )
            {
                if ( $this->isMonitoringAllowed( $testId ) )
                {
                    $noTestRunning = FALSE;
                    break;
                }
            }
        }
        
        return $noTestRunning;
    }
    
    /**
     * Returns the number of peaks monitored on the test
     * 
     * @param SMTOtuSocket $socket
     * @param int $testId
     */
    static function getPeakCount(SMTOtuSocket $socket, $testId)
    {
    	$peakNumber = 0;
    	try
    	{
    		$testDetailString = $socket->sendReceive( SMTOtuApi::getTestDescriptionCommand( $testId ), FALSE );
    		preg_match_all("/\"(?:[^\"]*?)\",|(?:[^,]*?),/", $testDetailString.',', $output_array);
    		
    		//tests if it's a normal monitoring test or a high sensibility test (highsens, tapping...)
    		$mode = rtrim($output_array[0][5],",");
    		if ( SMTTestMode::isHighsensOrTappingMode( $mode) ) //high sensibility
    		{
    			$peakNumber= rtrim($output_array[0][14],",");
    		}
    		else
    		{
    			$thresholdCount= rtrim($output_array[0][11],",");
    			$peakNumber= $thresholdCount - SMTThresholdBufferEncoderDecoder::TOPAZ_THRESHOLD_COUNT;
    		}
    	}
    	catch( Exception $e )
    	{
    		$this->getContext()->getLogger()->traceException($e);
    		$peakNumber = 0;
    	}
    	return $peakNumber;
    }
    
    /**
     * Fetch monitoring test details (SMTTestDetailDto)
     * 
     * @param number $testId
     * 
     * @return \app\services\monitoring\SMTTestDetailDto
     * @throws \Exception if the test is not found
     */
    function retrieveLinkTestDetail( $testId )
    {
        $testDetail=new SMTTestDetailDto();
        
        try 
        {
            $testDetailString = $this->sendReceive( SMTOtuApi::getTestDescriptionCommand( $testId ), FALSE );
            
            preg_match_all("/\"(?:[^\"]*?)\",|(?:[^,]*?),/", $testDetailString.',', $output_array);
            $dateBegining = rtrim($output_array[0][0],",");
            $period= rtrim($output_array[0][1],",");
            $linkId= rtrim($output_array[0][2],",");
            $availability= rtrim($output_array[0][3],",");
            $custom= rtrim($output_array[0][4],",");
            
            //tests if it's a normal monitoring test or a high sensibility test (highsens, tapping...)
            $mode = rtrim($output_array[0][5],",");
            if ( SMTTestMode::isHighsensOrTappingMode( $mode) ) //high sensibility
            {
                $testDetail->setTestMode($mode);
                $shortAcq = rtrim($output_array[0][6],",");                
                $testDetail->setShortAcq(strcmp($shortAcq,"1") == 0? TRUE : FALSE);
                $traceFileName= rtrim(str_replace("\"","",$output_array[0][7]),",");
                $firstMarker= rtrim($output_array[0][9],",");
                $lastMarker= rtrim($output_array[0][10],",");                
                $thresholdBuffer= rtrim($output_array[0][11],",");
                $thresholdNewReflectance= rtrim($output_array[0][12],",");
                $thresholdFiberLengthExtension= rtrim($output_array[0][13],",");
                $peakNumber= rtrim($output_array[0][14],",");
                $thresholdPeakBuffer= rtrim($output_array[0][15],",");
                $positionPeakBuffer= rtrim($output_array[0][16],",");
                $monitoringType = rtrim($output_array[0][17],",");
                $peakShift= rtrim($output_array[0][18],",");
                $peakNames= rtrim(str_replace("\"","", $output_array[0][19]),"," );
            }
            else if ( SMTTestMode::isOtdrMode($mode) ) //otdr standard
            {
            	$testDetail->setTestMode($mode);
            	$traceFileName= rtrim(str_replace("\"","",$output_array[0][6]),",");
                $firstMarker= rtrim($output_array[0][7],",");
                $lastMarker= rtrim($output_array[0][8],",");
                $peakShift= rtrim($output_array[0][9],",");
                $monitoringType = rtrim($output_array[0][10],",");
                $thresholdCount= rtrim($output_array[0][11],",");
                $thresholdBuffer= rtrim($output_array[0][12],",");
                $positionPeakBuffer= rtrim($output_array[0][13],",");
                $peakNames= rtrim(str_replace("\"","", $output_array[0][14]),"," );                
            }
            else
            {
            	$this->getContext()->getLogger()->trace(sprintf("Unsupported monitoring test type: %s", $mode), SMTLogger::ERROR);
            	throw new SMTLinkTestException( SMTLinkTestException::ERROR_TEST_NOT_FOUND, sprintf("Unsupported monitoring test type: %s", $mode));
            }
        }
        catch ( \Exception $e )
        {
            throw new SMTLinkTestException( SMTLinkTestException::ERROR_TEST_NOT_FOUND, sprintf("Test not found %s!", $testId ) );
        }
        
        $minAttenuationAlarmThreshold = $this->getMinAttenuationAlarmThreshold( $this->getModuleRunningNumber() );
        $hysteresisAttenuationAlarmThresholdListDto = $this->getHysteresisAttenuationAlarmThresholdList( $this->getModuleRunningNumber() );
                
        $testDetail->setTestId( $testId );
        $testDetail->setFirstMarkerDistanceM( $firstMarker );
        $testDetail->setLastMarkerDistanceM( $lastMarker );
        $testDetail->setOtdrTraceName( SMTOtdrTraceFileUtil::checkSorFileExtension( $traceFileName ) );

        if (SMTTestMode::isTappingMode($mode) )
        {
        	$testDetail->setMonitoringAcceptableNoiseMarginDbDefault( SMTSmartOtuSettings::getTappingAcceptableNoiseMarginDbDefault() );
        	$testDetail->setAttenuationToBtoDeltaDbDefault( SMTSmartOtuSettings::getTappingToBtoDeltaDbDefault() );
        }
        else
        {
	        $testDetail->setMonitoringAcceptableNoiseMarginDbDefault( SMTSmartOtuSettings::getMonitoringAcceptableNoiseMarginDbDefault() );
        	$testDetail->setAttenuationToBtoDeltaDbDefault( SMTSmartOtuSettings::getAttenuationToBtoDeltaDbDefault() );
        }
        
        try 
        {
        	$testCriticalThresholdString = $this->sendReceive( SMTOtuApi::getGet_Critical_Threshold( $testId ), FALSE );
        	list ($criticalThreshold, $criticalThresholdHyst)= explode(",",$testCriticalThresholdString);
        	$testDetail->setAttenuationCriticalThresholdDbDefault( floatval($criticalThreshold)/100.0 );
        }
        catch ( \Exception $e )
        {
        	$this->getContext()->getLogger()->trace(sprintf("Failed to fetch critical threshold for test: %d", $testId), SMTLogger::ERROR);
        	$testDetail->setAttenuationCriticalThresholdDbDefault( SMTSmartOtuSettings::getAttenuationCriticalThresholdDbDefault() );
        }

        $testDetail->setHysteresisAttenuationThresholdsDbList($hysteresisAttenuationAlarmThresholdListDto);
        
        //PEAKS
        //PEAKS
        //PEAKS
        $testDetail->setDefaultPeakThresholdsDb( SMTSmartOtuSettings::getDefaultPeakThresholdsDb() );
        $testDetail->setMaxNumberOfMonitorablePeaks(SMTSmartOtuThresholds::DEFAULT_MAX_NUMBER_MONITORABLE_PEAKS);
        $testDetail->setMonitorablePeakThresholdDb( SMTSmartOtuSettings::getMonitorablePeakThresholdDbDefault() );
        
        $testDetail->setPeakMonitoringOptionEnabled(SMTLicence::isPeakMonitoringOption($this->getContext()));
        $testDetail->setPeakToBtoDelta(SMTSmartOtuSettings::getPeakToBtoDeltaDbDefault());
        $testDetail->setMissingPeakThresholdPt($peakShift);
        
        $testDetail->setMinAttenuationAlarmThreshold( $minAttenuationAlarmThreshold);
        $testDetail->setTappingMonitoringOptionEnabled(SMTLicence::isTappingMonitoringOption($this->getContext()));
        
        $peakCount = 0;
        $bufferDecoder = new SMTThresholdBufferEncoderDecoder( $thresholdBuffer );
        
        //if thresholds are invalid, catch exception to let user fix the problem:
        //threshold will be null in test detail
        try 
        {              
            if ( $testDetail->isHighsensTest())
            {
                $positionCount = $peakNumber;
                $bufferDecoder->decodeHighsensMonitoringThresholdBuffer();
                $bufferDecoder->decodeHighsensPeakThresholdsBuffer($thresholdPeakBuffer, $peakNumber);
                $testDetail->setFirstMarkerThresholdsDb( $bufferDecoder->getMonitoringThresholdsDb() );
                $testDetail->setDeltaFirstLastMarkersThresholdsDb( $bufferDecoder->getMonitoringThresholdsDb() );
            }
            else
            {
                $positionCount = $thresholdCount - SMTThresholdBufferEncoderDecoder::TOPAZ_THRESHOLD_COUNT;
                $bufferDecoder->decodeThresholdsBuffer($positionCount);
                $testDetail->setFirstMarkerThresholdsDb( $bufferDecoder->getFirstMarkerThresholdsDb() );
                $testDetail->setDeltaFirstLastMarkersThresholdsDb( $bufferDecoder->getDeltaFirstLastMarkersThresholdsDb() );
            }
            //override default value with real hysteresis value
            $testDetail->setAttenuationToBtoDeltaDbDefault( $bufferDecoder->getHysteresis( SMTSmartOtuSettings::getAttenuationToBtoDeltaDbDefault() ));
            
            if ( $positionCount > 0)
            {
                $index = 0;
                $testDetail->setExistingPeaksMonitored(TRUE);
                                
                $positionBufferDecoder = new SMTPositionBufferEncoderDecoder($positionPeakBuffer);
                
                $positionBufferDecoder->decodePositionBuffer($positionCount);
                $positions = $positionBufferDecoder->getPeakPositions();
                
                $names = explode(self::PEAK_NAME_SEPARATOR, $peakNames);
                
                $peaksThreshold = $bufferDecoder->getPeakThresholds();                                
                foreach ($peaksThreshold as &$peakThresholds)
                {
                    $peak = new SMTMonitoringPeakDto();
                    $peak->setPeakThresholds($peakThresholds);
                    
                    if ( $index< count($positions))
                    {
                        $peak->setPeakTopDistanceM($positions[$index]);
                    }
                    else
                    {
                        $this->getContext()->getLogger()->trace(sprintf("Invalid peak position index: %d, peak count %d", $index, count($positions) ), SMTLogger::ERROR);
                    }
                    
                    if ( $index < count($names ) )
                    {
                        $peak->setPeakName($names[$index]);
                    }
                    else
                    {
                        $this->getContext()->getLogger()->trace(sprintf("Invalid peak name index: %d, peak count %d", $index, count($names) ), SMTLogger::ERROR);
                    }
                    
                    $testDetail->addMonitoredPeak($peak);
                    $index++;
                    $peakCount++;
                }
            }
            else 
            {
                $testDetail->setExistingPeaksMonitored(FALSE);
            }
        }
        catch(\Exception $e)
        {
            $this->getContext()->getLogger()->traceException($e);
        }
        
        $peakShiftMode= self::getSmartOtuMonitoringType($monitoringType, ($peakCount > 0));
        $testDetail->setPeakShiftMode($peakShiftMode);    
        
        return $testDetail;        
    }
    
    /**
     * Fetch the link info from the testId (SMTLinkTestDto)
     *
     * @param number $testId
     * @param boolean $monitoringInfo Whether link additional monitoring info (monitoring time stamp, monitoring status) must be retrieved. FALSE by default.
     * 
     * @return \app\services\monitoring\SMTLinkTestDto
     */
    function retrieveLinkInfoFromTest( $testId, $monitoringInfo = FALSE )
    {
        $testDetail = NULL;
        try 
        {           
            //fetch the test
//             $testDetail = $this->sendReceive( SMTOtuApi::getTestDescriptionCommand( $testId ) );
//             list( $dateBegining, $period, $linkId, $availability, $custom, $traceFileName, $firstMarker, $lastMarker, $peakShift, $monitoringType, $thresholdCount, $thresholdBuffer ) = explode (",", $testDetail);
            $linkId = $this->getLinkIdFromTest($testId);
        }
        catch(\Exception $e )
        {
            throw new SMTLinkTestException( SMTLinkTestException::ERROR_TEST_NOT_FOUND, sprintf("Test not found %s!", $testId ) );            
        }
        
        return $this->retrieveLinkInfo( $linkId, $monitoringInfo );    
    }
    
    /**
     * Returns linkId by resquesting test description
     * @param int $testId
     * @return int
     */
    function getLinkIdFromTest($testId)
    {
        $testDetail = $this->sendReceive( SMTOtuApi::getTestDescriptionCommand( $testId ) );        
        list( , , $linkId ) = explode (",", $testDetail);
        return $linkId;
    }
    
    /**
     * Fetch test acquisition parameters
     *
     * @param int $testId Test Identifier
     * @param int $linkId Link Identifier (NULL by default). If not set, it is retrieved by a query. 
     * @param boolean $forceTestAcquisitionParametersRetrieval Force test acquisition parameters retrieval (FALSE by default).
     * @return SMTTestAcquisitionParametersDto
     */
    function fetchTestAcquisitionParameters( $testId, $linkId = NULL, $forceTestAcquisitionParametersRetrieval = FALSE )
    {
        if ( (( $testDetailedDto = SMTMemoryManager::fetch( SMTTestAcquisitionParametersDto::getClass(), $testId ) ) == NULL) || $forceTestAcquisitionParametersRetrieval )
        {
            $testDetailedDto = new SMTTestAcquisitionParametersDto();
            $testDetailedDto->setTestId( $testId );
             
            //retrieve test info
            //TODO SHOULD USE NEW SI FUNCTION TO RETRIEVE PULSE,RANGE,... IN SI AND NOT FORMATTED VALUES:
            // PC : OTU:MON:INF:DET? 0
            // RTU : 20 s,1625 nm,300 ns,79740 m,5m
    
            $testDetailedInfo = $this->sendReceive( sprintf( SMTOtuApi::CMD_info_test_mon, $testId ) );
            list ($acquisitionTimeSec, $wavelength, $pulse, $range, $resolution)= explode(",",$testDetailedInfo);
            
            if ( $linkId === NULL )
            {
//                 $testDetail = $this->sendReceive( SMTOtuApi::getTestDescriptionCommand( $testId ) );                
//                 list( $dateBegining, $period, $linkId, $availability, $custom, $traceFileName, $firstMarker, $lastMarker, $peakShift, $monitoringType, $thresholdCount, $thresholdBuffer ) = explode (",", $testDetail);
                $linkId = $this->getLinkIdFromTest($testId);
            }
            
            $testDetailedDto->setLinkId( $linkId );
            $testDetailedDto->setPulse( $pulse );
            $testDetailedDto->setRange( $range );
            $testDetailedDto->setResolution( $resolution );
            $testDetailedDto->setAcquisitionTime( $acquisitionTimeSec );
            $testDetailedDto->setWavelengthFrequency( $wavelength );
            $testDetailedDto->save();
        }
    
        return $testDetailedDto;
    }    
    
    
    /**
     * Init a new link test DTO with its test detail info initialized with default values and a default link-test name.
     * If link already exists, retrieve link details and set test detail info initialized with default values.
     * 
     * @param String $linkId If set, the link is retrieved. If NULL and if no link is found on given port, a new link DTO with Id NULL is created.
     * @param int $portNumber Used if linkId is not set: we check that no link is found on the given port.
     * @param int $moduleRunningNumber Used if linkId is not set. (if $moduleRunningNumber is an int encoded in a string, the function try to convert it in integer)
     * @return \app\services\monitoring\SMTLinkTestDto
     * @throws exception if the link is not found
     */
    function initLinkTest( $linkId, $portNumber, $moduleRunningNumber, $mode = SMTTestMode::HIGHSENS)
    {
        //fetch the link if it already exists
        if ( SMTUtil::isInt( $linkId ) && $linkId >= 0 )
        {
            SMTLogger::getInstance()->trace( sprintf( "Init test object with default values on module %s, on port %s ", $moduleRunningNumber, $portNumber ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
                
            //retrieve link info
            try 
            {
                $linkTestDto = $this->parseLinkInfo( $linkId );
            }
            catch ( \Exception $e )
            {
                throw new SMTLinkTestException( SMTLinkTestException::ERROR_LINK_NOT_FOUND, sprintf("Link not found %s!", $linkId ) );
            }            
//             $linkTestDto->setMeasurementAllowed( $this->sendReceive( SMTOtuApi::getLinkAvailableCommand( $linkId ) ) == SMTOtuApi::RES_OK );
            //if measure is allowed allows monitoring by default
            if ( $linkTestDto->isMeasurementAllowed() )
            {
                $linkTestDto->setMonitoringAllowed( TRUE );
            }            
        }
        else
        {
            SMTLogger::getInstance()->trace( sprintf( "Create new default link test object with default values on module %s, on port %s ", $moduleRunningNumber, $portNumber ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
            $linkTestDto =  new SMTLinkTestDto();
            $linkTestDto->setTestMode($mode);
            $linkTestDto->setModuleRunningNumber($moduleRunningNumber);
            $linkTestDto->setPortNumber($portNumber);
            //new test: allows measurement and scheduling by default
            $linkTestDto->setMeasurementAllowed(TRUE);
            $linkTestDto->setMonitoringAllowed( TRUE );            
            
            //check that no link exists on the given port if no LinkId was given
            //and update $linkTestDto with linkId found
            $linkExists = $this->checkLinkAndUpdateDto( $linkTestDto );
            
            //if a link with a test was found throw an exception
            if ( $linkExists )
            {
            	if ( !$this->hasTest( $linkTestDto->getId() ) )
            	{
            		$this->deleteLink( $linkTestDto->getId() );
            		$linkTestDto->resetId();
            	}
            	else
            	{
                	throw new SMTLinkTestException( SMTLinkTestException::ERROR_LINK_ALREADY_EXIST, sprintf("Link already exists %s!", $linkTestDto->getId() ) );
            	}
                //$this->initLinkTest( $linkTestDto->getId(), $portNumber, $moduleRunningNumber );
            }
            //set a default link name
            else
            {
                $linkTestDto->setName( self::buildDefaultLinkTestName( $portNumber ) ) ;
            }
        }
        
        $minAttenuationAlarmThreshold = $this->getMinAttenuationAlarmThreshold( $linkTestDto->getModuleRunningNumber() );
        $hysteresisAttenuationAlarmThresholdListDto = $this->getHysteresisAttenuationAlarmThresholdList( $linkTestDto->getModuleRunningNumber() );
        
        $testDetail=new SMTTestDetailDto();
        $testDetail->setTestMode($mode);
        $testDetail->setFirstMarkerDistanceM( 0 );
        $testDetail->setLastMarkerDistanceM( 0 );
                        
        $testDetail->setMonitoringAcceptableNoiseMarginDbDefault( SMTSmartOtuSettings::getMonitoringAcceptableNoiseMarginDbDefault() );
        $testDetail->setAttenuationCriticalThresholdDbDefault( SMTSmartOtuSettings::getAttenuationCriticalThresholdDbDefault() );
        
        if (SMTTestMode::isTappingMode($mode) )
        {
        	$testDetail->setAttenuationToBtoDeltaDbDefault( SMTSmartOtuSettings::getTappingToBtoDeltaDbDefault() );
        	$firstMarkerThreshold = SMTSmartOtuSettings::getDefaultTappingThresholds();
        	$deltaFirstLastMarkerThreshold= SMTSmartOtuSettings::getDefaultTappingThresholds();
        	$testDetail->setMonitoringAcceptableNoiseMarginDbDefault( SMTSmartOtuSettings::getTappingAcceptableNoiseMarginDbDefault() );
        	//pas de seuil d'attenuation minimum positionne, on utilise la valeur par defaut du seuil de detection tapping
        	if ( $minAttenuationAlarmThreshold == 0 )
        	{
        		$testDetail->setMinAttenuationAlarmThreshold( SMTSmartOtuSettings::getDefaultTappingThresholds()->getPositiveToMajorThreshold() );
        	}
        	else
        	{
        		$testDetail->setMinAttenuationAlarmThreshold( $minAttenuationAlarmThreshold );
        	}
        }
        else
        {
        	$testDetail->setAttenuationToBtoDeltaDbDefault( SMTSmartOtuSettings::getAttenuationToBtoDeltaDbDefault() );
        	$firstMarkerThreshold = SMTSmartOtuSettings::getDefaultFirstMarkerAttenuationThresholds();
        	$deltaFirstLastMarkerThreshold= SMTSmartOtuSettings::getDefaultBudgetVariationThresholds();
        	$testDetail->setMinAttenuationAlarmThreshold( $minAttenuationAlarmThreshold );
        }
        
        $firstMarkerThreshold->setMonitored(SMTTestMode::isOtdrMode($mode));
        $deltaFirstLastMarkerThreshold->setMonitored(TRUE);
        
        $testDetail->setFirstMarkerThresholdsDb( $firstMarkerThreshold );
        $testDetail->setDeltaFirstLastMarkersThresholdsDb( $deltaFirstLastMarkerThreshold );
        $testDetail->setHysteresisAttenuationThresholdsDbList($hysteresisAttenuationAlarmThresholdListDto);
                
        //PEAKS
        //PEAKS
        //PEAKS
        $testDetail->setDefaultPeakThresholdsDb( SMTSmartOtuSettings::getDefaultPeakThresholdsDb() );
        $testDetail->setMaxNumberOfMonitorablePeaks(SMTSmartOtuThresholds::DEFAULT_MAX_NUMBER_MONITORABLE_PEAKS);
        $testDetail->setMonitorablePeakThresholdDb( SMTSmartOtuSettings::getMonitorablePeakThresholdDbDefault() );
        
        $testDetail->setPeakMonitoringOptionEnabled(SMTLicence::isPeakMonitoringOption($this->getContext()));        
        $testDetail->setPeakToBtoDelta(SMTSmartOtuSettings::getPeakToBtoDeltaDbDefault());
        $testDetail->setMissingPeakThresholdPt(SMTSmartOtuSettings::getMissingPeakThresholdPtDefault());
        $testDetail->setPeakShiftMode( SMTPeakShiftMode::getDefaultPeakShiftMode() );
        
        $testDetail->setMinAttenuationAlarmThreshold( $minAttenuationAlarmThreshold );
        $testDetail->setTappingMonitoringOptionEnabled(SMTLicence::isTappingMonitoringOption($this->getContext()));
        $linkTestDto->setTestDetail($testDetail);
            
        return $linkTestDto;
    }    
    
    /**
     * Whether OTU sequensor is not stopped
     * 
     * @return boolean true if sequensor is random or sequential (should be only sequential for smartOTU)
     */
    function isSequensorOn()
    {
        return ( $this->sendReceive( SMTOtuApi::getMonitoringSequensorStatus() ) != self::STOP );        
    }
    
    /**
     * Set OTU sequensor to SEQuential
     * 
     */
    function setSequensorOn()
    {
        $this->send( SMTOtuApi::setMonitoringSequensorStatus( self::SEQUENTIAL ) );
    }    
    
    /**
     * Retrieve the OTU output ports count (if no switch is available, port count = 1).
     * 
     * @return int
     */
    function getPortsCount()
    {
        $portCount = 1;
        //if no switch is available, the port count is 1 (OTDR module)
        if ( $this->configuredSwitch != NULL )
        {
            $portCount = $this->configuredSwitch->getOutputPorts();
        }
        else
        {            
            try
            {
                $portCount = $this->sendReceive( SMTOtuApi::CMD_switch_config_port_count);
                if ( !SMTOtuCurrentPortDto::isValid($portCount) )
                {
                    $portCount = 1;
                }
            }
            catch (\Exception $e)
            {
                SMTLogger::getInstance()->traceException($e);
                $portCount = 1;
            }
        }
        
        if ( ($portCount <= 1) && (SMTOtuType::isOtu8KV2() || SMTOtuType::isMts8KV2()))
        {
            try
            {
                SMTModule::checkModuleRunningNumber( $this->getModuleRunningNumber());
                $portCount = 1;
            }
            catch (\Exception $e)
            {
                $portCount = 0;
            }
        }
        
        return intval($portCount);
    }

    /**
     * Returns Minimum threshold for attenuation alarm or 0 if none is set
     *
     * @return float Minimum threshold for attenuation alarm or 0 if none is set
     */
    function getMinAttenuationAlarmThreshold( $moduleRunningNumber )
    {
        $min = $this->sendReceive( SMTOtuApi::getMinAttenuationAlarmThreshold( SMTOtuApi::RES_MOD.$moduleRunningNumber) );
        return floatval($min);
    }
    
    /**
     * Returns Hysteresis  for attenuation alarm threshold
     *
     * @return float Hysteresis for attenuation alarm threshold
     */
    function getMinHysteresisAttenuationAlarmThreshold( $moduleRunningNumber )
    {
    	$hyst = $this->sendReceive( SMTOtuApi::getMinHysteresisAttenuationAlarmThreshold( SMTOtuApi::RES_MOD.$moduleRunningNumber) );
    	return floatval($hyst);
    }
    
    /**
     * Returns Hysteresis  for attenuation alarm threshold
     *
     * @return SMTOtuTestThresholdHysteresisListDto Hysteresis for attenuation alarm threshold
     */
    function getHysteresisAttenuationAlarmThresholdList( $moduleRunningNumber )
    {
    	$hysteresisList = $this->sendReceive( SMTOtuApi::getHysteresisAttenuationAlarmThresholdList( SMTOtuApi::RES_MOD.$moduleRunningNumber) );
    	$hysteresisArray = SMTUtil::splitStringByComma( $hysteresisList);
    	$list = new SMTOtuTestThresholdHysteresisListDto();
    	
    	$index = 0;
    	while ($index < count($hysteresisArray))
    	{
    		$attenuation = trim( $hysteresisArray[$index] );
    		$hysteresis  = trim( $hysteresisArray[$index + 1] );
    		
    		$list->addHysteresis( $attenuation, $hysteresis );
    		
    		$index+=2;
    	}
    	
    	return $list;
    }

    /**
     * Whether the OTU has a registered switch.
     *
     * @return boolean
     */
    function hasSwitches()
    {
        //if no switch is available, the port count is 1 (OTDR module)
        return ( $this->getConfiguredSwitch() != NULL );
    }
    
    /**
     * Retrieves configured switch dto if it has not alredy been retrieved.
     * 
     * @return app\serviceshelper\otau\SMTMacroSwitch
     */
    private function getConfiguredSwitch()
    {
        if ( $this->configuredSwitch == NULL )
        {
            $this->configuredSwitch = SMTSwitchManager::retrieveConfiguredSwitch($this->getContext()->getOtuSocket());
        }
        return $this->configuredSwitch;
    }
    
    /**
     * Retrieve the SmartOTU OTDR module (only one module) number.
     *
     * @return int or NULL if not set
     */
    function getModuleRunningNumber()
    {
        if ( $this->moduleDto == NULL )
        {
            $moduleManager = new SMTModule();
            $moduleManager->setContext( $this->getContext() );
            $this->moduleDto = $moduleManager->fetchOTDRModuleConfig();
        }
        
        return ($this->moduleDto != NULL)? $this->moduleDto->getModuleRunningNumber() : NULL;
    }
    
    /**
     * Retrieve the port count when a switch not managed by OTU is used
     *
     * @return int or -1 if not set
     */
    function getVirtualSwitchPortCount()
    {    
        $portCount = -1;
        try 
        { 
            $portCount = $this->sendReceive( SMTOtuApi::getVirtualSwitchPortCount() );
        }
        catch (\Exception $e)
        {
            //nothing to do, virtual switch not used
        }
        return $portCount;
    }
    
    /**
     * Retrieve link info and returns a new link-test Dto
     *      
     * @param number $linkId
     * 
     * @return SMTLinkTestDto 
     * @throws \Exception
     */
    function parseLinkInfo( $linkId )
    {
        $linkDto = new SMTLinkTestDto();
        $linkDto->setId( $linkId );
        $linkInfo = $this->sendReceive( SMTOtuApi::getlinkUtf8Command( $linkId ), FALSE );
        
        if ( $linkInfo != NULL && $linkInfo != "" )
        {
            //look for  ","  sequence to explode the string into an array $conf
            //remove spaces and double quotes in array values
            $linkInfoArray = SMTUtil::splitStringByComma( $linkInfo );
            $name = trim( $linkInfoArray[self::LINK_UTF8_NAME_INDEX] );
            if ( strlen($name) <= 0 )
            {
                $name = trim( $linkInfoArray[self::LINK_NAME_INDEX] );
            }
            $linkDto->setName( $name );
            $linkDto->setModulePosition( trim( $linkInfoArray[self::LINK_MODULE_INDEX] ) );
            
            $linkDto->setMeasurementAllowed( strcasecmp( trim( $linkInfoArray[self::LINK_AVAILABLE_INDEX] ) , SMTOtuApi::RES_OK) == 0 );
            
            $switchBufferDecoder = new SMTSwitchBufferEncoderDecoder( $linkInfoArray[self::SWITCH_BUFFER_INDEX] );
            $switchOtu = $switchBufferDecoder->getSwitchOut();
            if ( $switchOtu != NULL )
            {
                $linkDto->setPortNumber( intval( trim( $switchOtu) ) );
            }
            //handle case of no switch available
            else
            {
                $linkDto->setPortNumber( 1 );
            }
            $gpsFileName = $linkInfoArray[self::GPS_FILE_NAME_INDEX];
            if ( !empty($gpsFileName) )
            {
                $linkDto->setGpsFileName( $gpsFileName );
                $linkDto->setKmlUrl( SMTGpsRoute::buildkmlUrl( $gpsFileName));
            }
        }                
        else
        {
            throw new SMTLinkTestException(SMTLinkTestException::ERROR_INVALID_LINK_ID, "Couldn't parse empty link info");
        }    
        return $linkDto;
    }
    
    /**
     * Retrieve test main info and update link test info
     *
     * @param number $testId
     *
     * @throws \Exception
     */
    function parseTestInfo($testId, SMTLinkTestDto $linkDto )
    {
    	$testInfo= $this->sendReceive( SMTOtuApi::getTestInfoCommand( $testId), FALSE );
    	
    	if ( ($testInfo != NULL) && (strlen($testInfo) > 0) )
    	{
    		//look for  ","  sequence to explode the string into an array $conf
    		//remove spaces and double quotes in array values
    		$testInfoArray = SMTUtil::splitStringByComma( $testInfo ); 
    		$monitoringAllowed = strcasecmp( trim($testInfoArray[self::TEST_AVAILABLE_INDEX]), SMTOtuApi::RES_OK ) == 0;
    		$linkDto->setTestMode( trim( $testInfoArray[self::TEST_MODE_INDEX] ));
    		$linkDto->setMonitoringAllowed( $monitoringAllowed);
    		
    		// the test is unknown (DTSS...), don't process it!
    		if( !(SMTTestMode::isHighsensOrTappingMode( $linkDto->getTestMode() )) && !(SMTTestMode::isOtdrMode( $linkDto->getTestMode() ) ) )
    		{
    			$this->getContext()->getLogger()->trace('Unknown test: mode: '. $linkDto->getTestMode() .'; ignoring test', SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    			$linkDto->resetTestId( );
    		}
    	}
    	
    }
    
    /**
     * Fetch link info without test info and without monitoring additional info from APC memory cache if it exists or retrieve it from OTU parser
     *
     * @param int $linkId Link Identifier.
     * @param boolean $forceLinkTestRetrieval Whether we force the retrieval of link-test dto from OTU parser and update the APC cache.
     * 
     * @return SMTLinkTestDto
     */
    function fetchLinkInfoFromCache( $linkId, $forceLinkTestRetrieval = FALSE )
    {
        if ( ( $linkDto = SMTMemoryManager::fetch( SMTLinkTestDto::getClass(), $linkId ) ) == NULL || $forceLinkTestRetrieval )
        {
            $linkDto = $this->parseLinkInfo( $linkId );
            $linkDto->save();
        }
    
        return $linkDto;
    }    
    
    /**
     * Fetch link info without test info and without monitoring additional info from APC memory cache if it exists or retrieve it from OTU parser
     *
     * @param int $testId Test Identifier.
     * @param boolean $forceLinkTestRetrieval Whether we force the retrieval of link-test dto from OTU parser and update the APC cache.
     *
     * @return SMTLinkTestDto
     */
    function fetchLinkInfoFromTestAndFromCache( $testId, $forceLinkTestRetrieval = FALSE )
    {
        $linkDto = NULL;
        $linkId = -1;
        try
        {
            //fetch the test
//             $testDetail = $this->sendReceive( SMTOtuApi::getTestDescriptionCommand( $testId ) );            
//             list( $dateBegining, $period, $linkId, $availability, $custom, $traceFileName, $firstMarker, $lastMarker, $peakShift, $monitoringType, $thresholdCount, $thresholdBuffer ) = explode (",", $testDetail);
            $linkId = $this->getLinkIdFromTest($testId);
        }
        catch(\Exception $e )
        {
            throw new SMTLinkTestException( SMTLinkTestException::ERROR_TEST_NOT_FOUND, sprintf("Test not found %s!", $testId ) );
        }
        
        if ( ( $linkDto = SMTMemoryManager::fetch( SMTLinkTestDto::getClass(), $linkId ) ) == NULL || $forceLinkTestRetrieval )
        {
            $linkDto = $this->parseLinkInfo( $linkId );
            $linkDto->save();
        }
    
        return $linkDto;
    } 
        
    /**
     * Retrieve the unique test id (for SmartOTU one test per link) from the given array of tests Ids in a link
     * 
     * @param array $linkTestIds
     * @return $testId the test identifier or null if none was found
     */
    public static function parseLinkTestId( $linkTestIds )
    {
        $testId = NULL;
        $linkIdsArray = explode (",", $linkTestIds );
        if ( count( $linkIdsArray ) > 0 )
        {
            $testId = trim( array_shift( $linkIdsArray ) );
            if ( ( !SMTUtil::isInt( $testId ) || $testId < 0 ) )
            {
                $testId = NULL;
            }               
        }
    
        return $testId;
    }    
    
    /**
     * Build a default Link Name
     * 
     * @param int $portNumber
     */
    static function buildDefaultLinkTestName( $portNumber )
    {
        return SMTLinkTestDto::DEFAULT_LINK_TEST_NAME_PREFIX.$portNumber;
    }
    
    /**
     * Returns optical port number of the given test
     *
     * @param SMTOtuSocket $socket
     * @param unknown $testId
     */
    public static function getPortNumber(SMTOtuSocket $socket,  $testId)
    {
    	return $socket->sendReceive( SMTOtuApi::getMonitoringTestPortCommand($testId));
    }
    
    /**
     * Create test data on the given port number
     * 
     * @param portNumber WARNING: start from 1 
     * 
     * @return \app\services\monitoring\SMTLinkTestDto
     */
    static function createTestData( $portNumber )
    {
    	$mode = SMTTestMode::OTDR;
        $linkTestDto = new SMTLinkTestDto();
        $linkTestDto->setTestMode($mode);
        $linkTestDto->setMeasurementAllowed(TRUE);
        $linkTestDto->setMonitoringAllowed(TRUE);
        $linkTestDto->setModuleRunningNumber( 2 );
        $linkTestDto->setPortNumber( intval( $portNumber ) );
         
        $linkTestDto->setName('Link'.($portNumber - 1) );
         
        $testDetail=new SMTTestDetailDto();
        $testDetail->setFirstMarkerDistanceM( 50.000 );
        $testDetail->setLastMarkerDistanceM( 1000.000 );
         
        $testDetail->setMonitoringAcceptableNoiseMarginDbDefault( SMTSmartOtuSettings::getMonitoringAcceptableNoiseMarginDbDefault() );
        $testDetail->setAttenuationToBtoDeltaDbDefault( SMTSmartOtuSettings::getAttenuationToBtoDeltaDbDefault() );        
        
        $testDetail->setAttenuationCriticalThresholdDbDefault( SMTSmartOtuSettings::getAttenuationCriticalThresholdDbDefault() );
        $testDetail->setTestMode($mode);
        
        $firstMarkerThreshold = SMTSmartOtuSettings::getDefaultFirstMarkerAttenuationThresholds();         
        $firstMarkerThreshold->setMonitored(TRUE);
        $testDetail->setFirstMarkerThresholdsDb( $firstMarkerThreshold );           
                 
        $deltaFirstLastMarkerThreshold = SMTSmartOtuSettings::getDefaultBudgetVariationThresholds();
        $deltaFirstLastMarkerThreshold->setMonitored(TRUE);
        
        $testDetail->setDeltaFirstLastMarkersThresholdsDb( $deltaFirstLastMarkerThreshold );
        
        $hysteresisAttenuationAlarmThresholdListDto = $this->getHysteresisAttenuationAlarmThresholdList( $linkTestDto->getModuleRunningNumber() );
        $testDetail->setHysteresisAttenuationThresholdsDbList($hysteresisAttenuationAlarmThresholdListDto);
        
        
        $linkTestDto->setTestDetail($testDetail);
        return $linkTestDto;
    }

    private static function getLockTestSetupFile()
    {
        if ( self::$testSetupLockFile == NULL )
        {
            $handle = fopen( __DIR__.self::LOCK_TEST_SETUP_FILE, "w+");
             
            self::$testSetupLockFile = ( $handle != FALSE)? $handle : NULL;
        }
        return self::$testSetupLockFile;
    }
    
    /**
     * Acquire TestSetup exclusive lock
     *
     * @return boolean
     */
    private static function acquireTestSetupLock()
    {
        $retries = 0;
        $max_retries = 10;
    
        //SMTLogger::getInstance()->trace( "acquireTestSetupLock" );
    
        // keep trying to get a lock as long as possible ( max about 4s n(n+1)(2n+1)/6)
        do
        {
            if ($retries > 0)
            {
                usleep( 10000 * ( $retries*$retries ) );
            }
            $retries += 1;
        } while ( ( !flock(self::getLockTestSetupFile(), LOCK_EX, $eWouldBlock) || $eWouldBlock ) && $retries <= $max_retries);
    
        return ($retries < $max_retries);
    }
    
    /**
     * Releases TestSetup exclusive lock
     *
     * @return boolean
     */
    private static function releaseTestSetupLock()
    {
        //SMTLogger::getInstance()->trace( "releaseTestSetupLock" );
        return isset(self::$testSetupLockFile) && self::$testSetupLockFile != NULL? flock( self::$testSetupLockFile, LOCK_UN ) : TRUE;
    }    
}
?>