<?php
namespace app\util\upload;

//chdir to root dir
chdir('../../../');

/**
 * include autoloader
 */
require_once('lib/restler.php');

use app\http\SMTContext;
use app\services\maintenance\SMTUploadStatusDto;
use app\serviceshelper\maintenance\SMTOtuBackupRestore;
use app\util\SMTLogger;
use app\util\SMTUtil;
use app\util\upload\SMTUploadStatus;
use Luracast\Restler\Format\JsonFormat;
use app\serviceshelper\monitoring\SMTGpsRoute;
use app\serviceshelper\maintenance\SMTLicence;
use app\serviceshelper\maintenance\SMTCertificate;
use app\serviceshelper\maintenance\SMTUpgrade;


function init( SMTContext $context )
{
	$success = FALSE;	

	//Restore PHP session context: PHP session context must be restored to handle session time-out and user identifiers.
	$session = $context->getSessionContext();

	if ( $session->getUser() == null )
	{
		$success = FALSE;
		$context->closeSession();
		SMTLogger::getInstance()->trace('Not logged in! ', SMTLogger::ERROR);
	}
	else if ( $context->getHttpRequest()->checkReferer() && $context->getHttpRequest()->checkToken( "" ) )
	{
		$success = TRUE;
// 		SMTLogger::getInstance()->trace("User authentified ", SMTLogger::INFO);
	}
	else
	{
		SMTLogger::getInstance()->trace('Invalid access! ', SMTLogger::ERROR);
	}
	return $success;
}

/////////////////////////////////////////////////////////////////////////////////////////
//
// KML check
//
//
/////////////////////////////////////////////////////////////////////////////////////////
function checkKmlExtension($filename)
{
	return strcasecmp(pathinfo($filename, PATHINFO_EXTENSION), 'kml' ) == 0;
}
function checkKml($filename, SMTContext $context)
{
	$checked = FALSE;
	if ( checkKmlExtension($filename) )
	{
		$checked = SMTGpsRoute::checkXMLFile($_FILES['datafile']['tmp_name']);
		if (!$checked)
		{
			unlink($_FILES['datafile']['tmp_name']);
		}
	}
	else
	{
		unlink($_FILES['datafile']['tmp_name']);
	}
	return $checked;
}

/////////////////////////////////////////////////////////////////////////////////////////
//
// License check
//
//
/////////////////////////////////////////////////////////////////////////////////////////
function checkLicenseExtension($filename)
{
	return strcasecmp(pathinfo($filename, PATHINFO_EXTENSION), 'lic' ) == 0;
}
function checkLicense($filename, SMTContext $context)
{
	$checked = FALSE;
	if ( checkLicenseExtension($filename) )
	{
		$licenseHelper = new SMTLicence();
		$licenseHelper->setContext( $context);
		
		if ( !empty( $_FILES['datafile']['tmp_name']) )
		{
			try 
			{
				$licenseHelper->installLicense($filename, $_FILES['datafile']['tmp_name']);				
				$checked = TRUE;
			}
			catch(\Exception $e)
			{
				unlink($_FILES['datafile']['tmp_name']);
				SMTLogger::getInstance()->trace('Invalid licence file content : '.$filename, SMTLogger::ERROR);
			}
		}
	}
	else
	{
		unlink($_FILES['datafile']['tmp_name']);
	}
	return $checked;
}

/////////////////////////////////////////////////////////////////////////////////////////
//
// Backup check
//
//
/////////////////////////////////////////////////////////////////////////////////////////
function checkBackupExtension($filename)
{
	return SMTUtil::startsWith($filename, 'smartotu-backup') && SMTUtil::endsWith( $filename, '.tar.gz');
}
function checkBackup($filename, SMTContext $context)
{
	$checked = FALSE;
	if ( checkBackupExtension($filename) )
	{
		//remove old backups
		array_map('unlink', glob( SMTOtuBackupRestore::OTU_BACKUP_DIR.SMTOtuBackupRestore::OTU_BACKUP_NAME_PREFIX."*".SMTOtuBackupRestore::OTU_BACKUP_NAME_SUFFIX ) );
		
		//check backup integrity
		$backupHelper = new SMTOtuBackupRestore();
		$backupHelper->setContext( $context);
		if ( !empty( $_FILES['datafile']['tmp_name']) )
		{
			try
			{
				$backupHelper->checkBackupIntegrity($filename, $_FILES['datafile']['tmp_name']);
				$checked = TRUE;
			}
			catch(\Exception $e)
			{
				unlink($_FILES['datafile']['tmp_name']);
				SMTLogger::getInstance()->trace('Check Backup integrity failed for file : '.$filename, SMTLogger::ERROR);
			}
		}	
	}
	else
	{
		unlink($_FILES['datafile']['tmp_name']);
	}
	return $checked;
}

/////////////////////////////////////////////////////////////////////////////////////////
//
// Patch check
//
//
/////////////////////////////////////////////////////////////////////////////////////////
function checkPatchExtension($filename)
{
	return SMTUtil::startsWith($filename, 'patch_otu') && SMTUtil::endsWith( $filename, '.tar.gz');
}
function checkPatch($filename, SMTContext $context)
{
	$checked = FALSE;
	if ( checkPatchExtension($filename) )
	{
		$checked = TRUE;
	}
	else
	{
		unlink($_FILES['datafile']['tmp_name']);
	}
	return $checked;
}

/////////////////////////////////////////////////////////////////////////////////////////
//
// Release upgrade check
//
//
/////////////////////////////////////////////////////////////////////////////////////////
function check8kReleaseExtension($filename)
{
	return ( (strpos($filename, '8kv2') !== FALSE) && SMTUtil::endsWith( $filename, '.tar.gz') );
}
function check8kRelease($filename, SMTContext $context)
{
	$checked = FALSE;
	if ( check8kReleaseExtension($filename) )
	{
		try
		{
			$checked = SMTUpgrade::checkUpgradeArchive($_FILES['datafile']['tmp_name'], $context);
		}
		catch(\Exception $e)
		{
			unlink($_FILES['datafile']['tmp_name']);
			SMTLogger::getInstance()->trace('Invalid 8k release file content : '.$filename, SMTLogger::ERROR);
		}
	}
	else
	{
		unlink($_FILES['datafile']['tmp_name']);
	}
	return $checked;
}
function check5kReleaseExtension($filename)
{
	return ( strpos($filename, '5000') !== FALSE) && (strcasecmp(pathinfo($filename, PATHINFO_EXTENSION), 'tar' ) == 0 );
}
function check5kRelease($filename, SMTContext $context)
{
	$checked = FALSE;
	if ( check5kReleaseExtension($filename) )
	{
		try
		{
			$checked = SMTUpgrade::checkUpgradeArchive($_FILES['datafile']['tmp_name'], $context);
		}
		catch(\Exception $e)
		{
			unlink($_FILES['datafile']['tmp_name']);
			SMTLogger::getInstance()->trace('Invalid 5k release file content : '.$filename, SMTLogger::ERROR);
		}
	}
	else
	{
		unlink($_FILES['datafile']['tmp_name']);
	}
	return $checked;
}
/////////////////////////////////////////////////////////////////////////////////////////
//
// Certificates check
//
//
/////////////////////////////////////////////////////////////////////////////////////////
function checkKeyCertificateExtension($filename)
{
	return (strcasecmp(pathinfo($filename, PATHINFO_EXTENSION), 'key' ) == 0 );
}
function checkKeyCertificate($filename, SMTContext $context)
{
	$checked = FALSE;
	if ( checkKeyCertificateExtension($filename) )
	{
		try
		{
			SMTCertificate::checkSslCertificate($filename, $_FILES['datafile']['tmp_name'], 'sslkey');
			$checked = TRUE;
		}
		catch(\Exception $e)
		{
			//unlink done in check
			SMTLogger::getInstance()->trace('Invalid certificate key file content : '.$filename, SMTLogger::ERROR);
		}
	}
	else
	{
		unlink($_FILES['datafile']['tmp_name']);
	}
	return $checked;
}
function checkCertificateExtension($filename)
{
	return (strcasecmp(pathinfo($filename, PATHINFO_EXTENSION), 'crt' ) == 0 );
}
function checkCertificate($filename, SMTContext $context)
{
	$checked = FALSE;
	if ( checkCertificateExtension($filename) )
	{
		try 
		{
			SMTCertificate::checkSslCertificate($filename, $_FILES['datafile']['tmp_name'], 'cert');
			$checked = TRUE;
		}
		catch(\Exception $e)
		{
			//unlink done in check
			SMTLogger::getInstance()->trace('Invalid certificate file content : '.$filename, SMTLogger::ERROR);
		}
	}
	else
	{
		unlink($_FILES['datafile']['tmp_name']);
	}
	return $checked;
}
function checkChainCertificate($filename, SMTContext $context)
{
	$checked = FALSE;
	if ( checkCertificateExtension($filename) )
	{
		try
		{
			SMTCertificate::checkSslCertificate($filename, $_FILES['datafile']['tmp_name'], 'chain');
			$checked = TRUE;
		}
		catch(\Exception $e)
		{
			//unlink done in check
			SMTLogger::getInstance()->trace('Invalid certificate chain file content : '.$filename, SMTLogger::ERROR);
		}
	}
	else
	{
		unlink($_FILES['datafile']['tmp_name']);
	}
	return $checked;
}

/**
 * Check php upload errors
 * Handle copy in the target directory 
 * 
 * @param string $path
 * @param boolean $isBackup Whether the uploaded file is a backup
 * @param boolean $copyFile True by default: used to disable rename of uploaded file.
 *
 * @return SMTUploadStatusDto
 */
function upload_file( $path, $checkCallBack='', $copyFile = TRUE )
{
	$tempFilePath = $_FILES['datafile']['tmp_name'];
	$fileName = strip_tags($_FILES['datafile']['name']);
	//retrieve errors for php upload in its temporary directory
	$err=$_FILES['datafile']['error'];
	$success=FALSE;
	
    $uploadStatus = new SMTUploadStatusDto();
    $context = new SMTContext();
    
    if ( init( $context ) )
    {
    	SMTLogger::getInstance()->trace( 'Temporary uploaded file: '.$tempFilePath.' err:'.$err);
    	
		//if no upload error, check content
		if ( ($err <= 0) && strlen( $checkCallBack) > 0 )
		{
			if( call_user_func(__NAMESPACE__.'\\'.$checkCallBack, $fileName, $context) != TRUE)
			{
				SMTLogger::getInstance()->trace('Invalid file content : '.$fileName, SMTLogger::ERROR);
				$err = '1111';
			}
		}		
		
		$uploadStatus->setFileName(rawurlencode($fileName));
			
		//process php error status at the end of the upload
		if ($err > 0)
		{
			//delete file
			if ( file_exists($tempFilePath) )
			{
				@unlink($tempFilePath);
			}
	
		    $message='Upload failed! ';	    
		    $uploadStatus->setProgress( SMTUploadStatusDto::NO_UPLOAD );
		    
			switch($err)
			{
				case '1':
					$message.='php.ini max file size exceeded.';
					$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::MAX_FILE_SIZE_EXCEEDED );
					break;
				case '2':
					$message.='max file size exceeded.';
					$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::MAX_FILE_SIZE_EXCEEDED );
					break;
				case '3':
					$message.='file upload was only partial.';
					$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::PARTIAL_FILE_UPLOAD );
					break;
				case '4':
					$message.='no file was attached.';
					$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::NO_FILE_ATTACHED );
					break;
				case '7':
					$message.='Error: file permission denied.';
					$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::FILE_PERMISSION_DENIED );
					break;
				case '1111':
					$message.='Error: invalid file.';
					$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::INVALID_FILE);
					break;
				default :
					$message.='Unexpected error occured.';
					$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::UNEXPECTED_ERROR );
					break;
			}
			
			SMTLogger::getInstance()->trace($message, SMTLogger::ERROR );
		}
		//no php error at the end of the upload
		else
		{
		    $message = "";
		    
		    //we replace existing file
		    if (file_exists($path.$fileName))
			{
				$message='file already exist.';
			}
			
			//move file to its target destination
			if ( $copyFile)
			{
				$success = @move_uploaded_file($tempFilePath, $path.$fileName);
			}
			else
			{
				$success = TRUE;
			}
			
			if ( $success )
			{
	    		$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::NO_ERROR );
	    		$uploadStatus->setProgress( SMTUploadStatusDto::UPLOAD_COMPLETE );
	    		
	    		//retrieve upload rate from cache
	    		$cacheDto = SMTUploadStatus::getCurrentUploadInfoFromCache();
	    		if ( $cacheDto != NULL )
	    		{
	        		$uploadRate = $cacheDto->getRateMBs();
	        		$uploadStatus->setRateMBs( $uploadRate );
	    		}
	    		$message.=' File '.$fileName.' has been successfully uploaded.';
	    		SMTLogger::getInstance()->trace($message);
			}
			//failed to copy file to its target destination
			else
			{
			    $uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::COPY_FAILED );
			    $uploadStatus->setProgress( SMTUploadStatusDto::NO_UPLOAD );
			    
			    $message.=' Error: Failed to copy temporary file to destination file '.$path.$fileName;
			    SMTLogger::getInstance()->trace($message, SMTLogger::ERROR);
			}
		}
		
		//upload is finished, remove the php upload progress from APC
		SMTUploadStatus::cleanPhpUploadProgressFromApcCache();
		
		//save the upload status in APC cache
		$uploadStatus->save();
    }
    else
    {
    	SMTLogger::getInstance()->trace('Failed to init upload engine', SMTLogger::ERROR);
    	
    	//delete file
    	@unlink($tempFilePath);
    	
    	$uploadStatus->setFileName(rawurlencode($fileName));
    	$uploadStatus->setUploadErrorStatus( SMTUploadStatusDto::UNEXPECTED_ERROR );
    	
    	//upload is finished, remove the php upload progress from APC
    	SMTUploadStatus::cleanPhpUploadProgressFromApcCache();
    	
    	//save the upload status in APC cache
    	$uploadStatus->save();
    	
    }
	
    if (!$success)
    {
    	// add error code 401 in case of upload failure
    	@header('Cache-Control: no-cache, must-revalidate', true, 401 );    	
    }
    
	return $uploadStatus;
}

/**
 * Build JSON response
 * 
 * @param SMTUploadStatusDto $uploadStatus
 * @return string
 */
function buildJsonResponse( SMTUploadStatusDto $uploadStatus )
{
    $uploadStatusJson = $uploadStatus->getJsonData();
    
    @header('Cache-Control: no-cache, must-revalidate');
    @header('Expires: 0');
    @header('Content-Type: application/json; charset=utf-8');
    
    $responseFormat = new JsonFormat();
    $responseFormat->setCharset('utf-8');
    
    return $responseFormat->encode( $uploadStatusJson, FALSE );
}

?>