<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\sharedmemory;

use app\util\SMTLogger;

/**
 * Performs read and writes operations on APC shared and persistent memory module.
 * Used to persist data in memory and share data between sessions.
 *
 * @author Sylvain Desplat
 */
class SMTAPCMemoryUtil
{
    const MEMORY_STATUS = "memory_status";
    const MEMORY_STATUS_ON = "on";
    const MEMORY_MANAGER_LOCK_FILE = "/../../tmp/memoryManager.lock";
    /**
     * apc_info metadata structures of APC user cache contains a list of metadata for each stored data:
     * 
     *  0 => array (
     * 'info' => 'app\\settings\\SMTOtuStatusCacheDto',
     * 'ttl' => 0,
     * 'type' => 'user',
     * 'num_hits' => 10,
     * 'mtime' => 1373961137,
     * 'creation_time' => 1373961137,
     * 'deletion_time' => 0,
     * 'access_time' => 1373961148,
     * 'ref_count' => 0,
     * 'mem_size' => 1112,
     * )
     * 1 => array (
     *'info' => 'app\\settings\\SMTSmartOtuSettingsCache',
     *'ttl' => 0, 
     * ...
     *  
     * "info" key give us the identifiers of our dtos objects
     * 
     * @var string
     */
    const APC_CACHE_INFO_DTOS_KEY = "info";
    /**
     * APC memory user cache entry key in apc_info metadata structures:
     * Our dtos are stored in user cache.
     * 
     * @var string
     */
    const APC_USER_CACHE = "user";
    
    
    const MAX_APC_READ_RETRY = 3;
    
    const MAX_APC_WRITE_RETRY = 10;
    
    const MAX_LOCK_RETRY = 20;
    
    private static $memoryManagerFile;
    
    /**
     * Cleanup cache memory.
     * 
     * @throws SMTAPCMemoryException
     */
    private static function releaseMemory()
    {
        if ( self::acquireLock() )
        {
            try 
            {                
                SMTLogger::getInstance()->trace("Cache memory reinitialized!", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
                apc_clear_cache();
            	self::releaseLock();
            }
            catch (\Exception $e )
            {
                self::releaseLock();
                throw $e;
            }
        }
        else
        {
        	throw new SMTAPCMemoryException( SMTAPCMemoryException::ERROR_WRITING_APC );
        }        
    }    
    
    /**
     * Read Dto array from apc store.
     *
     * @param string $dto_key
     * 
     * @return SMTIMemorySupport[] $dtos
     * 
     * @throws SMTAPCMemoryException
     */
    static function readFromApcStore( $dto_key )
    {
    	$success = FALSE;
    	$readTrialCount = 0;
    	$dtos = array();
    	
    	if( apc_exists( $dto_key ) )
    	{
        	$dtos = apc_fetch( $dto_key, $success );
        	
        	while ( !$success && $readTrialCount < self::MAX_APC_READ_RETRY)
        	{
        		$readTrialCount++;
        		usleep(rand(1000, 10000));
        		$dtos = apc_fetch( $dto_key, $success );
        		SMTLogger::getInstance()->trace("Couldn't read from APC store for :".$dto_key, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	}
        
            if ( !$success )
        	{
        	    self::releaseMemory();
        		throw new SMTAPCMemoryException( SMTAPCMemoryException::ERROR_READING_APC );
        	}
    	}
    	return $dtos;
    }    
    
    /**
     * Dump all Dtos array from apc store.
     *
     *
     * @return SMTIMemorySupport[] $dtos
     *
     * @throws SMTAPCMemoryException
     */
    static function fetchAllFromApcStore()
    {
    	$success = FALSE;
    	$readTrialCount = 0;
    	$cache_user = array();
    	 
    	if ( PHP_MAJOR_VERSION >= 6 )
    	{
    		$cache_user = apcu_cache_info();
    		
    		while ( $cache_user === FALSE && $readTrialCount < self::MAX_APC_READ_RETRY)
    		{
    			$readTrialCount++;
    			usleep(rand(1000, 10000));
    			$cache_user = apcu_cache_info();
    			SMTLogger::getInstance()->trace("Couldn't read from APC store for fetching all dtos.", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		}
    	}
    	else
    	{
			$cache_user = apc_cache_info( self::APC_USER_CACHE );
			 
			while ( $cache_user === FALSE && $readTrialCount < self::MAX_APC_READ_RETRY)
			{
				$readTrialCount++;
				usleep(rand(1000, 10000));
				$cache_user = apc_cache_info( "user" );
				SMTLogger::getInstance()->trace("Couldn't read from APC store for fetching all dtos.", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
			}
    	}
    	
    	if ( $cache_user === FALSE )
    	{
    		self::releaseMemory();
    		throw new SMTAPCMemoryException( SMTAPCMemoryException::ERROR_READING_APC );
    	}
    	return $cache_user;
    }
    
    /**
     * Write Dto array to apc store.
     * 
     * @param string $dto_key
     * @param SMTIMemorySupport[] $dtos
     * 
     * @throws SMTAPCMemoryException
     */
    static function writeToApcStore( $dto_key, $dtos )
    {
        $success = FALSE;
        $writeTrialCount = 0;
        $writeId = rand(1, 10000);
//         SMTLogger::getInstance()->trace("Start writing to APC store for :".$dto_key.$writeId);

        self::touchAPCMemory();
        
        //WARNING with current APC version, writing can fail!!!
        while ( TRUE != ( $success = apc_store( $dto_key, $dtos ) ) && $writeTrialCount < self::MAX_APC_WRITE_RETRY)
        {
            $writeTrialCount++;
            usleep(rand(1000, 10000));
            if ( apc_exists( $dto_key ) )
            {
                SMTLogger::getInstance()->trace("Reset APC store for :".$dto_key." with value ".$success, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
                $success = apc_delete( $dto_key );
                self::touchAPCMemory();
                                
                $success = FALSE;
            }
            else
            {
                self::touchAPCMemory();
                
                SMTLogger::getInstance()->trace("Write fake value to APC store for :".$dto_key, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            }
            SMTLogger::getInstance()->trace("Couldn't write to APC store for :".$dto_key." and write Id:".$writeId, SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);       
        }
//         SMTLogger::getInstance()->trace("End writing to APC store for :".$dto_key.$writeId );
        if ( !$success )
        {
            throw new SMTAPCMemoryException( SMTAPCMemoryException::ERROR_WRITING_APC );
        }
    }    
    
    /**
     * WARNING: required to workaround the problem of memory not writable from time to time
     */
    private static function touchAPCMemory()
    {
        apc_store( "touch", NULL);
    }
    
    
    //TODO WARNING: All readings and writings to shared memory are synchronized 
    //TODO WARNING: Could be implemented with a lock by object Type
    private static function getMemoryManagerFile()
    {
    	if ( self::$memoryManagerFile == NULL )
    	{
    		$handle = fopen( __DIR__.self::MEMORY_MANAGER_LOCK_FILE, "w+");
    		
    		self::$memoryManagerFile = ( $handle != FALSE)? $handle : NULL;
    	}
    	return self::$memoryManagerFile;
    }
    
    //TODO WARNING: All readings and writings to shared memory are synchronized
    //TODO WARNING: Could be implemented with a lock by object Type        
    static function acquireLock()
    {
        $retries = 0;
        
        // keep trying to get a lock as long as possible (max 1.5s)
        do {
        	if ($retries > 0) 
        	{
        		usleep( 500 * ( $retries*$retries ) );
        	}
        	$retries += 1;
        } while ( ( !flock(self::getMemoryManagerFile(), LOCK_EX, $eWouldBlock) || $eWouldBlock ) && $retries <= self::MAX_LOCK_RETRY);   
        
        return ($retries < self::MAX_LOCK_RETRY);
    }
    
    //TODO WARNING: All readings and writings to shared memory are synchronized
    //TODO WARNING: Could be implemented with a lock by object Type    
    static function releaseLock()
    {
        return (isset(self::$memoryManagerFile) && self::$memoryManagerFile != NULL)? flock( self::$memoryManagerFile, LOCK_UN ) : TRUE;
    }    
}
?>