<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of VIAVI Solutions
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from VIAVI Solutions
// Copyright VIAVI Solutions 2019. All rights reserved.
// *********************************************************
namespace app\serviceshelper\otau;

use app\parser\SMTOtuSocket;
use app\serviceshelper\SMTServiceHelper;
use app\parser\SMTOtuApi;
use app\services\setup\SMTConfDetRotauDto;
use app\util\SMTInvalidValueException;
use app\services\setup\SMTRotauDto;

/**
 * Retrieve rotaus config and detection.
 *
 * @author Sylvain Desplat
 */
class SMTRotauManager extends SMTServiceHelper
{
	/**
	 *
	 * @var array of string
	 */
	private $switchDifferences = NULL;
	
	/**
	 * 
	 * @var array of int
	 */
	private $switchConfigIdsList = NULL;
		
	/**
	 * Retrieve switch difference, then the switch config and then the switch detected.
	 * If the switch difference has already been retrieved, don't do it again. Same for detection.
	 * 
	 * @param SMTConfDetRotauDto $rotaus
	 * @param string $forceDetection
	 */
	function retrieve(SMTConfDetRotauDto $rotaus, $forceDetection = TRUE)
	{
		//retrieve macro switch config:
		$this->retrieveConfiguredSwitch($this->getContext()->getOtuSocket(), $rotaus);
		
		//retrieve detection if a difference exists
		if ( $this->isDifference($rotaus, $forceDetection) )
		{
			$this->retrieveDetectedSwitch($this->getContext()->getOtuSocket(), $rotaus);
		}
		if ( $this->switchDifferences != NULL )
		{
			$index = 0;
			foreach ( $this->switchDifferences as $diff )
			{
				$rotaus->addSwitchDifference( $index++, strcasecmp( $diff, SMTOtuApi::RES_SAME) != 0);
			}
		}
	}
		
	/**
	 * 
	 * @param string $rotauIp
	 * @throws \InvalidArgumentException
	 * @throws SMTInvalidValueException
	 */
	function add( $rotauIp)
	{			
		if( strlen($rotauIp) > 0 )
		{
			$rotauIp= self::formatAddressIfIPV6($rotauIp);
			try 
			{
				$result = $this->getContext()->getOtuSocket()->sendReceive( sprintf(SMTOtuApi::CMD_verify_ROTAU, $rotauIp) );				
				$list_rotau_settings = explode(",", $result );
				
				//Ajout du ROTAU dans la detection de l'OTU maitre
				// - add rotau ip
				$cmd_switch_config_Add_Chaine_ROTAU_Formatted = sprintf(SMTOtuApi::CMD_switch_config_add_chaine_ROTAU, $list_rotau_settings[1],$list_rotau_settings[2], $rotauIp);
				$this->getContext()->getOtuSocket()->sendReceive($cmd_switch_config_Add_Chaine_ROTAU_Formatted, 0, 90);
			}
			catch(\Exception $e)
			{
				throw new \InvalidArgumentException( sprintf( MSG_BAD_ROTAU_IP, $rotauIp ) );
			}
		}
		else
		{
			throw new SMTInvalidValueException(SMTInvalidValueException::INVALID_VALUE, NULL, "rotauIp", $rotauIp);
		}
		
	}
	
	/**
	 * 
	 * @param int $macroSwitchId
	 * 
	 * @throws SMTInvalidValueException
	 */
	function remove( $macroSwitchId )
	{
		if( intval($macroSwitchId) > 0)
		{
			$rotau = SMTRotauSwitchConf::retrieve( $this->getContext()->getOtuSocket(), $macroSwitchId);
						
			$this->getContext()->getOtuSocket()->send( sprintf(SMTOtuApi::CMD_remove_otau, $macroSwitchId) );
			
			if( strlen($rotau->getRotauIp()) > 0 )
			{
				//remove ip address of the rotau in the OTAU
				$rotauIp= self::formatAddressIfIPV6($rotau->getRotauIp());
				$this->getContext()->getOtuSocket()->send(sprintf(SMTOtuApi::CMD_switch_detect_remove_chaine_ROTAU, $rotau->getRotauIp()));
				
				//sleep 500ms before requesting the config
				usleep(500000);				
			}
		}
		else
		{
			throw new SMTInvalidValueException(SMTInvalidValueException::INVALID_VALUE, NULL, "macroSwitchId", $macroSwitchId);
		}
	}
	
	/**
	 * WARNING TRAC 907: UPDATE ROTAU in configuration when its detection is different from calibration
     * if the detection is empty, we use the method remove which removes the rotau from the configuration 
     * and removes its ip declaration
     * 
	 * @param SMTRotauDto $rotau
	 * @throws \InvalidArgumentException
	 */
	function update(SMTRotauDto $rotau)
	{
		if( strlen($rotau->getRotauIp()) > 0 )
		{
			//remove ip address of the rotau in the OTAU
			$rotauIp= self::formatAddressIfIPV6($rotau->getRotauIp());
			if ( strcasecmp($rotau->getDevice(), "IPROTAU") == 0)
			{
				$device="IPR";
			}
			else if ( strcasecmp($rotau->getDevice(), "RTUIPROTAU") == 0)
			{
				$device="RTUIPR";
			}
			else
			{
				$device="IPR";
			}
			$this->getContext()->getOtuSocket()->sendReceive( sprintf(SMTOtuApi::CMD_update_rotau_in_config, $device, $rotau->getGlobalSwitchEncodedString(), $rotauIp), 0, 60 );
			
			//sleep 2s before requesting the config
			sleep(2);
		}
		else
		{
			throw new \InvalidArgumentException( sprintf( MSG_BAD_ROTAU_IP, $rotauIp ) );
		}
	}
	
	/**
	 * 
	 * @param SMTOtuSocket $socket
	 * @param SMTConfDetRotauDto $rotaus
	 */
	function retrieveDetectedSwitch(SMTOtuSocket $socket, SMTConfDetRotauDto $rotaus)
	{
		//search macro switches configured (on ne peut detecter que des rotau enregistres)
		if ( $this->switchConfigIdsList == NULL )
		{
			$this->switchConfigIdsList = explode(",", $socket->sendReceive( SMTOtuApi::CMD_switch_config_list ));
		}
		
		//process rotaus (macroswitch > 0)
		foreach ( $this->switchConfigIdsList as $id)
		{
			if ( intval($id) > 0 )
			{
				$rotau = SMTRotauSwitchDet::retrieve($socket, intval($id));
				if ( $rotau != NULL )
				{
					$rotaus->addDetected($rotau);
				}
			}
		}
	}
	
	/**
	 *
	 * @param SMTOtuSocket $socket
	 * @param SMTConfDetRotauDto $rotaus
	 *
	 */
	function retrieveConfiguredSwitch(SMTOtuSocket $socket, SMTConfDetRotauDto $rotaus)
	{
		//search macro switches configured
		if ( $this->switchConfigIdsList == NULL )
		{
			$this->switchConfigIdsList = explode(",", $socket->sendReceive( SMTOtuApi::CMD_switch_config_list ));
		}
		
		//process rotaus (macroswitch > 0)
		foreach ( $this->switchConfigIdsList as $id)
		{
			if ( intval($id) > 0 )
			{
				$rotau = SMTRotauSwitchConf::retrieve($socket, intval($id));
				if ( $rotau != NULL )
				{
					$rotaus->addConfigured($rotau);
				}					
			}
		}
	}
	
	/**
	 * Compare the current macro switch in detection and calibration
	 *
	 * @return Whether the macro switch is identical, different in detection and calibration or whether conf is missing or det is missing
	 */
	private function compareOTAUDetConfig( $forceDetection = TRUE )
	{
		//search macro switches configured (on ne peut detecter que des rotau enregistres)
		if ( $this->switchConfigIdsList == NULL )
		{
			$this->switchConfigIdsList = explode(",", $socket->sendReceive( SMTOtuApi::CMD_switch_config_list ));
		}
		
		foreach ( $this->switchConfigIdsList as $id)
		{
			if ( intval($id) > 0 )
			{
				if ( $forceDetection )
				{
					$this->switchDifferences[$id] = $this->getContext()->getOtuSocket()->sendReceive( sprintf( SMTOtuApi::CMD_compare_switches, intval($id)), SMTOtuSocket::DEFAULT_SLEEP, 90, TRUE );
				}
				else
				{
					$this->switchDifferences[$id] = $this->sendReceive( sprintf( SMTOtuApi::CMD_compare_switches_result, intval($id)) );
				}
			}
		}
	}
	
	/**
	 * Whether a difference between detection and config was found.
	 *
	 * @return boolean true if a difference between detection and config was found.
	 */
	function isDifference(SMTConfDetRotauDto $rotaus, $forceDetection = TRUE)
	{
		$hasDifference = FALSE;
		
		//detect differences if they have not already been retrieved
		if ( $this->switchDifferences == NULL )
		{
			$this->compareOTAUDetConfig($forceDetection);
		}
		if ($this->switchDifferences != NULL)
		{
			foreach ( $this->switchDifferences as $diff )
			{
				$hasDifference |= strcasecmp( $diff, SMTOtuApi::RES_SAME) != 0;
			}
		}
		
		return $hasDifference;
	} 
	
	/**
	 * Reset comparison and detection to force their retrieval.
	 *
	 */
	function reset()
	{
		$this->switchDifferences = NULL;
		$this->switchConfigIdsList= NULL;
	}
	
	/**
	 * If the address is an IPV6 address, convert it to the format [ipV6]:port
	 *
	 * @param $address the address to test
	 *
	 * @return the address unchanged or converted if it was an IPV6 address
	 */
	private static function formatAddressIfIPV6($address)
	{
// 		$IPv6 = strpos($address, ":");
		
		if ( filter_var($address, FILTER_VALIDATE_IP, FILTER_FLAG_IPV6) )
		{
			$address = '['.$address.']:1400';
		}
		return $address;
	}
}
?>