<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\alarm;

use app\database\SMTDatabaseException;

use app\util\SMTLogger;

/**
 * System alarm event Dto 
 * 
 * @author Sylvain Desplat
 * @SMTClassTableNameAnnotation(tablename='system_alarm_event', tablekey='id') 
 */
class SMTSystemAlarmEventDto extends SMTAlarmEventDto
{
    const FETCH_CURRENT_SYSTEM_ALARM_EVENT_OF_ALARM =
    "SELECT system_alarm_event.* FROM system_alarm_event,system_alarm WHERE system_alarm = :alarmId AND system_alarm_event.id=system_alarm.current_alarm_event_id";
    
    const FETCH_SYSTEM_ALARM_EVENTS_OF_ALARM =
    "SELECT * FROM system_alarm_event WHERE system_alarm_event.alarm_id=:alarmId";
    
    
    /**
     * Fetch the current system alarm event of the given alarm
     *
     * @param \SQLite3 $dbConnection
     * @param string $alarmId
     *
     * @return SMTSystemAlarmEventDto
     */
    public static function fetch( \SQLite3 $dbConnection, $alarmId )
    {
        $query = self::FETCH_CURRENT_SYSTEM_ALARM_EVENT_OF_ALARM;
    	$statement = $dbConnection->prepare( $query );
    	
    	if ( $statement === FALSE )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Prepare Fetch statement failed. Query: %s.", $query), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace( sprintf("Prepare Fetch statement Successfull. Query: %s.", $query), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    	}
    	
    	$bindParameters = ':alarmId=>'.$alarmId;
    	$statement->bindValue(':alarmId', $alarmId, SQLITE3_TEXT );
    	$resultSet = $statement->execute();
    	
    	if ( $resultSet === FALSE )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Fetch failed. Query: %s. Bind parameters: %s", $query, $bindParameters), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace( sprintf("Fetch successfull for Query: %s. Bind parameters: %s", $query, $bindParameters), SMTLogger::DEBUG);
    	}    	
    
    	return parent::getSingleInstanceFromResultSet($resultSet, get_class() );
    }
    
    /**
     * Fetch the list of system alarm events of an alarm
     *
     * @param \SQLite3 $dbConnection
     * @param string $alarmId
     *
     * @return array SMTSystemAlarmEventDto
     */
    public static function fetchAlarmEvents( \SQLite3 $dbConnection, $alarmId )
    {
    	$alarmEvents = array();
    	$query = self::FETCH_SYSTEM_ALARM_EVENTS_OF_ALARM;
    	$statement = $dbConnection->prepare( $query );
    	
    	if ( $statement === FALSE )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Prepare Fetch statement failed. Query: %s.", $query), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace( sprintf("Prepare Fetch statement Successfull. Query: %s.", $query), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    	}
    	
    	$bindParameters = ':alarmId=>'.$alarmId;
    	$statement->bindValue(':alarmId', $alarmId, SQLITE3_TEXT );
    	$result = $statement->execute();
    	
    	if ( $result === FALSE )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Fetch failed. Query: %s. Bind parameters: %s", $query, $bindParameters), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
    	}
    	else
    	{
    		SMTLogger::getInstance()->trace( sprintf("Fetch successfull for Query: %s. Bind parameters: %s", $query, $bindParameters), SMTLogger::DEBUG);
    	}    	    
    
    	return parent::getListFromResultSet($result, get_class() );
    }    
        
    /**
     * Serialize as Json data the user object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }
}
?>