<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\activity\otu;

use app\util\SMTRwDto;

use app\util\SMTUtil;

use app\services\activity\SMTActivityState;

use app\services\activity\SMTActivityType;

use app\sharedmemory\SMTMemoryManager;

use app\sharedmemory\SMTIMemorySupport;

/**
 * Otu activity dto class mapping Otu json activity notification.
 * 
 * @author Sylvain Desplat
 */
class SMTOtuActivityDto extends SMTRwDto  implements SMTIMemorySupport
{
    const OTU_LAST_ACTIVITY_ID = 0;
    /**
     * Last access to update the OTU activity
     *
     * @var float
     */
    private $lastAccess;
    
    /**
     * Activity type
     *  
     * @var string
     */
    protected $type = SMTActivityType::UNKNOWN;
    
    /**
     * Activity state
     * 
     * @var string
     */
    protected $state = SMTActivityState::UNKNOWN;
    
    /**
     * Test identifier
     * @var integer
     */
    protected $testId;
    
    /**
     * OTU timestamp in seconds and in current timezone
     * 
     * @var integer/float OTU timestamp in seconds and in current timezone
     */
    protected $date;
    
    /**
     * Alarm creation date as a string (not used just added to map info sent by OTU)
     *
     * @var string
     */
    protected $dateHuman;
    
    /**
     * Operation Id (not used just added to map info sent by OTU)
     *
     * @var string
     */
    protected $operationId;
    
    /**
     * Returns the activity type with OTU code.
     *
     * @return string
     */
    public function getType()
    {
    	return $this->type;
    }    

    /**
     * Returns the activity state with OTU code.
     *
     * @return string
     */
    public function getState()
    {
    	return $this->state;
    }
    
    /**
     * Decodes the activity type.
     *
     * @return string
     */
    public function getDecodedType()
    {
    	return SMTActivityType::decodeOtuCode( $this->type );
    }
    
    /**
     * Whether the activity is a measurement
     * 
     * @return boolean
     */
    public function isMeasurement()
    {
        return SMTActivityType::isMeasurementActivityOtuCode( $this->type );
    }
    
    /**
     * Whether the activity is running
     *
     * @return boolean
     */
    public function isNotRunning()
    {
    	return SMTActivityState::isNotRunningState( $this->state );
    }
    
    /**
     * Whether the activity is an autotest
     *
     * @return boolean
     */
    public function isAutotest()
    {
    	return SMTActivityType::isAutotestActivityOtuCode( $this->type );
    }
    
    /**
     * Decodes the activity state.
     *
     * @return string
     */
    public function getDecodedState()
    {
    	return SMTActivityState::decodeOtuCode( $this->state );
    }
    
    /**
     * Returns the test identifier or null.
     *
     * @return integer
     */
    public function getTestId()
    {
    	return $this->testId;
    }    
    
    /**
     * Whether activity test id is set
     * 
     * @return boolean Whether activity test id is set
     */
    public function isTestIdValid()
    {
        return isset($this->testId) && $this->testId !== NUll && $this->testId != 65535; //cTest_Aucun = cTest_Devalide = 0xFFFF
    }

    /**
     * OTU timestamp in seconds and in current timezone
     * 
     * @return integer/float OTU timestamp in seconds and in current timezone
     */
    public function getDate()
    {
    	return ( isset( $this->date) && SMTUtil::isInt( $this->date) )? $this->date : 0;
    }
    
    /**
     * OTU activity timestamp in UTC
     * @param \SQLite3 $dbConnection
     * 
     * @return integer/float otu activity timestamp in UTC
     */
    public function getTimestampUTC( \SQLite3 $dbConnection )
    {
        return SMTUtil::getTimestampUTC( $this->date, $dbConnection );
    }
    
    /**
     * Serialize as Json data the object.<br>
     *
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars($this);
    	$this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }
    
    /**
     * Create the Dto from a Json Dto in an array
     *
     * @param array $json_array json data in an array
     *
     * @return \app\services\activity\otu\SMTOtuActivityDto
     */
    static function getInstance( array &$json_array )
    {
    	$dto = self::forge( get_class(), $json_array, FALSE );
    	$dto->touchAccess();
    	return $dto;
    }    
    
    /**
     * Activity Identifier: return always 0 because we don't want to keep more than one OTU activity in cache
     * 
     * @return integer
     */
    public function getId()
    {
    	return self::OTU_LAST_ACTIVITY_ID;
    }
        
    /**
     * Save Dto in shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function save()
    {
    	SMTMemoryManager::saveDto( $this );
    }
    
    /**
     * Delete Dto from shared memory
     *
     * @throws SMTAPCMemoryException
     */
    public function delete()
    {
    	SMTMemoryManager::deleteDto( $this );
    }
    
    /**
     * Returns the dto class name.
     *
     * @return the dto class name.
     */
    public static function getClass()
    {
    	return __CLASS__;
    }
    
    /**
     * activity is obsolete after 120s
     *
     * @return FALSE
     */
    public function isObsolete()
    {
    	if ( !isset( $this->lastAccess ) )
	    {
	        $this->touchAccess();
	    }
    	return ( microtime(TRUE) - $this->getLastAccess() ) > 120;
    }    
    
    public function getLastAccess()
    {
    	return $this->lastAccess;
    }    

    public function touchAccess()
    {
    	$this->lastAccess = microtime(true);
    }   
}
?>