import dbus
import dbus.service
import threading
import signal
import os
from gi.repository import GLib
from jdsu.mts.upgrade import PrintException, get_hooks, run_hook, check_release, get_release_xml, show_diff, show_highlighted
from jdsu.mts.upgrade_tar import start_upgrade_tar
from configparser import ConfigParser
from queue import Queue



class SmartManager(dbus.service.Object):
    version = 2.0

    def __init__(self, daemon=False, loop=900000, debug=False):
        self.__daemon = daemon
        self.__workqueue = Queue()
        self.__proxy = None
        self.__last_state = "NOP"
        # default values for configuration
        self.__use_background_mode = False
        self.__auto_switch = True
        self.__test_update_part = False
        self.__parts = [5, 6]
        self.upgrade_way = [start_upgrade_tar]
        ifa = "text"
        if os.path.exists('/etc/releasemanager.conf'):
            config = ConfigParser()
            config.read('/etc/releasemanager.conf')
            if config.has_option('GENERAL', 'parts'):
                parts = config.get('GENERAL', 'parts')
                self.__parts = list(map(int, parts.strip().split(',')))
            if config.has_option('GENERAL', 'background_mode'):
                self.__use_background_mode = config.getboolean('GENERAL', 'background_mode')
            if config.has_option('GENERAL', 'auto_switch_part'):
                self.__auto_switch = config.getboolean('GENERAL', 'auto_switch_part')
            if config.has_option('GENERAL', 'test_update_part'):
                self.__test_update_part = config.getboolean('GENERAL', 'test_update_part')
            print("Config read : ")
            print(self.__parts)
            print(self.__use_background_mode)
            print(self.__auto_switch)
            print(self.__test_update_part)
        if self.__daemon:
            dbus.mainloop.glib.DBusGMainLoop(set_as_default=True)
            bus_name = dbus.service.BusName('org.jdsu.releasemanager', bus=dbus.SystemBus())
            dbus.service.Object.__init__(self, bus_name, "/org/jdsu/releasemanager")
            GLib.threads_init()
            self.__mainloop = GLib.MainLoop()
            with open("/proc/cmdline", "r") as f:
                bootargs = f.readline().split(' ')
                rootpart = [arg for arg in bootargs if arg.startswith("root=")]
                if len(rootpart) < 1:
                    print("Cannot determine root partition, fatal error, exiting")
                    exit(-1)
                rootpart = rootpart[0].split("=")[1]
                self.__flash_block_device = rootpart[:rootpart.rindex("p")]
                print("Root part is %s" % self.__flash_block_device)

        self.__release_monitoring = None
        self.__updated = False
        self.__timeout = loop
        if self.__daemon:
            self.__thread = threading.Thread(target=self.__worker)
            self.__thread.start()
            print("Daemon started\n")
        if not os.path.isdir("/var/lib/release-manager"):
            os.mkdir("/var/lib/release-manager")
        if os.path.isfile("/var/lib/release-manager/upgrade_status"):
            with open("/var/lib/release-manager/upgrade_status", "r") as f:
                if f.readline().strip() == "Success":
                    with open("/tmp/upgrade_success", "w") as t:
                        t.write("OK")
                else:
                    with open("/tmp/upgrade_failure", "w") as t:
                        t.write("FAIL")
            os.remove("/var/lib/release-manager/upgrade_status")

    def __worker(self):
        next_job = self.__workqueue.get()
        while type(next_job[0]) != str and next_job[0] != "EXIT":
            try:
                print("Worker: %s Start with param=%s" % (next_job[0].__name__, next_job[1]))
                next_job[0](**next_job[1])
                print("Worker: %s End" % next_job[0].__name__)
            except Exception as err:
                print("ERROR: %s" % str(err))
                PrintException()
            finally:
                self.__workqueue.task_done()
                next_job = self.__workqueue.get()

    def __del__(self):
        self.stop_daemon()
        if self.__daemon:
            print("Daemon Stoped")

    @dbus.service.signal(dbus_interface='org.jdsu.releasemanager', signature='s')
    def new_release(self, release):
        print("New release: %s" % str(release))

    def __check_new_release(self):
        if self.__release_monitoring is not None:
            oldxml = current_release_xml()
            (newxml, valid) = get_release_xml(self.__release_monitoring, self.__proxy)

            old_name = oldxml.xpath("@name")[0]
            new_name = newxml.xpath("@name")[0]
            print("Check release %s ?= %s" % (old_name, new_name))

            if old_name != new_name:
                self.new_release(new_name)

    def __timer(self):
        print("Timer")
        if self.__release_monitoring is not None:
            self.__workqueue.put((self.__check_new_release, {}))
            GLib.timeout_add(self.__timeout, self.__timer)

    def __set_release_monitoring(self, url):
        if len(url) > 0:
            if check_release(url, self.__proxy):
                print("Enable release monitoring on %s" % url)
                self.__release_monitoring = url
                self.__check_new_release()
                GLib.timeout_add(self.__timeout, self.__timer)
        else:
            print("Disable release monitoring")
            self.__release_monitoring = None

    @dbus.service.method(dbus_interface='org.jdsu.releasemanager', in_signature='s')
    def set_release_monitoring(self, url):
        print("CALL release-manager set_release_monitoring %s" % url)
        self.__set_release_monitoring(url)

    def __run_task(self, task, param={}):
        if self.__daemon:
            print("Enqueue %s with param=%s" % (task.__name__, param))
            self.__workqueue.put((task, param))
        else:
            print("%s Start with param=%s" % (task.__name__, param))
            task(**param)
            print("%s End" % task.__name__)

    def start_daemon(self):
        signal.signal(signal.SIGTERM, self.stop_daemon)
        try:
            self.__mainloop.run()
        except:
            self.stop_daemon()

    def stop_daemon(self):
        if self.__daemon:
            self.__workqueue.put(("EXIT"))
            self.__thread.join()
            self.__mainloop.quit()

    @dbus.service.signal(dbus_interface='org.jdsu.releasemanager', signature='s')
    def log(self, line):
        print(line)

    @dbus.service.signal(dbus_interface='org.jdsu.releasemanager', signature='s')
    def set_action(self, line):
        print(line)

    @dbus.service.signal(dbus_interface='org.jdsu.releasemanager', signature='sissi')
    def expose(self, topic, percent, subkey, subtopic, subpercent):
        print("%s %i %s %s %i" % (topic, percent, subkey, subtopic, subpercent))

    # End of SMART API
    ###

    def __upgrade_failed(self):
        if os.path.exists("/tmp/Disable_OnOff_Button"):
            os.remove("/tmp/Disable_OnOff_Button")
        with open("/var/lib/release-manager/upgrade_status", "w") as f:
            f.write("Failed")
        self.start_upgrades_terminate("FAIL")

    def __start_upgrades(self, url):
        self.__last_state = "UPGRADING"

        (xmlrelease, valid) = get_release_xml(url, self.__proxy)
        if not valid:
            self.__upgrade_failed()
            return

        with open("/tmp/Disable_OnOff_Button", "w") as f:
            f.write("LOCK")
        with open("/var/lib/release-manager/upgrade_status", "w") as f:
            f.write("Started")
        with open("/var/volatile/upgrade_in_progress", "w") as f:
            f.write("lock")

        hook = get_hooks(xmlrelease)

        if not run_hook(hook, 'pre_process'):
            self.__upgrade_failed()
            return

        # in background mode, do not change runlevel because apps are still working
        if not self.__use_background_mode:
            os.system("init 4; sleep 5")

        error = True
        options = {
            'proxy': self.__proxy,
            'daemon': self.__daemon,
            'flash_block_device': self.__flash_block_device,
            'auto_switch': self.__auto_switch,
            'test_update_part': self.__test_update_part,
            'parts': self.__parts
        }

        for upgrade_method in self.upgrade_way:
            print("Try upgrade with %s"%upgrade_method.__name__)
            if upgrade_method(xmlrelease, hook, options, self):
                error = False
                break
            else:
                print("\t%s failed, Try next method"%upgrade_method.__name__)

        if not run_hook(hook, 'post_process'):
            error = True

        os.remove("/tmp/Disable_OnOff_Button")
        os.remove("/var/volatile/upgrade_in_progress")
        if error:
            self.start_upgrades_terminate("FAIL")
            with open("/var/lib/release-manager/upgrade_status", "w") as f:
                f.write("Failed")
        else:
            self.start_upgrades_terminate("PASS")
            with open("/var/lib/release-manager/upgrade_status", "w") as f:
                f.write("Success")

    @dbus.service.signal(dbus_interface='org.jdsu.releasemanager', signature='s')
    def start_upgrades_terminate(self, state):
        self.__last_state = state
        print("Upgrade process terminate: %s" % state)

    @dbus.service.method('org.jdsu.releasemanager', in_signature='s')
    def start_upgrade(self, url):
        print("CALL release-manager start_upgrade %s" % url)
        self.__run_task(self.__start_upgrades, {'url': url})

    @dbus.service.signal(dbus_interface='org.jdsu.releasemanager', signature='s')
    def set_release_terminate(self, name):
        print("Release changed to %s" % name)

    @dbus.service.method('org.jdsu.releasemanager', out_signature='b')
    def check_new_release(self):
        print("CALL release-manager check_new_release")
        return self.__check_new_release()

    @dbus.service.method('org.jdsu.releasemanager', out_signature='s')
    def get_release_info(self):
        print("CALL release-manager get_release_info")
        releasexml = current_release_xml()
        if releasexml is None:
            return "<release />"
        return releasexml.serialize()

    @dbus.service.method('org.jdsu.releasemanager', out_signature='s')
    def get_last_upgrade_state(self):
        print("CALL release-manager get_last_upgrade_state")
        return self.__last_state

    @dbus.service.signal(dbus_interface='org.jdsu.releasemanager', signature='s')
    def upgrade_by_tar_terminate(self, name):
        print("Release changed to %s" % name)

    @dbus.service.method('org.jdsu.releasemanager')
    def upgrade_by_tar(self):
        print("CALL release-manager upgrade_by_tar")
        self.__run_task(self.__upgrade_by_tar)

    @dbus.service.method(dbus_interface='org.jdsu.releasemanager', in_signature='s', out_signature='s')
    def show_highlighted(self, url):
        print("CALL release-manager show_highlighted %s" % url)
        return show_highlighted(url, self.__proxy)

    @dbus.service.method(dbus_interface='org.jdsu.releasemanager', in_signature='s', out_signature='s')
    def show_diff(self, url):
        print("CALL release-manager show_diff %s" % url)
        return show_diff(url, self.__proxy)

    @dbus.service.method(dbus_interface='org.jdsu.releasemanager', in_signature='s', out_signature='b')
    def check_release(self, url):
        print("CALL release-manager check_release %s" % url)
        return check_release(url, self.__proxy)

    @dbus.service.method('org.jdsu.releasemanager')
    def exit(self):
        print("CALL release-manager exit")
        self.stop_daemon()

    def __set_proxy(self, proxy):
        print("SET PROXY: %s" % proxy)
        if len(proxy) > 1:
            self.__proxy = proxy
        else:
            self.__proxy = None

    @dbus.service.method(dbus_interface='org.jdsu.releasemanager', in_signature='s')
    def set_proxy(self, proxy):
        print("CALL release-manager set_proxy %s" % proxy)
        self.__set_proxy(proxy)

    @dbus.service.method(dbus_interface='org.jdsu.releasemanager', in_signature='s')
    def set_background_mode(self, state):
        in_background = bool(state)
        print("Set background mode: %s" % str(in_background))
        self.__use_background_mode = in_background
