<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\alarm;

use app\serviceshelper\monitoring\SMTGpsRoute;

use app\services\alarm\SMTAlarmEventDto;

use app\services\alarm\SMTAlarmDto;

use app\util\SMTResultSetDto;

use app\database\SMTIExecuteInTransaction;

use app\database\SMTSmartOtuDB;

use app\util\SMTIOException;

use app\serviceshelper\otdr\SMTOtdrTraceFileUtil;

use app\services\alarm\otu\SMTOtuOpticalAlarmDto;

use app\services\alarm\otu\SMTOtuSystemAlarmDto;

use app\util\SMTLogger;

use app\services\alarm\SMTAttenuationOpticalAlarmDto;

use app\services\alarm\SMTSystemAlarmDto;

use app\services\alarm\SMTOpticalAlarmDto;

use app\services\alarm\otu\SMTOtuAlarmDto;

use app\parser\SMTOtuApi;
use app\serviceshelper\SMTServiceHelper;
use app\services\alarm\otu\SMTOtuOpticalAlarmPeakDto;
use app\services\alarm\SMTPeakOpticalAlarmDto;

/**
 * Helper class to retrieve and clear alarms.
 * 
 * @author Sylvain Desplat
 */
class SMTAlarm extends SMTServiceHelper
{    
    const INFO = "INFO";
    const CLEAR = "CLEAR";
    const WARNING = "WARNING";
    const MINOR = "MINOR";
    const MAJOR = "MAJOR";
    const CRITICAL = "CRITICAL";
    
    const UNIT = "UNIT";
    const COM = "COM";
    const FIBER = "FIBER";
    const PERF = "PERF";

    const RETRY_ACKNOWLEDGE_SLEEP_DURATION = 10;
    const MAX_RETRY_ACKNOWLEDGE = 10; 
        
    const LOCK_ALARM_FILE = "/../../../tmp/alarm.lock";
    
    /**
     * Alarm file exclusive lock
     * 
     * @var string
     */
    private static $alarmLockFile = NULL;
    
    
    /**
     * Create or update a system or optical alarm from the given otu alarm
     * Critical section to process alarms one by one
     *
     * @param SMTOtuAlarmDto $otuAlarm
     * @return array of SMTAlarmDto The created or update alarm. Returns NULL is the otu alarm was not processed.
     */
    public function processAlarm( SMTOtuAlarmDto $otuAlarm )
    {
        $alarms = array();
        $peakProcessed = FALSE;
        $updateTraces = TRUE;
        $updatePeaksAlams = FALSE;
        
        if ( self::acquireAlarmLock() )
        {        
            try 
            {
                //retrieve database connection
                $dbConnection = $this->getContext()->getDatabase();
                
                //Process optical alarm whether it is confirmed or not
            	if ( $otuAlarm instanceof SMTOtuOpticalAlarmDto )
            	{
            		//build alarm discriminators
            		$alarmDiscriminators = SMTAlarmFactory::buildOpticalAlarmDiscriminators( $otuAlarm, $dbConnection );
            
            		if ( count ( $alarmDiscriminators ) > 0 )
            		{        		
            			self::cleanCurrentAlarmBeforeProcesing( $otuAlarm, $dbConnection);
            			
                		foreach ( $alarmDiscriminators as $alarmDiscriminator )
                		{            
                		    //Attenuation alarm
                			if ( ($otuAlarm->getAttenuation() != null) && SMTAlarmFactory::isAttenuationAlarm( $alarmDiscriminator ) )
                			{                			     
                				//process optical attenuation alarm
                				SMTLogger::getInstance()->trace( sprintf("Optical alarm discriminator: %s. Alarm attenuation specific problem: %s. Alarm attenuation otu severity: %s.", $alarmDiscriminator, $otuAlarm->getAttenuation()->getSpecificProblemCode(), $otuAlarm->getOtuSeverity()), SMTLogger::INFO );
                				
                			    try
                			    {
                			        $attenuationAlarm = self::processAttenuationAlarm($otuAlarm, $alarmDiscriminator, $dbConnection);
                                    if ( $attenuationAlarm != NULL )
                                    {
                                        array_push( $alarms, $attenuationAlarm );
                                        if ( $attenuationAlarm->getCurrentAlarmEvent() != NULL )
                                        {
                                            //update google map url if needed
                                            $route = new SMTGpsRoute();
                                            $route->setContext($this->getContext());
                                            $route->updateGoogleMapsUrl($attenuationAlarm->getCurrentAlarmEvent()->isGpsValid());
                                        }
                                        $updateTraces = FALSE;
                                        $updatePeaksAlams = TRUE;
                                    }
                                }
                                catch( \Exception $e)
                                {
                                    //delete OTDR trace
                                	SMTOtdrTraceFileUtil::removeOTUAlarmTraceFromSmartOtu( $otuAlarm->getTestId(), $alarmDiscriminator );                        
                                	throw $e;
                                }
                			}
                			else if ( SMTAlarmFactory::isPeakAlarm( $alarmDiscriminator ) )
                			{
               					if ( !$peakProcessed )
               					{
                					// process all peak alarms once
                					$peakProcessed = TRUE;
                					                					
                					foreach ($otuAlarm->getPeaks() as $peak)
                					{                					
                						$alarmPeakDiscriminator = sprintf( PEAK_ALARM_DISCRIMINATOR_FORMAT, $otuAlarm->getTestId(), $peak->getReferenceTopDistance());
                						
                						//process optical peak alarm
                						SMTLogger::getInstance()->trace( sprintf("getReferenceTopDistance %0.2f, Optical peak alarm discriminator: %s. Alarm peak otu severity: %s.", $peak->getReferenceTopDistance(), $alarmPeakDiscriminator, $otuAlarm->getOtuSeverity()), SMTLogger::INFO );
                						
                						try
                						{
                							$peakAlarm= self::processPeakAlarm($otuAlarm, $peak, $alarmPeakDiscriminator, $dbConnection, $updateTraces);
                							if ( $peakAlarm!= NULL )
                							{
                								$updateTraces = FALSE;
                								$updatePeaksAlams = TRUE;
                							}
                						}
                						catch( \Exception $e)
                						{
                							//delete OTDR trace
                							$alarmPeakByTestDiscriminator = SMTAlarmFactory::getDiscriminatorForOtdrTrace( $otuAlarm->getTestId() );
                							SMTOtdrTraceFileUtil::removeOTUAlarmTraceFromSmartOtu( $otuAlarm->getTestId(), $alarmPeakByTestDiscriminator);
                							throw $e;
                						}                						
                					}
                					
               					}
               				}
                				
                			//process ORL:
                			//NOT implemented in SmartOTU
                			//process fiber length extension:
                			//NOT implemented in SmartOTU   			
                			else
                			{
                			    //Not an attenuation alarm
                			    SMTLogger::getInstance()->trace("Optical alarm is not an attenuation. Alarm with discriminator: ".$alarmDiscriminator." is not processed.", SMTLogger::INFO, __FILE__,__METHOD__,__LINE__ );
                			}
                		}
                		
                		//force fetch of peaks alarm on that test because some can have changed and other can be unchanged: all alarms on that test must be sent to client not only changes
                		if ($updatePeaksAlams)
                		{
                			$peakAlarms = $this->fetchPeakAlarmFromTest( $otuAlarm->getTestId(), TRUE );
                			if ( ($peakAlarms != NULL) && count($peakAlarms) > 0)
                			{
                				foreach ( $peakAlarms as $peakAlarm)
                				{
                					if ( $peakAlarm!=NULL )
                					{
                						array_push( $alarms, $peakAlarm );
                					}
                				}
                			}
                		}
                		
            		}
            		else
            		{
            		    SMTLogger::getInstance()->trace( "Alarm discriminator not processed by SmartOtu: unknown alarm; ".$otuAlarm->getJsonData(), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__ );
            		}    		    
            	}
            	else if ( $otuAlarm instanceof SMTOtuSystemAlarmDto )
            	{
            	    //build alarm discriminators
            	    $alarmDiscriminator = SMTAlarmFactory::buildSystemAlarmDiscriminator( $otuAlarm );
            	    $systemAlarm = self::processSystemAlarm($otuAlarm, $alarmDiscriminator, $dbConnection);
            	    if ( $systemAlarm != NULL )
            	    {
                        array_push( $alarms, $systemAlarm );
            	    }   	        	    
            	}
            	else
            	{
            	    SMTLogger::getInstance()->trace( "Alarm not processed by SmartOtu: unknown alarm; ".$otuAlarm->getJsonData(), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__ );
            	}
            	self::releaseAlarmLock();
            }
            catch( \Exception $e)
            {
                self::releaseAlarmLock();
                throw $e;
            }
        }
        else
        {
        	SMTLogger::getInstance()->trace("Couldn't acquire lock to process alarm: ", SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTIOException( SMTIOException::COULD_NOT_ACQUIRE_LOCK );
        }
    	
    	return $alarms;
    }

    /**
     * If the OTU alarm (attenuation and peaks) has a severity transition, clean current attenuation and peaks alarms if their severity is cleared.
     * 
     * @param SMTOtuOpticalAlarmDto $otuAlarm
     * @param SMTSmartOtuDB $dbConnection
     */
    public static function cleanCurrentAlarmBeforeProcesing(SMTOtuOpticalAlarmDto $otuAlarm, SMTSmartOtuDB $dbConnection)
    {
    	$clean = FALSE;
    	
    	//search if optical alarm exist on test
    	$attenuationAlarm = SMTAttenuationOpticalAlarmDto::fetchFromTest($dbConnection, $otuAlarm->getTestId());
    	$alarmDiscriminator = sprintf( ATTENUATION_ALARM_DISCRIMINATOR_FORMAT, $otuAlarm->getTestId() );
    	//if the alarm has a severity transition, clean current alarm if it is cleared;
    	if ( self::filterAttenuationOpticalAlarm( $otuAlarm, $alarmDiscriminator, $attenuationAlarm ) == FALSE )
    	{
    		$clean = TRUE;
    	}
    	
    	if ( !$clean && count($otuAlarm->getPeaks()) > 0 )
    	{
	    	//if the alarm has a severity transition, clean current alarm if it is cleared;
	    	foreach ($otuAlarm->getPeaks() as $peak)
	    	{  
	    		if ($peak != NULL)
	    		{
		    		$alarmPeakDiscriminator = sprintf( PEAK_ALARM_DISCRIMINATOR_FORMAT, $otuAlarm->getTestId(), $peak->getReferenceTopDistance());
		    		//search if optical alarm with same discriminator exists
		    		$peakAlarm = SMTPeakOpticalAlarmDto::fetch($dbConnection, $alarmPeakDiscriminator);
		    		
		    		if ( self::filterPeakAlarm( $peak, $alarmPeakDiscriminator, $peakAlarm) == FALSE )
			    	{
			    		$clean = TRUE;
			    		break;
			    	}
	    		}
	    	}
    	}
    	
    	if ( $clean )
    	{
    		self::cleanupAlarms($otuAlarm, $attenuationAlarm, $dbConnection);
    	}
    }
    
    /**
     * Clean current attenuation and peaks alarms if their severity is cleared on the given test.
     * 
     * @param SMTOtuOpticalAlarmDto $otuAlarm
     * @param SMTAttenuationOpticalAlarmDto $alarm
     * @param SMTSmartOtuDB $dbConnection
     */
    private static function cleanupAlarms(SMTOtuOpticalAlarmDto $otuAlarm, SMTAttenuationOpticalAlarmDto $alarm = NULL, SMTSmartOtuDB $dbConnection)
    {
    	$traceRemoved = FALSE;
    	
    	// if alarm with same discriminator already exists and is cleared and if we have a change of severity on the new alarm,
    	// delete the cleared alarm with its events before creating a new one.
    	if ( $alarm != NULL && $alarm->isCleared() )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Delete cleared alarm on test %s with same discriminator: %s.", $alarm->getTestId(), $alarm->getDiscriminator()) );

    		//delete OTDR trace
    		SMTOtdrTraceFileUtil::removeOTUAlarmTraceFromSmartOtu( $alarm->getTestId(), $alarm->getDiscriminator() );
    		$traceRemoved = TRUE;
    		
    		//delete alarm
    		$alarm->delete( $dbConnection );
    		$alarm = NULL;
    	}
    	
    	//search all peak alarms on test
    	$peaksAlarm = SMTPeakOpticalAlarmDto::fetchFromTest( $dbConnection, $otuAlarm->getTestId());
    	// if alarm on same test already exists and is cleared and if we have a change of severity on the new alarm,
    	// delete the cleared alarms with their events before creating a new one.
    	if ( ($peaksAlarm!= NULL) && count($peaksAlarm) > 0 )
    	{
    		$alarmDiscriminator = SMTAlarmFactory::getDiscriminatorForOtdrTrace( $otuAlarm->getTestId() );    		
    		foreach( $peaksAlarm as $peakAlarm)
    		{
    			if (($peakAlarm!= NULL) && $peakAlarm->isCleared())
    			{
    				SMTLogger::getInstance()->trace( sprintf("Delete cleared peak alarm on test %s with same discriminator: %s.", $peakAlarm->getTestId(), $peakAlarm->getDiscriminator()) );
    				
    				if (!$traceRemoved)
    				{
    					//delete OTDR trace
    					SMTOtdrTraceFileUtil::removeOTUAlarmTraceFromSmartOtu( $peakAlarm->getTestId(), $alarmDiscriminator);
    					$traceRemoved = TRUE;
    				}
    				//delete alarm
    				$peakAlarm->delete( $dbConnection );
    				$peakAlarm= NULL;
    			}
    		}
    	}
    }
    
    /**
     * Create or update attenuation alarm from the given otu alarm and alarm discriminator
     *
     * @param SMTOtuOpticalAlarmDto $otuAlarm
     * @param string $alarmDiscriminator
     * @param \SQLite3 $dbConnection
     * 
     * @return SMTAttenuationOpticalAlarmDto The created or update alarm. Returns NULL if the otu alarm was not processed.
     */
    private static function processAttenuationAlarm( SMTOtuOpticalAlarmDto $otuAlarm, $alarmDiscriminator, SMTSmartOtuDB $dbConnection )    
    {
        $alarm = NULL;
        
        //attenuation processing
        SMTLogger::getInstance()->trace( sprintf("Attenuation processing for alarm discriminator: %s.", $alarmDiscriminator ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
        	
        //search if optical alarm with same discriminator exists
        $alarm = SMTAttenuationOpticalAlarmDto::fetch($dbConnection, $alarmDiscriminator);
        	
        if ( $alarm != NULL )
        {
        	SMTLogger::getInstance()->trace( sprintf("Attenuation alarm with discriminator: %s already exists.", $alarmDiscriminator ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
        }
        
        //if the alarm has a severity transition, create or update the alarm;
        if ( self::filterAttenuationOpticalAlarm( $otuAlarm, $alarmDiscriminator, $alarm ) == FALSE )
        {                     
			//search again if optical alarm with same discriminator exists
			$alarm = SMTAttenuationOpticalAlarmDto::fetch($dbConnection, $alarmDiscriminator);
        	
        	//retrieve OTDR traces if they are available (not a test on demand on a link in alarm)
        	//copy the OTDR trace for a transition of the alarm (new alarm, alarm severity change... and back to normal (clear of the alarm) )
        	if ( ($otuAlarm->getLocalisationFileName() != NULL && count($otuAlarm->getLocalisationFileName()) > 0 )|| 
        			(($otuAlarm->getDetectionFileName() != NULL ) && count($otuAlarm->getDetectionFileName()) > 0 ) )
        	{
        	    //first, remove the old trace
        	    SMTOtdrTraceFileUtil::removeOTUAlarmTraceFromSmartOtu( $otuAlarm->getTestId(), $alarmDiscriminator );
        	    //copy the new files
        	    SMTOtdrTraceFileUtil::copyOtuAlarmTraceInSmartOtu( $otuAlarm->getLocalisationFileName(), $otuAlarm->getDetectionFileName(), $otuAlarm->getShortAcqFileName() ,$otuAlarm->getTestId(), $otuAlarm->getOtuMonitoringTestPort(), $alarmDiscriminator );
        	}
        	
        	//start attenuation processing
        	SMTLogger::getInstance()->trace( sprintf("Start attenuation processing for alarm discriminator: %s on test %s.", $alarmDiscriminator, $otuAlarm->getTestId() ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
        	 
        	$newAlarm = ($alarm == NULL );
        	
        	//create the alarm without its current event 
        	if ( $newAlarm )
        	{ 
        	    $alarm = SMTAlarmFactory::createAttenuationOpticalAlarmDto( $dbConnection, $otuAlarm, $alarmDiscriminator );
        	}
        	
        	//create the event and update the alarm with its current event if needed
        	$alarmEvent = SMTAlarmFactory::updateAttenuationOpticalAlarmDto( $dbConnection, $otuAlarm, $alarm, $alarmDiscriminator, $newAlarm );
        	 
        	//save the alarm and the event
        	if ( $alarm != NULL )
        	{
        	    //if alarm event is the current alarm event: saving the alarm will save the alarm event
        	    if ( $alarm->getCurrentAlarmEvent() === $alarmEvent )
        	    {
        	        $alarm->save( $dbConnection );
        	    }
        	    //event is not the current alarm event and thus must be saved apart
        	    else
        	    {
        	        $dbConnection->runInTransaction( new SMTSaveAlarmAndEvent( $alarm, $alarmEvent ) );
        	    }
        	}
        	 
        	SMTLogger::getInstance()->trace( sprintf("End attenuation processing for alarm discriminator: %s on test %s.", $alarmDiscriminator, $otuAlarm->getTestId() ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
        }
        
        return $alarm;
    }
        
    /**
     * Create or update peak alarm from the given otu alarm and alarm discriminator
     *
     * @param SMTOtuOpticalAlarmDto $otuAlarm
     * @param SMTOtuOpticalAlarmPeakDto $peakAlarmDto
     * @param string $alarmDiscriminator
     * @param \SQLite3 $dbConnection
     * @param boolean $updateTrace whether the traces must be update 
     *
     * @return SMTAttenuationOpticalAlarmDto The created or update alarm. Returns NULL if the otu alarm was not processed.
     */
    private static function processPeakAlarm( SMTOtuOpticalAlarmDto $otuAlarm, SMTOtuOpticalAlarmPeakDto $peakAlarmDto, $alarmDiscriminator, SMTSmartOtuDB $dbConnection, $updateTrace = FALSE )
    {
    	$alarm = NULL;
    	
    	//attenuation processing
    	SMTLogger::getInstance()->trace( sprintf("Peak processing for alarm discriminator: %s.", $alarmDiscriminator ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    	
    	//search if optical alarm with same discriminator exists
    	$alarm = SMTPeakOpticalAlarmDto::fetch($dbConnection, $alarmDiscriminator);
    	
    	if ( $alarm != NULL )
    	{
    		SMTLogger::getInstance()->trace( sprintf("Peak alarm with discriminator: %s already exists.", $alarmDiscriminator ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    	}
    	
    	//search all peak alarms on test
    	$peaksAlarm = SMTPeakOpticalAlarmDto::fetchFromTest( $dbConnection, $otuAlarm->getTestId());
    	
    	//if the alarm has a severity transition, create or update the alarm;
    	if ( self::filterPeakAlarm( $peakAlarmDto, $alarmDiscriminator, $alarm ) == FALSE )
    	{    		
    		$alarmPeakByTestDiscriminator = SMTAlarmFactory::getDiscriminatorForOtdrTrace( $otuAlarm->getTestId() );
    		
    		//search again if optical alarm with same discriminator exists
    		$alarm = SMTPeakOpticalAlarmDto::fetch($dbConnection, $alarmDiscriminator);
    		
    		//retrieve OTDR traces if they are available (not a test on demand on a link in alarm)
    		//copy the OTDR trace for a transition of the alarm (new alarm, alarm severity change... and back to normal (clear of the alarm) )
    		if ( $updateTrace && 
   				( ($otuAlarm->getLocalisationFileName() != NULL && count($otuAlarm->getLocalisationFileName()) > 0 )||
				  (($otuAlarm->getDetectionFileName() != NULL ) && count($otuAlarm->getDetectionFileName()) > 0 ) ) ) 
    		{
    			//first, remove the old trace
    			SMTOtdrTraceFileUtil::removeOTUAlarmTraceFromSmartOtu( $otuAlarm->getTestId(), $alarmPeakByTestDiscriminator);
    			//copy the new files
    			SMTOtdrTraceFileUtil::copyOtuAlarmTraceInSmartOtu( $otuAlarm->getLocalisationFileName(), $otuAlarm->getDetectionFileName(), $otuAlarm->getShortAcqFileName() ,$otuAlarm->getTestId(), $otuAlarm->getOtuMonitoringTestPort(), $alarmPeakByTestDiscriminator);
    		}
    		
    		//start peak processing
    		SMTLogger::getInstance()->trace( sprintf("Start peak processing for alarm discriminator: %s on test %s.", $alarmDiscriminator, $otuAlarm->getTestId() ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    		
    		$newAlarm = ($alarm == NULL );
    		
    		//create the alarm without its current event
    		if ( $newAlarm )
    		{
    			$alarm = SMTAlarmFactory::createPeakOpticalAlarmDto( $dbConnection, $otuAlarm, $peakAlarmDto, $alarmDiscriminator );
    		}
    		
    		//create the event and update the alarm with its current event if needed
    		$alarmEvent = SMTAlarmFactory::updatePeakOpticalAlarmDto( $dbConnection, $otuAlarm, $peakAlarmDto, $alarm, $alarmDiscriminator, $newAlarm );
    		
    		//save the alarm and the event
    		if ( $alarm != NULL )
    		{
    			//if alarm event is the current alarm event: saving the alarm will save the alarm event
    			if ( $alarm->getCurrentAlarmEvent() === $alarmEvent )
    			{
    				$alarm->save( $dbConnection );
    			}
    			//event is not the current alarm event and thus must be saved apart
    			else
    			{
    				$dbConnection->runInTransaction( new SMTSaveAlarmAndEvent( $alarm, $alarmEvent ) );
    			}
    		}
    		
    		SMTLogger::getInstance()->trace( sprintf("End peak processing for alarm discriminator: %s on test %s.", $alarmDiscriminator, $otuAlarm->getTestId() ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    	}
    	
    	return $alarm;
    }
    
    /**
     * Create or update system alarm from the given otu alarm and alarm discriminator
     *
     * @param SMTOtuSystemAlarmDto $otuAlarm
     * @param string $alarmDiscriminator
     * @param \SQLite3 $dbConnection
     * 
     * @return SMTOtuSystemAlarmDto The created or update alarm. Returns NULL is the otu alarm was not processed.
     */
    private static function processSystemAlarm( SMTOtuSystemAlarmDto $otuAlarm, $alarmDiscriminator, SMTSmartOtuDB $dbConnection )
    {
        $alarm = NULL;
        
        SMTLogger::getInstance()->trace( sprintf("System alarm discriminator: %s. Alarm specific problem: %s. Alarm otu severity: %s.", $alarmDiscriminator, $otuAlarm->getSpecificProblemCode(), $otuAlarm->getOtuSeverity()), SMTLogger::INFO );
        	
        //search if system alarm with same discriminator exists
        $alarm = SMTSystemAlarmDto::fetch($dbConnection, $alarmDiscriminator);
        	
        if ( $alarm != NULL )
        {
        	SMTLogger::getInstance()->trace( sprintf("System alarm with discriminator: %s already exists.", $alarmDiscriminator ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
        }
        
        //In the case of a NEW system alarm with no severity, don't create the alarm
        if ( self::filterSystemAlarm( $otuAlarm, $alarmDiscriminator, $alarm ) == FALSE )
        {
        	//if alarm already exists and is cleared and if we have a change of severity, delete the alarm with its events before creating a new one.
        	if ( $alarm != NULL && $alarm->isCleared() )
        	{
        		SMTLogger::getInstance()->trace( sprintf("Delete cleared alarm with same discriminator: %s.", $alarmDiscriminator ) );
        		$alarm->delete( $dbConnection );
        		$alarm = NULL;
        	}
        		
        	//process system alarm
        	SMTLogger::getInstance()->trace( sprintf("Start system alarm processing for alarm discriminator: %s.", $alarmDiscriminator ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
        	 
        	$alarm = SMTAlarmFactory::createOrUpdateSystemAlarmDto( $dbConnection, $otuAlarm, $alarm, $alarmDiscriminator );
        	 
        	//save alarm
        	if ( $alarm != NULL )
        	{
        	    //alarm event is the current alarm event: saving the alarm will save the alarm event
        	    $alarm->save($dbConnection);
        	}
        	 
        	SMTLogger::getInstance()->trace( sprintf("End system alarm processing for alarm discriminator: %s.", $alarmDiscriminator ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
        }
        
        return $alarm;
    }
    
    /**
     * Whether the given alarm must be filtered or not.
     * In the case of a NEW attenuation alarm with no severity, don't create the alarm.
     * WARNING: If the last alarm on test is cleared, never update the events of that cleared alarm:
     * we can only delete a cleared alarm to create a new one, never update it.
     *
     * @param SMTOtuOpticalAlarmDto $otuAlarm
     * @param $alarmDiscriminator optical alarm discriminator
     * @param SMTAttenuationOpticalAlarmDto $alarm
     *
     * @return TRUE if the alarm must be filtered => not processed
     */
    private static function filterAttenuationOpticalAlarm( SMTOtuOpticalAlarmDto $otuAlarm, $alarmDiscriminator, SMTAttenuationOpticalAlarmDto $alarm = NULL  )
    {
        $filter = FALSE;
        
        // In the case of a NEW attenuation alarm with no severity, don't create the alarm.
        // WARNING: If the last alarm on test is cleared, never update the events of that cleared alarm:
        // - we can only delete a cleared alarm to create a new one, never update it.
        if ( ( $alarm == NULL || ( $alarm != NULL && $alarm->isCleared() ) ) &&
        		!SMTOtuAlarmSeverity::hasSeverityTransitionAndNotCleared( $otuAlarm->getAttenuation()->getOtuSeverity() ) )
        {
        	SMTLogger::getInstance()->trace( sprintf("WARNING: Attenuation Alarm with discriminator: %s has no severity: %s. No alarm created.", $alarmDiscriminator, $otuAlarm->getAttenuation()->getOtuSeverity() ),SMTLogger::INFO, __FILE__,__METHOD__,__LINE__ );
        	$filter = TRUE;
        }
        return $filter;
    }
    
    /**
     * Whether the given alarm must be filtered or not.
     * In the case of a NEW attenuation alarm with no severity, don't create the alarm.
     * WARNING: If the last alarm on test is cleared, never update the events of that cleared alarm:
     * we can only delete a cleared alarm to create a new one, never update it.
     *
     * @param SMTOtuOpticalAlarmDto $otuAlarm
     * @param $alarmDiscriminator optical alarm discriminator
     * @param SMTAlarmDto $alarm
     *
     * @return TRUE if the alarm must be filtered => not processed
     */
    private static function filterPeakAlarm( SMTOtuOpticalAlarmPeakDto $peakAlarmDto, $alarmDiscriminator, SMTAlarmDto $alarm = NULL  )
    {
    	$filter = FALSE;
    	
    	// In the case of a NEW peak alarm with no severity, don't create the alarm.
    	// WARNING: If the last alarm on test is cleared, never update the events of that cleared alarm:
    	// - we can only delete a cleared alarm to create a new one, never update it.
    	if ( ( $alarm == NULL || ( $alarm != NULL && $alarm->isCleared() ) ) &&
    			!SMTOtuAlarmSeverity::hasSeverityTransitionAndNotCleared( $peakAlarmDto->getOtuSeverity() ) )
    	{
    		SMTLogger::getInstance()->trace( sprintf("WARNING: Alarm with discriminator: %s has no severity: %s. No alarm created.", $alarmDiscriminator, $peakAlarmDto->getOtuSeverity() ),SMTLogger::INFO, __FILE__,__METHOD__,__LINE__ );
    		$filter = TRUE;
    	}
    	return $filter;
    }
    
    /**
     * Whether the given alarm must be filtered or not.
     * In the case of a NEW system alarm with no severity, don't create the alarm
     * 
     * @param SMTOtuAlarmDto $otuAlarm
     * @param $alarmDiscriminator system alarm discriminator
     * @param SMTSystemAlarmDto $alarm
     * 
     * @return TRUE if the alarm must be filtered => not processed 
     */
    private static function filterSystemAlarm( SMTOtuSystemAlarmDto $otuAlarm, $alarmDiscriminator, SMTSystemAlarmDto $alarm = NULL  )
    {
        $filter = FALSE;
        
        //new FO alarm with software problem and no severity must be ignored, don't create the alarm
        if ( $alarm == NULL 
                && !SMTOtuAlarmSeverity::hasSeverityTransition( $otuAlarm->getOtuSeverity() ) 
                && SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_SOFTWARE_PROBLEM_CODE == $otuAlarm->getSpecificProblemCode() 
                && SMTSystemAlarmAdditionalInfoCode::OTU_SYST_ALARM_ADDITIONAL_INFO_FO_COMMUNICATION == $otuAlarm->getAdditionalInfo() )
        {
        	SMTLogger::getInstance()->trace( sprintf("WARNING: New FO communication system Alarm with discriminator: %s has no severity. No alarm created.", $alarmDiscriminator ) );
        	$filter = TRUE;
        }
        //new Reboot alarm with no severity must be ignored, don't create the alarm
        else if ( $alarm == NULL
        		&& !SMTOtuAlarmSeverity::hasSeverityTransition( $otuAlarm->getOtuSeverity() )
        		&& SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_REBOOT_CODE == $otuAlarm->getSpecificProblemCode() )
        {
        	SMTLogger::getInstance()->trace( sprintf("WARNING: New Reboot system Alarm with discriminator: %s has no severity. No alarm created.", $alarmDiscriminator ) );
        	$filter = TRUE;
        }        
        //new heartbeat alarm with no severity must be ignored, don't create the alarm
        else if ( $alarm == NULL
        		&& !SMTOtuAlarmSeverity::hasSeverityTransition( $otuAlarm->getOtuSeverity() )
        		&& SMTAlarmSpecificProblemCode::OTU_SYST_ALARM_HEARTBEAT_CODE == $otuAlarm->getSpecificProblemCode() )
        {
        	SMTLogger::getInstance()->trace( sprintf("WARNING: New heartbeat system Alarm with discriminator: %s has no severity. No alarm created.", $alarmDiscriminator ) );
        	$filter = TRUE;
        }        
        //In the case of a NEW system alarm with no severity, don't create the alarm
        else if ( ( $alarm == NULL || ( $alarm != NULL && $alarm->isCleared() ) ) &&
        		!SMTOtuAlarmSeverity::hasSeverityTransitionAndNotCleared( $otuAlarm->getOtuSeverity() ) )
        {
        	SMTLogger::getInstance()->trace( sprintf("WARNING: New System Alarm with discriminator: %s has no severity. No alarm created.", $alarmDiscriminator ) );
        	$filter = TRUE;
        }
        
        return $filter;
    }
    
    
    /**
     * Acknowledge OTU alarm on OTU; that alarm was sent by OTU and processed successfully on SmartOTU 
     * 
     * @param SMTOtuAlarmDto $otuAlarm
     */
    public function acknowledgeAlarm( SMTOtuAlarmDto $otuAlarm, $retryCount = 0 )
    {
        SMTLogger::getInstance()->trace( sprintf("Try Acknowledgment of alarm %s result on OTU..", $otuAlarm->getAcknowledgeString() ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
        
        //acknowledgment of alarm is critical. If parser is unavailable, retry up to MAX_RETRY_ACKNOWLEDGE
        try 
        {
            $result = $this->sendReceive( sprintf( SMTOtuApi::CMD_acknowledge_alarm, $otuAlarm->getAcknowledgeString() ) );
            SMTLogger::getInstance()->trace( sprintf("Acknowledgment result of alarm %s on OTU: %s.", $otuAlarm->getAcknowledgeString(), $result ) );
        }
        catch( \Exception $e )
        {
            if ( $retryCount < self::MAX_RETRY_ACKNOWLEDGE )
            {
                SMTLogger::getInstance()->trace( sprintf("Acknowledgment of alarm %s on OTU FAILED. Retrying...", $otuAlarm->getAcknowledgeString() ) );
                sleep( self::RETRY_ACKNOWLEDGE_SLEEP_DURATION );
                $this->acknowledgeAlarm( $otuAlarm, $retryCount++ );
            }
            else
            {
                throw $e;
            }
        }                
    }
           
    /**
     * Fetch the attenuation optical alarms (limited to attenuation alarm)
     *
     * @throws SMTAlarmException
     * @return \app\services\alarm\SMTOpticalAlarmDto array
     */
    function fetchOpticalAlarms( )
    {
        //retrieve database connection
        $dbConnection = $this->getContext()->getDatabase();
        
        SMTLogger::getInstance()->trace( "Fetch all attenuation optical alarms.",SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
        
        //search if optical alarm with same discriminator exists
        return SMTAttenuationOpticalAlarmDto::fetchAll( $dbConnection );
    }
    
    /**
     * Fetch the peak alarms (limited to peak alarm)
     *
     * @throws SMTAlarmException
     * @return \app\services\alarm\SMTOpticalAlarmDto array
     */
    function fetchPeakAlarms($fetchEvent = FALSE)
    {
    	//retrieve database connection
    	$dbConnection = $this->getContext()->getDatabase();
    	
    	SMTLogger::getInstance()->trace( "Fetch all peak optical alarms.",SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    	
    	return SMTPeakOpticalAlarmDto::fetchAll( $dbConnection, $fetchEvent);
    }
    
    /**
     * Fetch the optical alarm (limited to attenuation alarm)
     * 
     * @param int $id The alarm identifier
     * 
     * @throws SMTAlarmException
     * @return \app\services\alarm\SMTOpticalAlarmDto
     */
    function fetchOpticalAlarmDetail( $id )
    {
    	//retrieve database connection
    	$dbConnection = $this->getContext()->getDatabase();
    
    	SMTLogger::getInstance()->trace( "Fetch optical alarms of id ".$id,SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    
    	//search optical alarm
    	return SMTAttenuationOpticalAlarmDto::joinFetch( $dbConnection, $id );
    }
    
    /**
     * Delete optical alarm on SmartOTU database and remove alarm traces from SmartOTU alarm traces directory
     * WARNING: SHOULD ONLY BE CALLED WHEN DELETING A TEST: Delete optical alarm of the given test from SmartOTU database and from SmartOTU alarm traces directory.
     *   
     * @param int $testId
     */
    function deleteOpticalAlarmOnSmartOtu( $testId )
    {
    	$traceRemoved = FALSE;
    	
    	//retrieve database connection
    	$dbConnection = $this->getContext()->getDatabase();
    	
        $alarm = $this->fetchOpticalAlarmFromTest( $testId );
        
        // if alarm with same discriminator already exists and is cleared and if we have a change of severity on the new alarm,
        // delete the cleared alarm with its events before creating a new one.
        if ( $alarm != NULL )
        {            
        	SMTLogger::getInstance()->trace( sprintf("Delete alarm on test %s with discriminator: %s.", $testId, $alarm->getDiscriminator() ) );
        	//delete OTDR trace
        	SMTOtdrTraceFileUtil::removeOTUAlarmTraceFromSmartOtu( $testId, $alarm->getDiscriminator() );
        	$traceRemoved = TRUE;
        	
        	//delete alarm
        	$alarm->delete( $dbConnection );
        	$alarm = NULL;
        }
        
        $peakAlarms = $this->fetchPeakAlarmFromTest( $testId);
        if ( ($peakAlarms != NULL) && count($peakAlarms) > 0)
        {
        	foreach ( $peakAlarms as $peakAlarm)
        	{
        		if ( $peakAlarm != NULL )
        		{
        			//delete alarm
        			$peakAlarm->delete( $dbConnection );
        		}
        	}
        	
        	if (!$traceRemoved)
        	{
        		$alarmPeakByTestDiscriminator = SMTAlarmFactory::getDiscriminatorForOtdrTrace( $testId);
        		//delete OTDR trace
        		SMTOtdrTraceFileUtil::removeOTUAlarmTraceFromSmartOtu( $testId, $alarmPeakByTestDiscriminator );
        		$traceRemoved = TRUE;
        	}
        	
        }
    }
    
    /**
     * Delete optical on OTU (no notification is sent).     
     * @param number $testId
     * @param string $type INJ, ATT Atténuation and Injection alarm are processed identically in OTU
     */
    function deleteOpticalAlarmOnOtu( $testId, $type = SMTAlarmSpecificProblemDiscriminatorCode::ATTENUATION_ALARM )
    {
    	try
    	{
    		SMTLogger::getInstance()->trace( sprintf( "Delete optical alarms of type %s on test %s ", $type, $testId ), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
    
    		$this->send( SMTOtuApi::deleteMonitoringTestAlarmsCommand($type, $testId) );
    	}
    	catch(\Exception $e)
    	{
    		$this->getContext()->getLogger()->traceException($e);
    		throw new SMTAlarmException( SMTAlarmException::ERROR_DELETE_OPTICAL_ALARM_FAILED, sprintf( MSG_ERROR_DELETE_OPTICAL_ALARM_ON_TEST_FAILED, $testId ) );
    	}
    }
    
    /**
     * Delete optical on OTU (no notification is sent).
     * @param number $testId
     */
    function deleteAllPeakAlarmOnOtu( $testId)
    {
    	try
    	{
    		$type = SMTAlarmSpecificProblemDiscriminatorCode::PEAK_ALARM;
    		SMTLogger::getInstance()->trace( sprintf( "Delete optical alarms of type %s on test %s ", $type, $testId ), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);
    		
    		$this->send( SMTOtuApi::deleteMonitoringTestAlarmsCommand($type, $testId , -2) ); //all peaks on test
    	}
    	catch(\Exception $e)
    	{
    		$this->getContext()->getLogger()->traceException($e);
    		throw new SMTAlarmException( SMTAlarmException::ERROR_DELETE_OPTICAL_ALARM_FAILED, sprintf( MSG_ERROR_DELETE_OPTICAL_ALARM_ON_TEST_FAILED, $testId ) );
    	}
    }
    
    /**
     * Fetch the optical alarm (limited to attenuation alarm)
     *
     * @param int $testId The test identifier
     *
     * @throws SMTAlarmException
     * @return \app\services\alarm\SMTOpticalAlarmDto
     */
    function fetchOpticalAlarmFromTest( $testId, $fetchEvent = FALSE)
    {
    	//retrieve database connection
    	$dbConnection = $this->getContext()->getDatabase();
    
    	SMTLogger::getInstance()->trace( "Fetch optical alarms from testId ".$testId,SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    
    	//search optical alarm
    	return SMTAttenuationOpticalAlarmDto::fetchFromTest( $dbConnection, $testId, $fetchEvent);
    }

    /**
     * Fetch the peak alarms (limited to peak alarm)
     *
     * @param int $testId The test identifier
     *
     * @throws SMTAlarmException
     * @return array of \app\services\alarm\SMTOpticalAlarmDto
     */
    function fetchPeakAlarmFromTest( $testId, $fetchEvent = FALSE)
    {
    	//retrieve database connection
    	$dbConnection = $this->getContext()->getDatabase();
    	
    	SMTLogger::getInstance()->trace( "Fetch peak alarms from testId ".$testId,SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    	
    	//search optical alarm
    	return SMTPeakOpticalAlarmDto::fetchFromTest( $dbConnection, $testId, $fetchEvent);
    }
    
    /**
     * Fetch the system alarms
     *
     * @throws SMTAlarmException
     * @return \app\services\alarm\SMTSystemAlarmDto
     */
    function fetchSystemAlarms( )
    {
    	//retrieve database connection
    	$dbConnection = $this->getContext()->getDatabase();
    
    	SMTLogger::getInstance()->trace( "Fetch all system alarms.",SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    
    	//search if optical alarm with same discriminator exists
    	return SMTSystemAlarmDto::fetchAll( $dbConnection );
    }    
    
    /**
     * Fetch the active system alarms
     *
     * @throws SMTAlarmException
     * @return \app\services\alarm\SMTSystemAlarmDto
     */
    function fetchActiveSystemAlarms( )
    {
    	//retrieve database connection
    	$dbConnection = $this->getContext()->getDatabase();
    
    	SMTLogger::getInstance()->trace( "Fetch active system alarms.",SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    
    	//fetch active system alarms
    	return SMTSystemAlarmDto::fetchActiveAlarms( $dbConnection );
    }
    
    private static function getLockAlarmFile()
    {
    	if ( self::$alarmLockFile == NULL )
    	{
    		$handle = fopen( __DIR__.self::LOCK_ALARM_FILE, "w+");
    		 
    		self::$alarmLockFile = ( $handle != FALSE)? $handle : NULL;
    	}
    	return self::$alarmLockFile;
    }
    
    /**
     * Acquire alarm exclusive lock
     * 
     * @return boolean
     */
    private static function acquireAlarmLock()
    {
    	$retries = 0;
    	$max_retries = 50;
    
    	//SMTLogger::getInstance()->trace( "acquireAlarmLock" );
    
    	// keep trying to get a lock as long as possible ( max 43s)
    	do
    	{
    		if ($retries > 0)
    		{
    			usleep( 1000 * ( $retries*$retries ) );// Must be long because optical alarm processing lasts about 1s: sleep 100ms->1s
    		}
    		$retries += 1;
    	} while ( ( !flock(self::getLockAlarmFile(), LOCK_EX, $eWouldBlock) || $eWouldBlock ) && $retries <= $max_retries);
    
    	return ($retries < $max_retries);
    }
    
    /**
     * Releases alarm exclusive lock
     * 
     * @return boolean
     */
    private static function releaseAlarmLock()
    {
    	//SMTLogger::getInstance()->trace( "releaseAlarmLock" );
    	return isset(self::$alarmLockFile) && self::$alarmLockFile != NULL? flock( self::$alarmLockFile, LOCK_UN ) : TRUE;
    }    
    
    /**
     * Clear all alarms on OTU (used to resync alarms)
     */
    public function clearAllAlarms($deleteOtuAlarms = TRUE)
    {
        try 
        {
        	SMTLogger::getInstance()->trace( "Delete all alarms!",SMTLogger::PINFO);
            $dbConnection = $this->getContext()->getDatabase();
            
            //remove all alarms on OTU
            if ( $deleteOtuAlarms )
            {
            	$this->send( SMTOtuApi::clearAllAlarmsCommand() );
            }
            
            //remove all attenuation alarms form SmartOTU database.
            //remove all system alarms from SmartOTU database.
            SMTAlarmDto::deleteAll( $dbConnection );

            //remove all optical alarms traces
            SMTOtdrTraceFileUtil::removeAllOTUAlarmTraceFromSmartOtu();
        }
        catch( \Exception $e)
        {
            $this->getContext()->traceException($e);
            throw $e;
        }                
    }
    
    /**
     * Clear system alarm
     * 
     * @param $id
     */
    public function clearSystemAlarm( $id )
    {
    	try
    	{
    		$dbConnection = $this->getContext()->getDatabase();
        		
    		$alarm = SMTSystemAlarmDto::fetchFromId( $dbConnection, $id );
    		
    		//remove system alarm on OTU
    		$this->send( SMTOtuApi::clearSystemAlarmCommand( $alarm->getSpecificProblemId(), $alarm->getOtuAdditionalInfoId() ) );

    		//remove system alarm from SmartOTU database.
    		$alarm->delete($dbConnection);
    	}
    	catch( \Exception $e)
    	{
    		$this->getContext()->traceException($e);
    		throw $e;
    	}
    }    
}

/**
 * Save alarm and event in a transaction (event is not necessary the current alarm event and thus must be saved apart)
 * 
 * @author Sylvain Desplat
 */
class SMTSaveAlarmAndEvent implements SMTIExecuteInTransaction
{
    /**
     * 
     * @var SMTAlarmDto
     */
    var $alarm;
    /**
     * 
     * @var SMTAlarmEventDto
     */
    var $alarmEvent;
    
    /**
     * 
     * @param SMTAlarmDto $alarm
     * @param SMTAlarmEventDto $alarmEvent
     */
    function __construct( SMTAlarmDto $alarm, SMTAlarmEventDto $alarmEvent )
    {
    	$this->alarm = $alarm;
    	$this->alarmEvent = $alarmEvent;
    }
    
    /**
     * Save alarm and event in a transaction (event is not necessary the current alarm event and thus must be saved apart)
     */
    public function run( SMTSmartOtuDB $dbConnection )
    {
        $this->alarm->save($dbConnection);
        if ( $this->alarmEvent != NULL )
        {
            $this->alarmEvent->save($dbConnection);
        }
    }
    
    /**
     * @return SMTResultSetDto
     */
    public function getResultSetDto()
    {
        return $this->alarm;
    }
}
?>