<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\otdr;


use app\util\SMTException;

use app\error\SMTRestException;

use app\settings\SMTSmartOtuSettings;

use app\serviceshelper\alarm\SMTAlarmException;

use app\serviceshelper\alarm\SMTAlarm;

use app\util\SMTLogger;

use app\serviceshelper\otdr\SMTOtdrTraceFileUtil;

use app\events\SMTEventFactory;

use app\view\SMTOtdrTraceFileDownloadPage;

use app\serviceshelper\otdr\SMTOtdrTraceType;

use app\serviceshelper\monitoring\SMTLinkTest;

use app\util\SMTIOException;

use app\serviceshelper\monitoring\cache\SMTOperationManager;

use app\http\SMTSessionDirectoryUtil;

use app\serviceshelper\otdr\SMTMeasurement;

use app\events\operations\SMTOperationEventDto;

use app\view\SMTOtdrTracePage;

use app\util\SMTUtil;
use app\view\SMTTestRestApiPage;
use app\services\SMTIService;
use app\services\SMTService;
use app\services\security\SMTAccessControl;
use app\parser\SMTOtuApi;
use app\serviceshelper\monitoring\SMTLinkTestException;
use app\serviceshelper\alarm\SMTAlarmFactory;
use app\serviceshelper\otdr\SMTBaseOtdrTraceFileUtil;
use app\serviceshelper\otdr\SMTTraceFileType;
use app\serviceshelper\monitoring\SMTTestMode;

/**
 * Service to setup and list link-tests.
 * For smartOTU: link = test
 *
 * @author Sylvain Desplat
 */
class SMTOtdrAcqService extends SMTService implements SMTIService
{      
    /**
     *
     * @url PUT /measure
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function startMeasure( $request_data )
    {
    	try
    	{
    		$acquisitionParametersDto = SMTStartOtdrMeasureOnDemandParametersDto::getInstance( $request_data );
    
    		$measureOnDemandHelper = new SMTMeasurement();
    		$measureOnDemandHelper->setContext( $this->getContext() );
    		$measurementStatusDto = $measureOnDemandHelper->startMeasure( $acquisitionParametersDto );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $measurementStatusDto->getJsonData();
    }
    
    /**
     * @url POST /measureParameters/{portNumber}
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchMeasureOnDemandParameters( $portNumber, $request_data )
    {
        try
        {
            $acquisitionParametersDto = NULL;
            //default acquisition parameters sent by client:
            if ( $request_data != null )
            {
                $acquisitionParametersDto = SMTStartOtdrMeasureOnDemandParametersDto::getInstance( $request_data );
            }
            $measureOnDemandHelper = new SMTMeasurement();
            $measureOnDemandHelper->setContext( $this->getContext() );
            $measureOnDemandParameters= $measureOnDemandHelper->fetchMeasureOnDemandParameters( NULL, $portNumber, $acquisitionParametersDto);
        }
        catch ( \Exception $e )
        {
            $this->handleException($e);
        }
        return $measureOnDemandParameters->getJsonData();        
    }
    
    /**
     * @url POST /measureParameters/{portNumber}/{linkId}
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchLinkMeasureOnDemandParameters( $portNumber, $linkId, $request_data )
    {
    	try
    	{
    		$acquisitionParametersDto = NULL;
    		//default acquisition parameters sent by client:
    		if ( $request_data != null )
    		{
    			$acquisitionParametersDto = SMTStartOtdrMeasureOnDemandParametersDto::getInstance( $request_data );
    		}
    		$measureOnDemandHelper = new SMTMeasurement();
    		$measureOnDemandHelper->setContext( $this->getContext() );
    		$measureOnDemandParameters= $measureOnDemandHelper->fetchMeasureOnDemandParameters( $linkId, $portNumber, $acquisitionParametersDto);
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $measureOnDemandParameters->getJsonData();
    }
    
    
    /**
     *WARNING: POLLING MUST BE DONE AS POST TO GUARANTEE THAT THE HTTP REQUEST WILL BE RECEIVED BY THE SERVER
     *
     * @url POST /measure/{operationId}/status
     *
     * Access granted to everybody (used by asynchronous scripts).
     * Control is made on operation identifier which is generated from the server.
     */
    function getMeasurementStatus( $operationId )
    {
        $measureStatusDto = SMTEventFactory::createMeasureOnDemandEvent(NULL, $operationId, SMTOperationEventDto::NO_PROGRAMMED );
    
    	SMTLogger::getInstance()->trace( sprintf("getMeasurementStatus from PHP REST API for undeclared link and operation id: %s !",  $operationId ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    	
    	try
    	{
    		if ( SMTOperationManager::hasOperation( $operationId ) )
    		{
    			$measureOnDemandHelper = new SMTMeasurement();
    			$measureOnDemandHelper->setContext( $this->getContext() );
    			$measureStatusDto = $measureOnDemandHelper->getMeasurementStatus( $operationId );
    		}
    	}
    
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $measureStatusDto->getJsonData();
    }    
    
    /**
     *
     * @url PUT /measure/{operationId}/stop
     *
     * Access granted to everybody (used by asynchronous scripts).
     * Control is made on operation identifier which is generated from the server.
     */
    function stopMeasurement( $operationId )
    {
    	$measureStatusDto = SMTEventFactory::createMeasureOnDemandEvent(NULL, $operationId, SMTOperationEventDto::NO_PROGRAMMED );
    	try
    	{
    		if ( SMTOperationManager::hasOperation( $operationId ) )
    		{
    			$measureOnDemandHelper = new SMTMeasurement();
    			$measureOnDemandHelper->setContext( $this->getContext() );
    			$measureOnDemandHelper->stopMeasure( $operationId );
    		}
    	}
    
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $measureStatusDto->getJsonData();
    }    
    
    /**
     * Retrieve OTDR acquisition for the given trace name and the given trace type
     *
     * @url GET /downloadtracefile
     * @param $attachmentFileName
     * @param $traceStorageType
     * @status 200
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function downloadOtdrTraceFile( $attachmentFileName, $traceStorageType )
    {
    	$traceName = $attachmentFileName;
    	$otdrTraceType = $traceStorageType;
        $newFilePath = '';
        
        try
        {                    
            if ( SMTOtdrTraceType::SMARTOTU_ACQUISITION_TRACE == $otdrTraceType )
            {
                $newFilePath = SMTSessionDirectoryUtil::getOTDRTraceFileNameFromSession( $traceName );
            }                        
            else if ( SMTOtdrTraceType::LAST_TEST_ACQUISITION_TRACE == $otdrTraceType )
            {
                $newFilePath = SMTSessionDirectoryUtil::getOTDRTraceFileNameFromSession( $traceName );
            }
            else if ( SMTOtdrTraceType::OTU_REFERENCE_TRACE == $otdrTraceType )
            {
                $newFilePath = SMTOtdrTraceFileUtil::OTU_MEASURE_REF_DET_DIR.$traceName;
            }
            else if ( (SMTOtdrTraceType::OTU_ALARM_TRACE == $otdrTraceType) || (SMTOtdrTraceType::OTU_SHORTACQ_ALARM_TRACE == $otdrTraceType) )
            {
                $newFilePath = SMTOtdrTraceFileUtil::CUSTOM_SMARTOTU_ALARM_DIR.$traceName;
            }
            
            SMTLogger::getInstance()->trace( sprintf("Retrieve otdr trace file %s ",  $newFilePath ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
            
            if ( file_exists($newFilePath) )
            {
            	$page = new SMTOtdrTraceFileDownloadPage( $newFilePath, $otdrTraceType );
    	        $this->getContext()->getHttpResponse()->send( $page );
            }
            else
            {
            	$e = new SMTIOException( SMTIOException::FILE_NOT_FOUND );
            	$this->handleException($e);
            }
        }
        catch ( \Exception $e )
        {
        	$this->handleException($e);
        }        
    }
    
    /**
     * Retrieve OTDR acquisition for the given operation Id
     *
     * @url POST /trace/{operationId}
     * @status 200
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getAcquisitionTrace( $operationId )
    {
    	$SOM = 0;
    	$EOM = 0;
    	$newFilePath = NULL;
        try 
        {
            if ( SMTOperationManager::hasOperation( $operationId ) )
            {
                $measureCache = SMTOperationManager::getMeasureInfo( $operationId );
                if ( $measureCache != NULL )
                {
                	if ( SMTOperationEventDto::isResultAvailable( $measureCache->getStatus() ) );
                	{
                		$traceFilePath = $measureCache->getTraceDirectory()."/".$measureCache->getTraceName();   
                		$traceFilePath = SMTBaseOtdrTraceFileUtil::replaceExtension($traceFilePath, SMTBaseOtdrTraceFileUtil::OTDR_FILE_SOR_EXTENSION);
                		
                	    SMTLogger::getInstance()->trace( sprintf("Acquisition trace %s ",  $traceFilePath ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
                	    
                	    $portNumber = $measureCache->getPortId();
                	    $SOM = $measureCache->getSOM();
                	    $EOM = $measureCache->getEOM();
                	}
                }            
                //remove operation from cache
                SMTOperationManager::deleteMeasureInfo($operationId);
            }
        
            if ( !isset($traceFilePath) )
            {
                throw new SMTIOException( SMTIOException::FILE_NOT_FOUND, "OTDR trace not found for operation id: ".$operationId );
            }
            
        	//backup OTDR file to the smartOTU directory to be used for monitoring test addition
            if ( file_exists( $traceFilePath ) )
            {
        		$newFilePath = SMTSessionDirectoryUtil::backupOTDRTraceToSessionDirectory( $traceFilePath, $portNumber, $operationId );
        		SMTLogger::getInstance()->trace( sprintf("Acquisition backup trace in SmartOtu %s ",  $newFilePath ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
            }
        	        	
        	$smartAcqTraceExtension = SMTBaseOtdrTraceFileUtil::getSmartAcqFileExtension($traceFilePath);
        	// if smart acq
        	$smartAcq = SMTBaseOtdrTraceFileUtil::isSmartAcqFileExtension($smartAcqTraceExtension);
        	if ( $smartAcq )
        	{
        		$smartAcqTraceFilePath = SMTBaseOtdrTraceFileUtil::replaceExtension($traceFilePath, $smartAcqTraceExtension);
        		SMTLogger::getInstance()->trace( sprintf("smartAcqTraceFilePath Acquisition backup trace in SmartOtu %s ",  $smartAcqTraceFilePath), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        		//backup smart acq file to the smartOTU directory to be used for monitoring test addition
        		$shortAcqNewFilePath = SMTSessionDirectoryUtil::backupOTDRTraceToSessionDirectory( $smartAcqTraceFilePath, $portNumber, $operationId, TRUE );
        		SMTLogger::getInstance()->trace( sprintf("SmartAcq Acquisition backup trace in SmartOtu %s ",  $shortAcqNewFilePath), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        		$page = new SMTOtdrTracePage( $shortAcqNewFilePath, SMTOtdrTraceType::SMARTOTU_ACQUISITION_TRACE, SMTTraceFileType::getFileType($shortAcqNewFilePath), $smartAcq, $SOM, $EOM );
        	}
        	else
        	{
        		$page = new SMTOtdrTracePage( $newFilePath, SMTOtdrTraceType::SMARTOTU_ACQUISITION_TRACE, SMTTraceFileType::getFileType($newFilePath), $smartAcq, $SOM, $EOM );
        	}
        	$this->getContext()->getHttpResponse()->send( $page );
    	}
    	catch ( \Exception $e )
    	{
    	    $this->handleException($e);
    	}    	
    }    
    
    /**
     * Retrieve OTDR monitoring reference trace by its testId
     * Warning: if a sliding reference exists, return it
     *          if the reference is requested to be displayed with a last test acquisition: display short and long acq ( smartAcq = TRUE ) only if the last test acquisition is a localisation or if it's a tapping test 
     * //param testId The test with the monitoring reference trace
     *
     * @url POST /referencetrace/{testId}
     * @status 200
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getMonitoringReferenceTrace( $testId,$request_data)
    {
    	$slidingReferenceNotFound= TRUE;
    	$hasAcquisitionLoc = FALSE;
    	$tapping = FALSE;
    	$SOM = 0;
    	$EOM = 0;
    	
        try 
        {
        	$slidingReference = ($request_data != NULL && $request_data['sliding'] != NULL)? filter_var( $request_data['sliding'], FILTER_VALIDATE_BOOLEAN ) : FALSE;
        	$lastAcquisition = ($request_data != NULL && $request_data['lastAcquisition'] != NULL)? filter_var( $request_data['lastAcquisition'], FILTER_VALIDATE_BOOLEAN ) : FALSE;
        	
            //retrieve OTDR trace name
            SMTLinkTest::checkTestId($testId);
            	
            $linkHelper = new SMTLinkTest();
            $linkHelper->setContext( $this->getContext() );
            $linkTestDetailDto = $linkHelper->retrieveLinkTestDetail( $testId );
            $tapping = SMTTestMode::isTappingMode( $linkTestDetailDto->getTestMode());
            $traceName = $linkTestDetailDto->getOtdrTraceName();
            $SOM = $linkTestDetailDto->getFirstMarkerDistanceM();
            $EOM = $linkTestDetailDto->getLastMarkerDistanceM();
        
            //check if sliding reference exists
            if ( $slidingReference )
            {
            	$traceFilePath = SMTOtdrTraceFileUtil::OTU_MEASURE_REF_SLIDING_DIR.SMTOtdrTraceFileUtil::checkSorFileExtension( $traceName );
            	$smartAcqTraceExtension = SMTBaseOtdrTraceFileUtil::getSmartAcqFileExtension($traceFilePath, $slidingReferenceNotFound);
            }
            if ( !$slidingReference || $slidingReferenceNotFound )
            {
            	$traceFilePath = SMTOtdrTraceFileUtil::OTU_MEASURE_REF_DET_DIR.SMTOtdrTraceFileUtil::checkSorFileExtension( $traceName );
        		$smartAcqTraceExtension = SMTBaseOtdrTraceFileUtil::getSmartAcqFileExtension($traceFilePath);
            }
            
            if ( !$tapping && $lastAcquisition )
            {
            	$portNumber = SMTLinkTest::getPortNumber($this->getContext()->getOtuSocket(), intval($testId));
            	$hasAcquisitionLoc = SMTSessionDirectoryUtil::hasLastAcquisitionLocalisation(intval($testId), intval($portNumber));
            }
        	
            $smartAcq = SMTBaseOtdrTraceFileUtil::isSmartAcqFileExtension($smartAcqTraceExtension) && ($tapping || !$lastAcquisition || $hasAcquisitionLoc);
            $traceFilePath= SMTBaseOtdrTraceFileUtil::replaceExtension($traceFilePath, $smartAcqTraceExtension);
            
            SMTLogger::getInstance()->trace( sprintf("Retrieve otdr reference monitoring trace file %s slidingReference %d lastAcquisition %d hasAcquisitionLoc %d",  $traceFilePath , $slidingReference, $lastAcquisition, $hasAcquisitionLoc), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        	
        	$page = new SMTOtdrTracePage( $traceFilePath, SMTOtdrTraceType::OTU_REFERENCE_TRACE, SMTTraceFileType::getFileType($traceFilePath), $smartAcq, $SOM, $EOM);
        	
        	$this->getContext()->getHttpResponse()->send( $page );
    	}    	
    	catch ( \Exception $e )
    	{
    	    $this->handleException($e);
    	}    
    }
    
    /**
     * Retrieve OTDR monitoring reference trace by its testId and its name; if name is invalid, same as getMonitoringReferenceTrace()
     * Warning: if a sliding reference exists, return it
     *          if the reference is requested to be displayed with a last test acquisition: display short and long acq ( smartAcq = TRUE ) only if the last test acquisition is a localisation or if it's a tapping test 

     * //param testId The test with the monitoring reference trace
     * //param name The reference trace name: usefull when replacing the reference of a monitoring test and the new reference is not yet used by OTU monitoring.
     *  
     * @url POST /referencetrace/{testId}/{traceName}
     * @status 200
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getMonitoringReferenceTraceByName( $testId, $traceName,$request_data)
    {
        $traceFileFound = FALSE;
        $slidingReferenceNotFound= TRUE;
        $hasAcquisitionLoc = FALSE;
        $tapping = FALSE;
        $SOM = 0;
        $EOM = 0;
        
    	try
    	{
    		$slidingReference = ($request_data != NULL && $request_data['sliding'] != NULL)? filter_var( $request_data['sliding'], FILTER_VALIDATE_BOOLEAN ) : FALSE;
    		$lastAcquisition = ($request_data != NULL && $request_data['lastAcquisition'] != NULL)? filter_var( $request_data['lastAcquisition'], FILTER_VALIDATE_BOOLEAN ) : FALSE;    		
    		
    		//retrieve OTDR trace name
    		SMTLinkTest::checkTestId($testId);
    		
    		//check if sliding reference exists
    		if ( $slidingReference )
    		{
    			$traceFilePath = SMTOtdrTraceFileUtil::OTU_MEASURE_REF_SLIDING_DIR.SMTOtdrTraceFileUtil::checkSorFileExtension( $traceName );
    			$smartAcqTraceExtension = SMTBaseOtdrTraceFileUtil::getSmartAcqFileExtension($traceFilePath, $slidingReferenceNotFound);
    		}
    		if ( !$slidingReference || $slidingReferenceNotFound )
    		{
    			$traceFilePath = SMTOtdrTraceFileUtil::OTU_MEASURE_REF_DET_DIR.SMTOtdrTraceFileUtil::checkSorFileExtension( $traceName );
    			$smartAcqTraceExtension = SMTBaseOtdrTraceFileUtil::getSmartAcqFileExtension($traceFilePath);
    		}
    		 
    		$linkHelper = new SMTLinkTest();
    		$linkHelper->setContext( $this->getContext() );
    		$linkTestDetailDto = $linkHelper->retrieveLinkTestDetail( $testId );
    		$tapping = SMTTestMode::isTappingMode( $linkTestDetailDto->getTestMode());
    		$SOM = $linkTestDetailDto->getFirstMarkerDistanceM();
    		$EOM = $linkTestDetailDto->getLastMarkerDistanceM();
    		
    		//if the file is not found, use the reference declared in OTU
    		if ( !file_exists( $traceFilePath ) )
    		{
    		    $traceName = $linkTestDetailDto->getOtdrTraceName();
    		    
    		    //check if sliding reference exists
    		    if ( $slidingReference )
    		    {
    		    	$traceFilePath = SMTOtdrTraceFileUtil::OTU_MEASURE_REF_SLIDING_DIR.SMTOtdrTraceFileUtil::checkSorFileExtension( $traceName );
    		    	$smartAcqTraceExtension = SMTBaseOtdrTraceFileUtil::getSmartAcqFileExtension($traceFilePath, $slidingReferenceNotFound);
    		    }
    		    if ( !$slidingReference || $slidingReferenceNotFound )
    		    {
    		    	$traceFilePath = SMTOtdrTraceFileUtil::OTU_MEASURE_REF_DET_DIR.SMTOtdrTraceFileUtil::checkSorFileExtension( $traceName );
    		    	$smartAcqTraceExtension = SMTBaseOtdrTraceFileUtil::getSmartAcqFileExtension($traceFilePath);
    		    }
    		}
    		
    		if ( !$tapping && $lastAcquisition )
    		{
    			$portNumber = SMTLinkTest::getPortNumber($this->getContext()->getOtuSocket(), intval($testId));
    			$hasAcquisitionLoc = SMTSessionDirectoryUtil::hasLastAcquisitionLocalisation(intval($testId), intval($portNumber));
    		}
    		
    		$smartAcq = SMTBaseOtdrTraceFileUtil::isSmartAcqFileExtension($smartAcqTraceExtension) && ($tapping || !$lastAcquisition || $hasAcquisitionLoc);
    		
    		$traceFilePath= SMTBaseOtdrTraceFileUtil::replaceExtension($traceFilePath, $smartAcqTraceExtension);
    		
    		SMTLogger::getInstance()->trace( sprintf("Retrieve otdr reference monitoring trace file by name %s slidingReference %d lastAcquisition %d hasAcquisitionLoc %d",  $traceFilePath , $slidingReference, $lastAcquisition, $hasAcquisitionLoc), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    		$page = new SMTOtdrTracePage( $traceFilePath, SMTOtdrTraceType::OTU_REFERENCE_TRACE, SMTTraceFileType::getFileType($traceFilePath), $smartAcq, $SOM, $EOM);
		    $this->getContext()->getHttpResponse()->send( $page );
    		
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    }    
    
    /**
     * Retrieve OTDR localisation alarm trace on the given test if it exists
     *
     * //param testId The test with the alarm acquisition trace. (One attenuation alarm by test).
     *
     * @url POST /alarmtrace/localisation/{testId}
     * @status 200
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getLocalisationAlarmTrace( $testId )
    {
    	try
    	{
    		//retrieve OTDR trace name
    		SMTLinkTest::checkTestId($testId);
    		$portNumber = SMTLinkTest::getPortNumber($this->getContext()->getOtuSocket(), intval($testId));
    		
    		$alarmHelper = new SMTAlarm();
    		$alarmHelper->setContext( $this->getContext() );
    		$alarmDto = $alarmHelper->fetchOpticalAlarmFromTest( $testId );
    		
    		if ( $alarmDto != NULL )
    		{
    			$traceFilePath = SMTOtdrTraceFileUtil::buildAlarmTraceName( $testId, $portNumber, $alarmDto->getDiscriminator() );
    		    
    		    $smartAcqTraceExtension = SMTBaseOtdrTraceFileUtil::getSmartAcqFileExtension($traceFilePath);
    		    $smartAcq = SMTBaseOtdrTraceFileUtil::isSmartAcqFileExtension($smartAcqTraceExtension);
    		    $traceFilePath= SMTBaseOtdrTraceFileUtil::replaceExtension($traceFilePath, $smartAcqTraceExtension);
    		    
    		    SMTLogger::getInstance()->trace( sprintf("Retrieve alarm trace file %s ",  $traceFilePath ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    		    
    		    $page = new SMTOtdrTracePage( $traceFilePath, SMTOtdrTraceType::OTU_ALARM_TRACE, SMTTraceFileType::getFileType($traceFilePath), $smartAcq);
    		}
    		else
    		{
    			$alarmsDtoList = $alarmHelper->fetchPeakAlarmFromTest($testId);
    			
    			if ( ($alarmsDtoList != NULL) && count($alarmsDtoList) > 0 )
    			{
    				foreach ( $alarmsDtoList as $alarmDto)
    				{
    					if ( $alarmDto!= NULL )
    					{
    						$alarmPeakByTestDiscriminator = SMTAlarmFactory::getDiscriminatorForOtdrTrace( $alarmDto->getTestId());
    						
    						$traceFilePath = SMTOtdrTraceFileUtil::buildAlarmTraceName( $testId, $portNumber, $alarmPeakByTestDiscriminator );
    						$smartAcqTraceExtension = SMTBaseOtdrTraceFileUtil::getSmartAcqFileExtension($traceFilePath);
    						$smartAcq = SMTBaseOtdrTraceFileUtil::isSmartAcqFileExtension($smartAcqTraceExtension);
    						$traceFilePath= SMTBaseOtdrTraceFileUtil::replaceExtension($traceFilePath, $smartAcqTraceExtension);
    						
    						SMTLogger::getInstance()->trace( sprintf("Retrieve alarm trace file %s ",  $traceFilePath ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    						$page = new SMTOtdrTracePage( $traceFilePath, SMTOtdrTraceType::OTU_ALARM_TRACE, SMTTraceFileType::getFileType($traceFilePath), $smartAcq);
    						break;
    					}
    				}
    			}
	    		else
	    		{
	    		    throw new SMTAlarmException( SMTAlarmException::ERROR_ALARM_RETRIEVAL_FAILED, "testId = ".$testId );
	    		}
    		}
    	$this->getContext()->getHttpResponse()->send( $page );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    }
    
    /**
     * Retrieve OTDR localisation alarm trace on the given test if it exists
     *
     * //param testId The test with the alarm acquisition trace. (One attenuation alarm by test).
     *
     * @url POST /alarmtrace/shortacq/{testId}
     * @status 200
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getShortAcqAlarmTrace( $testId )
    {
    	try
    	{
    		//retrieve OTDR trace name
    		SMTLinkTest::checkTestId($testId);
    
    		$alarmHelper = new SMTAlarm();
    		$alarmHelper->setContext( $this->getContext() );
    		$alarmDto = $alarmHelper->fetchOpticalAlarmFromTest( $testId );
    
    		if ( $alarmDto != NULL )
    		{
    			$portNumber = SMTLinkTest::getPortNumber($this->getContext()->getOtuSocket(), intval($testId));
    			$traceFilePath = SMTOtdrTraceFileUtil::buildAlarmShortAcqTraceName( $testId, $portNumber, $alarmDto->getDiscriminator() );
    			if ( file_exists( $traceFilePath ) )
    			{
    			    SMTLogger::getInstance()->trace( sprintf("Retrieve alarm trace file %s ",  $traceFilePath ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    			    $page = new SMTOtdrTracePage( $traceFilePath, SMTOtdrTraceType::OTU_SHORTACQ_ALARM_TRACE, false );
    			    $this->getContext()->getHttpResponse()->send( $page );
    			}
    		}
    		else
    		{
    			throw new SMTAlarmException( SMTAlarmException::ERROR_ALARM_RETRIEVAL_FAILED, "testId = ".$testId );
    		}    		
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    }    
    
    /**
     * Retrieve OTDR detection alarm trace on the given test if it exists (only sor file)
     *
     * //param testId The test with the alarm acquisition trace. (One attenuation alarm by test).
     *
     * @url POST /alarmtrace/detection/{testId}
     * @status 200
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getDetectionAlarmTrace( $testId )
    {
    	try
    	{
    		//retrieve OTDR trace name
    		SMTLinkTest::checkTestId($testId);
    		$portNumber = SMTLinkTest::getPortNumber($this->getContext()->getOtuSocket(), intval($testId));
    		
    		$alarmHelper = new SMTAlarm();
    		$alarmHelper->setContext( $this->getContext() );
    		$alarmDto = $alarmHelper->fetchOpticalAlarmFromTest( $testId );
    
    		if ( $alarmDto != NULL )
    		{
    			$traceFilePath = SMTOtdrTraceFileUtil::buildAlarmTraceName( $testId, $portNumber, $alarmDto->getDiscriminator(), FALSE );
    			$smartAcqTraceExtension = SMTBaseOtdrTraceFileUtil::getSmartAcqFileExtension($traceFilePath);
    			$smartAcq = SMTBaseOtdrTraceFileUtil::isSmartAcqFileExtension($smartAcqTraceExtension);
    			$traceFilePath= SMTBaseOtdrTraceFileUtil::replaceExtension($traceFilePath, $smartAcqTraceExtension);
    			
    			SMTLogger::getInstance()->trace( sprintf("Retrieve alarm trace file %s ",  $traceFilePath ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    			$page = new SMTOtdrTracePage( $traceFilePath, SMTOtdrTraceType::OTU_ALARM_TRACE, SMTTraceFileType::getFileType($traceFilePath), $smartAcq);
    		}
    		else
    		{
    			$alarmsDtoList = $alarmHelper->fetchPeakAlarmFromTest($testId);
    			
    			if ( ($alarmsDtoList != NULL) && count($alarmsDtoList) > 0 )
    			{
    				foreach ( $alarmsDtoList as $alarmDto)
	    			{
	    				if ( $alarmDto!= NULL )
	    				{
	    					$alarmPeakByTestDiscriminator = SMTAlarmFactory::getDiscriminatorForOtdrTrace( $alarmDto->getTestId());
	    					
	    					$traceFilePath = SMTOtdrTraceFileUtil::buildAlarmTraceName( $testId, $portNumber, $alarmPeakByTestDiscriminator, FALSE );
	    					$smartAcqTraceExtension = SMTBaseOtdrTraceFileUtil::getSmartAcqFileExtension($traceFilePath);
	    					$smartAcq = SMTBaseOtdrTraceFileUtil::isSmartAcqFileExtension($smartAcqTraceExtension);
	    					$traceFilePath= SMTBaseOtdrTraceFileUtil::replaceExtension($traceFilePath, $smartAcqTraceExtension);
	    					
	    					SMTLogger::getInstance()->trace( sprintf("Retrieve alarm trace file %s ",  $traceFilePath ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
	    					$page = new SMTOtdrTracePage( $traceFilePath, SMTOtdrTraceType::OTU_ALARM_TRACE, SMTTraceFileType::getFileType($traceFilePath), $smartAcq);
	    					break;
	    				}
	    			}
    			}
    			else
    			{    			
    				throw new SMTAlarmException( SMTAlarmException::ERROR_ALARM_RETRIEVAL_FAILED, "testId = ".$testId );
    			}
    		}
    
    		$this->getContext()->getHttpResponse()->send( $page );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    }    
    
    /**
     * Retrieve last OTDR acquisition on the given test
     *
     * //param testId The test at the origin of the last acquisition
     *
     * @url POST /lasttrace/{testId}
     * @status 200
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getLastMonitoringAcquisitionTrace( $testId )
    {
    	$SOM = 0;
    	$EOM = 0;
    	$isLoc = FALSE;
    	
    	try
    	{
    		//retrieve OTDR trace name
    		SMTLinkTest::checkTestId($testId);
    		
    		$linkHelper = new SMTLinkTest();
    		$linkHelper->setContext( $this->getContext() );
    		$linkTestDetailDto = $linkHelper->retrieveLinkTestDetail( $testId );
    		$SOM = $linkTestDetailDto->getFirstMarkerDistanceM();
    		$EOM = $linkTestDetailDto->getLastMarkerDistanceM();
    		
    		$portNumber = SMTLinkTest::getPortNumber($this->getContext()->getOtuSocket(), intval($testId));
    		$traceFilePath = SMTBaseOtdrTraceFileUtil::retrieveLastMonitoringAcquisitionTrace( $testId, $portNumber, $isLoc, TRUE);
    		$smartAcqTraceExtension = SMTBaseOtdrTraceFileUtil::getSmartAcqFileExtension($traceFilePath);
    		$smartAcq = SMTBaseOtdrTraceFileUtil::isSmartAcqFileExtension($smartAcqTraceExtension) && (SMTTestMode::isTappingMode( $linkTestDetailDto->getTestMode()) || $isLoc );
    		
    		SMTLogger::getInstance()->trace( sprintf("Retrieve last otdr monitoring acquisitiontrace file %s localisation : %d  smartAcq: %d",  $traceFilePath, $isLoc, $smartAcq), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
    		
    		$page = new SMTOtdrTracePage( $traceFilePath, SMTOtdrTraceType::LAST_TEST_ACQUISITION_TRACE, SMTTraceFileType::getFileType($traceFilePath), $smartAcq, $SOM, $EOM);
    		
    		$this->getContext()->getHttpResponse()->send( $page );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}    	 
    }    
    
    /**
     * Retrieve last OTDR acquisition, or alarm localisation (or detection) trace or reference trace
     *
     * @url POST /files
     * @status 200
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function downloadAcquisitionFile($request_data)
    {
        $portid = $request_data['Port_Number'];
        $traceType = trim($request_data['Trace_Type'],"\"");
        
        try
        {
            if ( SMTUtil::isInt( $portid ) && ($portid > 0) )
            {
                $linkHelper = new SMTLinkTest();
                $linkHelper->setContext( $this->getContext() );
                 
                //test if a link exists:
                $linkId = $this->sendReceive( SMTOtuApi::getLinkListPerPortCommand( intval($portid), intval($portid) ) );

                if ( SMTLinkTest::checkLinkTestId( $linkId ) )
                {
                	//does a test exist?
                	$testId = SMTLinkTest::parseLinkTestId( $this->sendReceive( SMTOtuApi::getTestOnLinkCommand( $linkId ) ) );
                	if ( SMTLinkTest::checkLinkTestId($testId) )
                	{
                	    switch ($traceType)
                	    {
                	        case SMTOtdrTraceType::REF:
                	            $otdrTraceType = SMTOtdrTraceType::OTU_REFERENCE_TRACE;
    			                $traceName = sprintf( SMTOtdrTraceFileUtil::REF_TRACE_NAME, $portid);
    			                $traceFilePath = SMTOtdrTraceFileUtil::OTU_MEASURE_REF_DET_DIR.$traceName;
                	            break;
                	        case SMTOtdrTraceType::ALARM:
                	            $otdrTraceType = SMTOtdrTraceType::OTU_ALARM_TRACE;
                	            $alarmHelper = new SMTAlarm();
                        		$alarmHelper->setContext( $this->getContext() );
                        		$alarmDto = $alarmHelper->fetchOpticalAlarmFromTest( $testId );
                        		
                        		if ( $alarmDto != NULL )
                        		{    		 
                        			$traceFilePath = SMTOtdrTraceFileUtil::buildAlarmTraceName( $testId, $portid, $alarmDto->getDiscriminator() );
                        		    SMTLogger::getInstance()->trace( sprintf("Retrieve alarm trace file %s ",  $traceFilePath ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);  		   
                        		}
                        		else
                        		{
                        		    throw new SMTAlarmException( SMTAlarmException::ERROR_ALARM_RETRIEVAL_FAILED, "testId = ".$testId );
                        		}
                	        	break;
            	        	case SMTOtdrTraceType::ACQ:
            	        	    $otdrTraceType = SMTOtdrTraceType::LAST_TEST_ACQUISITION_TRACE;
            	        	    //for trace download we want always a SOR file
            	        	    $traceFilePath = SMTOtdrTraceFileUtil::retrieveLastMonitoringAcquisitionTrace( $testId, $portid, $isLoc, FALSE);
            	        		break;
            	        	default:
            	        	    throw new SMTLinkTestException( SMTLinkTestException::ERROR_MISSING_OTDR_TRACE, "invalid trace type ".$traceType );
            	        	    break;
                	    }
                	    
                	    SMTLogger::getInstance()->trace( sprintf("Retrieve otdr trace file %s, for type %s ",  $traceFilePath, $otdrTraceType ), SMTLogger::INFO, __FILE__,__METHOD__,__LINE__);                	    
                	    $page = new SMTOtdrTraceFileDownloadPage( $traceFilePath, $otdrTraceType );                	    
                	    $this->getContext()->getHttpResponse()->send( $page );
                	}
                }
            }
            else
            {
                throw new SMTRestException( SMTRestException::SERVICE_NOT_FOUND , MSG_ERROR_INVALID_PORT, SMTLinkTestException::ERROR_INVALID_PORT, "invalid port ".$portid );
            }
        }
        catch ( SMTException $e )
        {
            $ex = new SMTRestException( SMTRestException::SERVICE_NOT_FOUND , $e->getMessage()." ".$e->getDetailedMessage(), $e->getCode(), $e->getTrace() );
            $this->handleException($ex);
        }      
        catch ( SMTException $catchAll )
        {
        	$ex = new SMTRestException( SMTRestException::SERVICE_NOT_FOUND , $catchAll->getMessage(), $catchAll->getCode(), $catchAll->getTrace() );
        	$this->handleException($ex);
        }  
    }
    
    /**
     * @url GET /test/measure/{portNumber} 
     * 
     */
    function testStartMeasureOnDemand( $portNumber )
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$measurementDto = SMTMeasurement::createMeasurementTestData( $portNumber );
        	$url = '/acquisitions/measure';
        	$json_string = json_encode( $measurementDto->getJsonData() );
        	$page = new SMTTestRestApiPage($url, $json_string, 'PUT');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }
    
    /**
     * @url GET /test/measureParameters/{portNumber}
     *
     */
    function testFetchMeasureOnDemandParameters( $portNumber )
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$measurementDto = SMTMeasurement::createMeasurementTestData( $portNumber );
        	$url = '/acquisitions/measureParameters/'.$portNumber;
        	$json_string = json_encode( $measurementDto->getJsonData() );
        	$page = new SMTTestRestApiPage($url, NULL, 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }    
  
    /**
     * @url GET /test/gettrace/{operationId}
     *
     */
    function testGetAcquisitionTrace( $operationId )
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/acquisitions/trace/'.$operationId;
        	//     	$page = new SMTTestRestApiPage($url, $json_string, 'POST');
        	$page = new SMTTestRestApiPage($url, NULL, 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }    
    
    /**
     * @url GET /test/gettracetest/{testId}
     *
     */
    function testGetMonitoringReferenceTrace( $testId )
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/acquisitions/referencetrace/'.$testId;
        	//     	$page = new SMTTestRestApiPage($url, $json_string, 'POST');
        	$page = new SMTTestRestApiPage($url, NULL, 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }
    
}