<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\http;

use app\serviceshelper\otau\SMTSwitchManager;

use app\serviceshelper\module\SMTModule;

use app\parser\SMTOtuSocketManager;

use app\database\SMTSmartOtuDB;

use app\SMTRestlerRouter;


use app\admin\SMTLanguage;
use app\admin\SMTUserDto;
use app\parser\SMTOtuSocket;
use app\parser\SMTSocketException;
use app\settings\SMTOtuStatusCacheDto;
use app\settings\SMTSmartOtuSettings;
use app\settings\SMTSmartOTUStatusValues;
use app\util\SMTLogger;

//include resource file
include_once("app/resources/resource_".SMTLanguage::retrieveLanguage().".php");

/**
 * HTTP request context:
 * 
 * - http request
 * - http response
 * - otu address
 * - otu port
 * - socket on the OTU parser application (on demand)
 * - database connection (on demand)
 * - user session (when PHP session is required)
 * 
 * @author sdesplat
 */
class SMTContext
{
    /**
     * 
     * @var SMTHttpResponse
     */
    private $httpResponse = NULL;
    
    /**
     *
     * @var SMTHttpRequest
     */    
    private $httpRequest = NULL;
    
    /**
     * PHP session: NULL by default: only authentification requests and request using access control (SMTAccessControl: @access protected)  need session. Others bypass PHP session. 
     * @var SMTSession
     */
    private $session = NULL;
    
    /**
     * @var SMTLogger
     */
    private $logger = NULL;
    
    /**
     * SmartOtu database object
     * 
     * @var SMTSmartOtuDB
     */
    private $database = NULL;
    
    /**
     * SmartOTU socket manager
     * 
     * @var SMTOtuSocketManager
     */
    private $otuSocketManager = NULL;
    
    /**
     * Build the application context:
     * - create the http request and response
     * - restaure the session.
     * 
     */
    function __construct()
    {
        $this->logger = SMTLogger::getInstance();
        $this->httpResponse = new SMTHttpResponse();
        $this->httpRequest = new SMTHttpRequest();                      
    }
    
    public function __destruct()
    {
    	$this->session = NULL;
    	$this->httpResponse = NULL;
    	$this->httpRequest = NULL;
    	
    	//clean up database connection if it was initialized
    	if ( $this->database != NULL )
    	{
        	$this->database->close();
        	$this->database = NULL;
    	}
    }
        
    /**
     * Restore user PHP session or create a new one.
     * WARNING: only authentification HTTP requests and HTTP request using access control (SMTAccessControl: @access protected) need session. Others bypass PHP session.
     * 
     * @param $checkSessionTimedOut boolean whether the session timeout must be checked: TRUE by default
     */
    private function restoreSessionContext( $checkSessionTimeout = TRUE )
    {              
        //restore session
        $this->session = new SMTSession( $checkSessionTimeout );
    }
    
    /**
     * Get current user PHP session or create a new one if needed
     *
     * @param $checkSessionTimedOut boolean whether the session timeout must be checked if the session has not yet been restored: TRUE by default
     * 
     * @return SMTSession the current PHP session or a new one if none existed
     */
    public function getSessionContext( $checkSessionTimeout = TRUE )
    {
    	if ( $this->session == NULL)
    	{
    		//restore session
    		$this->restoreSessionContext( $checkSessionTimeout );
    	}
    	return $this->session;
    }
    
    /**
     * Open database connection in user context
     */
    private function restoreDatabaseConnection()
    {
    	//open database connection
    	$this->database = SMTSmartOtuDB::getInstance( $this );
    }
            
    /**
     * Destroy running session if it exists.
     *
     */
    function closeSession()
    {
        //don't check timeout when closing a session
        $session = $this->getSessionContext( FALSE );
        $session->setUser( NULL );
        $session->closeSession();
        
        //cleanup options at logout
        SMTSmartOtuSettings::setOtuOption(NULL);
    }
    
    /**
     * Returns a SmartOTU database SQLite3 object (create the connection if needed)
     * 
     * @return \app\database\SMTSmartOtuDB
     */
    public function getDatabase()
    {
        if ( $this->database == NULL || $this->database->isClosed() )
        {
            $this->restoreDatabaseConnection();
        }
    	return $this->database;
    }

    /**
     * Retrieve context from the restler framework router
     * 
     * @param SMTRestlerRouter $router
     * @return SMTContext 
     */
    static function getContext( SMTRestlerRouter $router )
    {
        return $router->getContext();
    }           
    
    /**
     * 
     * @return \app\http\SMTHttpRequest
     */
    public function getHttpRequest()
    {
        return $this->httpRequest;
    }
    
    /**
     * 
     * @return \app\http\SMTHttpResponse
     */
    public function getHttpResponse()
    {
    	return $this->httpResponse;
    }    
    
    /**
     * Set user in session. If session has not yet been initialized, restore it
     *
     * @param SMTUserDto
     */
    public function setUser( SMTUserDto $user = NULL )
    {
        //mandatory: ensure session context is initialized
        $session = $this->getSessionContext();
        $session->setUser($user);
    }
    
    /**
     * Retrieve user from session. If session has not yet been initialized, restore it
     *
     * @return SMTUserDto
     */
    public function getUser()
    {
        //mandatory: ensure session context is initialized
        $session = $this->getSessionContext();
        return $session->getUser();
    }
    
    /**
     * 
     * @param integer $port
     */
    public function setOtuPort( $port )
    {
        SMTSmartOtuSettings::setOtuPort( $port );
    }
    
    /**
     * Get port number from settings if it exists or retrieve it be querying ISU 
     * 
     * @return number
     */
    public function getOtuPort()
    {
        $port = SMTSmartOtuSettings::getOtuPort();
        //check otu port
        if ( $port === NULL )
        {
        	$port = SMTOtuSocketManager::retrieveOtuPort( $this );
        	$this->setOtuPort( $port );
        }
        
    	return $port;
    }    
    
    /**
     * Set whether the OTU is in local mode.
     *
     */
    public function setLocalMode()
    {
        if ( !$this->isLocalMode() )
        {
	        self::setSmartOTUStatus( SMTSmartOTUStatusValues::LOCAL_MODE );
	        SMTLogger::getInstance()->trace("************* LOCAL MODE ***************", SMTLogger::ERROR );
        }	    
    }
    
    /**
     * Set whether the OTU is in rescue mode.
     *
     */
    public function setRescueMode()
    {
    	if ( !$this->isRescueMode() )
    	{
    		self::setSmartOTUStatus( SMTSmartOTUStatusValues::RESCUE_MODE );
    		SMTLogger::getInstance()->trace("************* RESCUE MODE ***************", SMTLogger::ERROR );
    	}
    }

    /**
     * Set whether the OTU is in normal mode.
     *
     * @param boolean $localMode
     */
    public function setNormalMode()
    {
        //never set OTU in normal mode when it is rebooting, starting or upgraded
    	if ( self::getSmartOTUStatus() != SMTSmartOTUStatusValues::OK 
    	        && self::getSmartOTUStatus() != SMTSmartOTUStatusValues::OTU_STARTING
    	        && self::getSmartOTUStatus() != SMTSmartOTUStatusValues::OTU_UPGRADING
    	        && self::getSmartOTUStatus() != SMTSmartOTUStatusValues::REBOOT )
    	{
    		self::setSmartOTUStatus( SMTSmartOTUStatusValues::OK );
    	}
    }
    
    /**
     * Returns whether the OTU is in local mode
     *
     * @param $forceReloadOtuStatus boolean FALSE by default. TRUE to force the reload of OTU status from APC memory
     * 
     * @return boolean
     */
    public function isLocalMode( $forceReloadOtuStatus = FALSE )
    {
        if ( $forceReloadOtuStatus )
        {
        	SMTOtuStatusCacheDto::reloadSmartOTUStatusFromMemory();
        }
    	return (self::getSmartOTUStatus() === SMTSmartOTUStatusValues::LOCAL_MODE);
    }                 

    /**
     * Returns whether the OTU is in rescue mode
     *
     * @param $forceReloadOtuStatus boolean FALSE by default. TRUE to force the reload of OTU status from APC memory
     *
     * @return boolean
     */
    public function isRescueMode( $forceReloadOtuStatus = FALSE )
    {
    	if ( $forceReloadOtuStatus )
    	{
    		SMTOtuStatusCacheDto::reloadSmartOTUStatusFromMemory();
    	}
    	return (self::getSmartOTUStatus() === SMTSmartOTUStatusValues::RESCUE_MODE);
    }
    
    /**
     * Set OTU application status as not available in SmartOTU
     *
     */
    public function setOTUNotAvailable ()
    {
    	try
    	{
    	    //Set OTU "not available" only if the otu is available (don't change reboot, starting, upgrade... status)
    	    if ( $this->isOTUAvailable() )
    	    {
    	    	//Ensure status has not been changed in memory; force the reload of status from memory:
    	    	if ( $this->isOTUAvailable( TRUE ) )
    	    	{
    	    		self::setSmartOTUStatus( SMTSmartOTUStatusValues::OTU_NOT_AVAILABLE );
    	    	}
    	    }    	       			
    	}
    	//never throw the exception: OTU "not available" property is set when processing socket errors ( except Otu "available" property)
    	catch ( \Exception $e )
    	{
    		$this->traceException( $e );
    	}
    }
    
    /**
     * Set OTU application status as available in SmartOTU
     *
     */
    public function setOTUAvailable ()
    {
        try 
        {
            //never set OTU available when it is rebooting or upgraded;
            //set otu available when it is not available or when it is starting
            if ( $this->isOTUStarting() || self::getSmartOTUStatus() === SMTSmartOTUStatusValues::OTU_NOT_AVAILABLE )
            {
            	//ensure status has not been changed in memory; force the reload of status from memory:
            	if ( $this->isOTUStarting( TRUE ) || self::getSmartOTUStatus() === SMTSmartOTUStatusValues::OTU_NOT_AVAILABLE )
            	{
            		self::setSmartOTUStatus( SMTSmartOTUStatusValues::OK );
            	}            	
            }            
        }
        //never throw the exception: OTU "not available" property is set when processing socket errors ( except Otu "available" property)
        catch ( \Exception $e )
        {
            $this->traceException( $e );
        }
    }
    
    /**
     * Set whether OTU application is starting
     *
     */
    public function setOTUStarting ()
    {
    	try
    	{
    	    //Force OTU status to "starting" if the current status is not already "starting"
    	    if ( !$this->isOTUStarting() )
    	    {
    	    	self::setSmartOTUStatus( SMTSmartOTUStatusValues::OTU_STARTING );
    	    }  			
    	}
    	//never throw the exception: OTU "not available" property is set when processing socket errors ( except Otu "available" property)
    	catch ( \Exception $e )
    	{
    		$this->traceException( $e );
    	}
    }
    

    /**
     * Returns true if the OTU application is starting.
     * 
     * @param $forceReloadOtuStatus boolean FALSE by default. TRUE to force the reload of OTU status from APC memory
     * 
     * @return boolean
     */
    public function isOTUStarting( $forceReloadOtuStatus = FALSE )
    {
        if ( $forceReloadOtuStatus )
        {
        	SMTOtuStatusCacheDto::reloadSmartOTUStatusFromMemory();
        }
    	return (self::getSmartOTUStatus() === SMTSmartOTUStatusValues::OTU_STARTING);
    }    
    
    /**
     * Returns true if the OTU application is available.
     * 
     * @param $forceReloadOtuStatus boolean FALSE by default. TRUE to force the reload of OTU status from APC memory
     * 
     * @return boolean
     */
    public function isOTUAvailable( $forceReloadOtuStatus = FALSE )
    {
        if ( $forceReloadOtuStatus )
        {
            SMTOtuStatusCacheDto::reloadSmartOTUStatusFromMemory();
        }
    	return (self::getSmartOTUStatus() === SMTSmartOTUStatusValues::OK || self::getSmartOTUStatus() === SMTSmartOTUStatusValues::LOCAL_MODE);
    }
    
    /**
     * Returns true if the OTU application is under upgrade.
     *
     * @param $forceReloadOtuStatus boolean FALSE by default. TRUE to force the reload of OTU status from APC memory
     * 
     * @return boolean
     */
    public function isOTUUpgrade( $forceReloadOtuStatus = FALSE )
    {
        if ( $forceReloadOtuStatus )
        {
        	SMTOtuStatusCacheDto::reloadSmartOTUStatusFromMemory();
        }
    	return (self::getSmartOTUStatus() === SMTSmartOTUStatusValues::OTU_UPGRADING);
    }
    
    /**
     * Set whether OTU application is under upgrade
     *
     */
    public function setOTUUpgrade ()
    {
    	try
    	{
    	    //Force OTU status to "upgrade" if the current status is not already "upgrade"
    	    if ( !$this->isOTUUpgrade() )
    	    {
    		    self::setSmartOTUStatus( SMTSmartOTUStatusValues::OTU_UPGRADING );
    	    }
    	}
    	//never throw the exception: OTU "not available" property is set when processing socket errors ( except Otu "available" property)
    	catch ( \Exception $e )
    	{
    		$this->traceException( $e );
    	}
    }
    
    /**
     * Set smartOTU application status as reboot.
     *
     */
    public function setOTUReboot ()
    {
        //Force OTU status to "reboot" if the current status is not already "reboot"
        if ( !$this->isOTUReboot() )
        {
        	self::setSmartOTUStatus( SMTSmartOTUStatusValues::REBOOT );
        }           		
    }
    
    /**
     * Returns true if the OTU application is rebooting
     * 
     * @param $forceReloadOtuStatus boolean FALSE by default. TRUE to force the reload of OTU status from APC memory
     * 
     * @return boolean
     */
    public function isOTUReboot( $forceReloadOtuStatus = FALSE )
    {
        if ( $forceReloadOtuStatus )
        {
        	SMTOtuStatusCacheDto::reloadSmartOTUStatusFromMemory();
        }
    	return (self::getSmartOTUStatus() === SMTSmartOTUStatusValues::REBOOT);
    }     
    
	/**
	 * Set smartOTU status among the SMTSmartOTUStatusValues possible values
	 * 
	 * @param string $smartOTUStatus
	 */
	private static function setSmartOTUStatus ( $smartOTUStatus )
    {
        SMTLogger::getInstance()->trace( "SmartOTUStatus: ".$smartOTUStatus, SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
    	SMTOtuStatusCacheDto::setSmartOTUStatus ( $smartOTUStatus );
    }
    
    /**
     * Returns the current smartOTUStatus or NULL if none is set
     *
     * @return number
     */
    public static function getSmartOTUStatus()
    {
    	return SMTOtuStatusCacheDto::getSmartOTUStatus();
    }
    
    /**
     * Set Module config
     *
     * @param boolean $moduleConfigOK 
     */
    public function setModuleConfigOK( $moduleConfigOK )
    {
    	SMTOtuStatusCacheDto::setModuleConfigOK( $moduleConfigOK );
    }
    
    /**
     * Returns whether the module config is OK
     *
     * @return boolean
     */
    public function getModuleConfigOK()
    {
        $moduleConfigOK = SMTOtuStatusCacheDto::getModuleConfigOK();
        if ( $moduleConfigOK == NULL )
        {
        	$module = new SMTModule();
        	$module->setContext( $this );
        	$moduleConfigOK = !$module->isDifference();
        	$this->setModuleConfigOK($moduleConfigOK);
        }
        
    	return $moduleConfigOK;
    }        

    /**
     * Set the switch config
     *
     * @param boolean $switchConfigOK
     */
    public function setSwitchConfigOK( $switchConfigOK )
    {
    	SMTOtuStatusCacheDto::setSwitchConfigOK( $switchConfigOK );
    }    
    
    /**
     * Returns whether the switch config is OK
     *
     * @return boolean
     */
    public function getSwitchConfigOK()
    {
        $switchConfigOK = SMTOtuStatusCacheDto::getSwitchConfigOK();
        if ( $switchConfigOK == NULL )
        {
        	$switchManager = new SMTSwitchManager();
        	$switchManager->setContext( $this );
        	$switchConfigOK = !$switchManager->isDifference(FALSE);
        	$this->setSwitchConfigOK( $switchConfigOK );
        }
    	return $switchConfigOK;
    }
    
    /**
     * 
     * @return string
     */
    public function getOtuAddress()
    {
    	return SMTSmartOtuSettings::getOtuAddress();
    }    
    
    /**
     *
     * @return \app\settings\SMTSmartOtuSettings
     */
    public function getSettings()
    {
    	return SMTSmartOtuSettings::getInstance();
    }    
    
    /**
     * Returns the current socket opened on OTU parser or create it if needed
     * 
     * @throws SMTSocketException Throws exception if a connection cannot be opened on otu parser
     * @return \app\parser\SMTOtuSocket
     */
    public function getOtuSocket()
    {
        $socketManager = $this->getOtuSocketManager();
        if ( $socketManager->getCurrentOtuSocket() === NULL )
        {
            //one socket on parser for one http query => no need to protect by a critical section that creation of socket             
            try
            {            	 
            	//retrieve connection from the socket manager
            	$socketManager->restoreSocket();
            }
            //no connection to OTU application
            catch (SMTSocketException $e)
            {
            	$this->traceException( $e );
           		throw $e;

            }                       
        }        
        return $socketManager->getCurrentOtuSocket();
    }
    
    /**
     * Returns the current OTU parser socket manager or create it if needed
     *
     * @return \app\parser\SMTOtuSocketManager
     */
    public function getOtuSocketManager()
    {
    	if ( $this->otuSocketManager === NULL )
    	{
    		//retrieve connection from the socket manager
    		$this->otuSocketManager = new SMTOtuSocketManager( $this );
    	}
    	return $this->otuSocketManager;
    }        
    
    /**
     * 
     * @return \app\util\SMTLogger
     */
    public function getLogger()
    {
        return $this->logger;
    }
    
    /**
     * Log message in OTU query file
     *
     * @param string $msg
     * @param string $level by default trace level = self::INFO
     * @param string $queryDate The date of the query execution (used for logs) or NULL
     * @param $resultMessage string the result of the query (NULL for a command)
     */
    function traceOtuQuery( $msg , $level = SMTLogger::INFO, $queryDate= NULL, $resultMessage = NULL)  
    {
    	$this->getLogger()->traceOtuQuery($msg, $level, $queryDate, $resultMessage);
    }
    
    /**
     *
     * @param Exception $e
     */
    function traceException( \Exception $e )
    {
    	$this->getLogger()->traceException($e);
    }           
}


?>