from datetime import datetime
from struct import pack, unpack, calcsize
from copy import deepcopy
from binascii import unhexlify, hexlify

class StructTypes(object):
    def __init__(self, name, pack, fmt, default):
        self._name = name
        self._pack = pack
        self._fmt = fmt
        self._value = default

    def pack(self):
        return self._pack

    def value(self, change = None):
        if change is not None:
            self._value = change
        return self._value

    def size(self):
        return calcsize(self._pack)

    def name(self):
        return self._name

    def load(self, buffer, raw = None):
        size = self.size()
        self._value = buffer[0]
        return 1

    def save(self):
        return [self._value]

    def __str__(self):
        if len(self._fmt) == 0:
            return ""
        return self._name + ": " + self._fmt % self._value

    def to_string(self):
        return ["%s"%self]

    def init(self, platform):
        pass


class STRING(StructTypes):
    def __init__(self, name, size, default=""):
        self._len = size
        self._name = name
        self._pack = "%is"%size
        self._fmt = "%s"
        self._value = default

    def load(self, buffer, raw = None):
        self._value = buffer[0].replace(b"\0", b"").decode("ASCII")
        return 1

    def save(self):
        raw = self._value.encode("ASCII")
        for i in range(self._len - len(raw)):
            raw += b'\0'
        return [raw]

    def __str__(self):
        return self._name + ": " + self._fmt % self._value

    def value(self, change = None):
        if change is not None:
            self._value = change
        return self._value


class CHAR(StructTypes):
    def __init__(self, name):
        super(CHAR, self).__init__(name, "b", "%c", 0)


class BLOB(StructTypes):
    def __init__(self, name, size):
        self._len = size
        self._name = name
        self._pack = "%iB"%size
        self._fmt = "%s"
        self._value = bytes(size)

    def load(self, buffer, raw = None):
        self._value = bytes(buffer[0:self._len])
        return 1

    def save(self):
        return self._value + (b'\0' * (self._len - len(self._value)))

    def __str__(self):
        if len(self._fmt) == 0:
            return ""
        return self._name + ": " + hexlify(self._value).decode("ascii")

    def value(self, change = None):
        if change is not None:
            self._value = unhexlify(change)
        return hexlify(self._value).decode("ascii")

class BYTE(StructTypes):
    def __init__(self, name):
        super(BYTE, self).__init__(name, "b", "0x%x", 0)


class BOOLEAN(StructTypes):
    def __init__(self, name):
        super(BOOLEAN, self).__init__(name, "?", "%r", False)


class SIGNED_8(StructTypes):
    def __init__(self, name):
        super(SIGNED_8, self).__init__(name, "c", "%i", 0)


class UNSIGNED_8(StructTypes):
    def __init__(self, name):
        super(UNSIGNED_8, self).__init__(name, "C", "%i", 0)


class SIGNED_16(StructTypes):
    def __init__(self, name):
        super(SIGNED_16, self).__init__(name, "h", "%i", 0)


class UNSIGNED_16(StructTypes):
    def __init__(self, name):
        super(UNSIGNED_16, self).__init__(name, "H", "%i", 0)


class SIGNED_32(StructTypes):
    def __init__(self, name):
        super(SIGNED_32, self).__init__(name, "i", "%i", 0)


class UNSIGNED_32(StructTypes):
    def __init__(self, name):
        super(UNSIGNED_32, self).__init__(name, "I", "%i", 0)


class DATETIME(StructTypes):
    def __init__(self, name):
        super(DATETIME, self).__init__(name, "BBHBBBB", "%s", datetime.now())

    def load(self, buffer, raw = None):
        (Day, Month, Year, Hour, Minute, Second, Day_Of_Week) = buffer[0:7]
        self._value = datetime(Year, Month, Day, Hour, Minute, Second)
        return 7

    def save(self):
        return [self._value.day, self._value.month, self._value.year, self._value.hour, self._value.minute, self._value.second, self._value.weekday() + 1]



class Padding(StructTypes):
    def __init__(self, name, size):
        self._name = name
        self._pack = ""
        self._size = size
        self._fmt = "%s"
        self._value = "0*"
        self._initialized = False

    def resize(self, size):
        if not self._initialized :
            self._pack = "x" * (self._size - size)
            self._value = "0*%i"%len(self._pack)
            self._initialized = True

    def load(self, buffer, raw = None):
        return 0

    def save(self):
        return []


class Align(StructTypes):
    def __init__(self):
        self._name = ""
        self._pack = ""
        self._size = 0
        self._fmt = ""
        self._value = 0
        self._initialized = False

    def resize(self, size):
        if not self._initialized :
            self._pack = "x" * (size % 2)
            self._initialized = True

    def load(self, buffer, raw = None):
        return 0

    def save(self):
        return []


class Section(StructTypes):
    def __init__(self, name = None):
        self._fields = deepcopy(self.fields)
        if name is None:
            self._name = self.__class__.__name__
        else:
            self._name = name
        self.data = {}
        struct_format = ['>']
        struct_format.extend([a.pack().strip("<>") for a in self._fields])
        self._pack = "".join(struct_format)
        for v in self._fields:
            self.data[v.name()] = v
        size = 0
        for fld in self._fields:
            size += fld.size()
            if type(fld) == Align:
                fld.resize(size)
        struct_format = ['>']
        struct_format.extend([a.pack().strip("<>") for a in self._fields])
        self._pack = "".join(struct_format)
        if type(self._fields[-1]) == Padding:
            self._fields[-1].resize(self.size())
            struct_format = ['>']
            struct_format.extend([a.pack().strip("<>") for a in self._fields])
            self._pack = "".join(struct_format)
        self._bin_data = None

    def load(self, buffer, raw = None):
        if type(buffer) == bytes:
            self._bin_data = buffer
            buffer = unpack(self.pack(), self._bin_data)
        else:
            self._bin_data = raw

        offset = 0
        bin_offset = 0
        for v in self._fields:
            offset += v.load(buffer[offset:], self._bin_data[bin_offset : bin_offset + v.size()])
            bin_offset += v.size()
        return offset

    def save(self):
        res = []
        for v in self._fields:
            res.extend(v.save())
        return res

    def is_valid(self):
        return None

    def to_string(self):
        res = [self._name]
        if self.is_valid() is not None:
            if self.is_valid():
                res[0] += " (valid)"
            else:
                res[0] += " (unvalid)"
        for item in self._fields:
            for line in item.to_string():
                res.append("\t%s" % line)
        return res

    def __str__(self):
        return "\n".join(self.to_string())

    def init(self, platform):
        for item in self._fields:
            if item.name() in platform.keys():
                item.init(platform[item.name()])
        for key in platform.keys():
            if type(platform[key]) != dict:
                if key in self.data.keys():
                    self.data[key].value(platform[key])


class Array(Section):
    def __init__(self, name, number, type):
        self._data = []
        self._pack = ""
        self._name = name
        self._bin_data = None
        for i in range(number):
            item = type("%s_%i"%(name, i))
            self._data.append(item)
            self._pack += item.pack().strip("<>")

    def load(self, buffer, raw=None):
        self._bin_data = raw
        offset = 0
        bin_offset = 0
        for i in range(len(self._data)):
            offset += self._data[i].load(buffer[offset:], self._bin_data[bin_offset : bin_offset + self._data[i].size()])
            bin_offset += self._data[i].size()

        return offset

    def save(self):
        res = []
        for v in self._data:
            res.extend(v.save())
        return res

    def to_string(self):
        res = [self._name]
        for item in self._data:
            for line in item.to_string():
                res.append("\t%s"%line)
        return res

    def init(self, platform):
        pass

    def item(self, index):
        return self._data[index]


def read_struct_from_file(filename, classtype):
    bin_struct = None
    with open(filename, "rb") as file:
        bin_struct = classtype()
        stream = file.read(bin_struct.size())
        bin_struct.load(stream)
    return bin_struct


def write_struct_to_file(filename, bin_struct):
    data = bin_struct.save()
    raw = pack(bin_struct.pack(), *data)
    with open(filename, "wb") as w:
        w.write(raw)
