<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\serviceshelper\monitoring;

use app\serviceshelper\otdr\SMTOtdrTraceFileUtil;

use app\services\otdr\SMTAcquisitionParametersDto;

use app\events\SMTEventFactory;

use app\services\otdr\SMTStartOtdrMeasureOnDemandParametersDto;



use app\serviceshelper\otdr\SMTCommonMeasurement;

use app\util\SMTLogger;

use app\events\operations\SMTOperationEventDto;

use app\parser\SMTOtuApi;
use app\services\setup\SMTModuleDto;
use app\serviceshelper\otdr\SMTBaseOtdrTraceFileUtil;

/**
 * Start and query the status of a measurement on demand
 * 
 * @author Sylvain Desplat
 */
class SMTMeasureOnDemand extends SMTCommonMeasurement
{    
    /**
     * Start measurement on link
     * 
     * @param SMTStartOtdrMeasureOnDemandParametersDto $acquisitionParameters
     * 
     * @return SMTMeasureOnDemandEventDto;
     * 
     * @throws SMTMeasureOnDemandException if measurement launch has failed
     */
    function startMeasureOnLink( SMTStartOtdrMeasureOnDemandParametersDto $acquisitionParameters )
    {                               
        $measurementStatus = SMTEventFactory::createMeasureOnDemandEvent( $acquisitionParameters->getLinkId(), $acquisitionParameters->getOperationId(), SMTOperationEventDto::NO_PROGRAMMED );        
        $measurementStatus->setPortNumber( $acquisitionParameters->getPortNumber() );
        $measurementStarted = FALSE;
        
        if ( $this->acquireMeasureOnDemandLock() )
        {
            if ( $acquisitionParameters->getOperationId() == NULL )
            {
            	$operationId = SMTMeasureOnDemandEventDto::generateMeasureOnDemandOperationId();
            }
            else
            {
            	$operationId = $acquisitionParameters->getOperationId();
            }
            
            //check that no measurement is running
            //if a measurement is already running, abort start measurement operation
            try
            {
            	$measurementStatus = $this->checkNoMeasurementRunning( $acquisitionParameters->getPortNumber() , $operationId );
            }
            catch ( \Exception $e )
            {
                $this->releaseMeasureOnDemandLock();
            	$this->getContext()->getLogger()->traceException($e);
            	throw $e;
            }
            
            // mesure dwdm?
            $idDwdm = $acquisitionParameters->getDwdmFrequencyGhz() != NULL;
            
            try 
            {
            	if ( $idDwdm )
            	{
            		$tracePathInfos = $this->sendReceive( SMTOtuApi::getExecuteMeasurementOnLinkDwdmCommand(
            			$acquisitionParameters->getLinkId(),
            			$acquisitionParameters->getAcquisitionMode(),
            			$acquisitionParameters->getPulseNs(),
            			$acquisitionParameters->getRangeKm(),
            			$acquisitionParameters->getAcquisitionTimeSec(),
            			$acquisitionParameters->getFiberIndex(),
            			0,
            			$acquisitionParameters->getResolutionCm(),
            			$acquisitionParameters->getFunctionName(),
            			$acquisitionParameters->getShortAcq(),
            			$acquisitionParameters->getDwdmFrequencyGhz() ) );
            	}
            	else
            	{
	                $tracePathInfos = $this->sendReceive( SMTOtuApi::getExecuteMeasurementOnLinkCommand( 
	                        $acquisitionParameters->getLinkId(), 
	                        $acquisitionParameters->getAcquisitionMode(), 
	                        $acquisitionParameters->getPulseNs(), 
	                        $acquisitionParameters->getRangeKm(), 
	                        $acquisitionParameters->getAcquisitionTimeSec(), 
	                        $acquisitionParameters->getFiberIndex(), 
	                        $acquisitionParameters->getWavelengthNm(), 
	                        $acquisitionParameters->getResolutionCm(), 
	                        $acquisitionParameters->getFunctionName(),
	                		$acquisitionParameters->getShortAcq()) );
            	}
                
                $measurementStarted = TRUE;
                $tracePathDirNameArray = preg_split( "/[;]/", $tracePathInfos );
            } 
            catch ( \Exception $e ) 
            {
                //Start measurement failed, don't start the monitoring of the measurement progress
                $measurementStarted = FALSE;
                //nothing to do, try to get info with getMeasurementStatus
                $this->getContext()->getLogger()->traceException($e);
            }    	
            
            try 
            {
                $measurementStatus = $this->getMeasurementAddStatus( $operationId);
                $this->releaseMeasureOnDemandLock();
                
                //if measurement was successfully added, start the monitoring
                if ( $measurementStarted )
                {
        			//1 -measurement was started: save some measure features
        			$traceDirectory = array_shift( $tracePathDirNameArray );
        			if ( $traceDirectory === NULL || $traceDirectory == '' ) { $traceDirectory = SMTOtdrTraceFileUtil::OTU_MEASURE_ON_DEMAND_DIR; }
        			$traceName = array_shift( $tracePathDirNameArray );
        			if ( $traceName === NULL || $traceName == '' ) { $traceName= SMTOtdrTraceFileUtil::OTU_MEASURE_ON_DEMAND_FILE; }
        			$portId = $acquisitionParameters->getPortNumber();
        			$linkId = $acquisitionParameters->getLinkId();
        			$this->cacheMeasureInfo($operationId, $portId, $traceDirectory, SMTBaseOtdrTraceFileUtil::removeExtension( $traceName ), $linkId );
        			
        			//2 - measurement was started: start the polling on measure in an asynchronous process    
                    $measureOnDemandClass = SMTMeasureOnDemandEventDto::getClass();
                    $measureOnDemandClass = str_replace("\\", "\\\\", $measureOnDemandClass);
                    self::executeAsynchronousMeasureProcess( __DIR__.'/measureondemandProcess.php', $measureOnDemandClass, $operationId, $linkId );
                }
            }
            catch( \Exception $e )
            {                
                $this->releaseMeasureOnDemandLock();
                throw $e;
            }
        }
        else
        {
        	$this->getContext()->getLogger()->trace("Couldn't acquire lock for SMTMeasureOnDemand to start measure on link with operation Id:".$acquisitionParameters->getOperationId(), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        	throw new SMTMeasureOnDemandException( SMTOperationEventDto::FAILED, "Couldn't acquire lock to start measure on link" ) ;
        }        
        
        
        return $measurementStatus;
    }    
    
    /**
     * Create measurement on demand data for test purpose
     *
     * @return SMTStartOtdrMeasureOnDemandParametersDto
     */
    static function createMeasureOnDemandTestData()
    {
    	$measurementDto = new SMTStartOtdrMeasureOnDemandParametersDto();
    	$measurementDto->setAcquisitionModeAuto(TRUE);
    	$measurementDto->setLinkId( 0 );
    	$measurementDto->setTestId( 0 );
    	$measurementDto->setPortNumber( 1 );
    	$measurementDto->setModulePosition(SMTModuleDto::MOD2);
    	$measurementDto->setAcquisitionTimeSec( 10 );
    	$measurementDto->setFunctionName("SM-OTDR");
    	$measurementDto->setPulseNs(SMTAcquisitionParametersDto::DEFAULT_PULSE_NS);//10ns
    	$measurementDto->setRangeKm(0);//auto
    	$measurementDto->setResolutionCm(0);//auto
    	$measurementDto->setWavelengthNm( 1625 );
    	$measurementDto->setFiberIndex( SMTCommonMeasurement::DEFAULT_FIBER_INDEX );
    	 
    	return $measurementDto;
    }    
}
?>