<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\services\maintenance;

use app\admin\SMTOtuType;

use app\serviceshelper\maintenance\SMTLicence;

use app\parser\SMTOtuSocket;

use app\serviceshelper\maintenance\SMTCertificate;

use app\serviceshelper\ethernet\SMTEthernet;

use app\serviceshelper\maintenance\SMTSmartAccessAnywhere;

use app\util\upload\SMTUploadStatus;

use app\serviceshelper\maintenance\SMTOtuBackupRestore;

use app\events\SMTEventMessageManager;

use app\serviceshelper\alarm\SMTAlarm;

use app\serviceshelper\system\SMTOtu;

use app\settings\SMTSmartOtuSettings;

use app\serviceshelper\maintenance\SMTUpgrade;
use app\services\monitoring\SMTLinkTestListDto;
use app\serviceshelper\monitoring\SMTLinkTest;
use app\serviceshelper\maintenance\SMTOtuDebug;
use app\view\SMTFileDownloadPage;
use app\util\SMTIOException;
use app\util\SMTLogger;
use app\parser\SMTOtuSocketManager;
use app\util\SMTUtil;
use app\view\SMTTestRestApiPage;
use app\services\SMTIService;
use app\services\SMTService;
use app\parser\SMTOtuApi;

/**
 * Service to launch debug operations.
 * 
 * @author Sylvain Desplat
 */
class SMTMaintenanceService extends SMTService implements SMTIService
{      
    /**
     * Check OTU application availibility
     *
     * @url POST /
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getSmartOtuMaintenanceInfo()
    {
    	try
    	{
    		//Reboot the OTU
    		$dto = new SMTMaintenanceDto();    		
    		$dto->setDegradedMode( !$this->getContext()->isOTUAvailable() );
    		$dto->setRescueMode( $this->getContext()->isRescueMode() );    		
    		
    		$context = $this->getContext();
    		$highSecurityMode = SMTLicence::isHighSecurityMode($context)? SMTLicence::ENHANCED_SECURITY_ON : SMTLicence::ENHANCED_SECURITY_OFF;
    		$dto->setEnhancedSecurity($highSecurityMode);
    		
    		$sshKeyRootAccess = SMTLicence::isSshKeyRootAccessOption($context)? SMTLicence::SSH_KEY_ROOT_ACCESS_ON : SMTLicence::SSH_KEY_ROOT_ACCESS_OFF;
    		$dto->setSshKeyRootAccess($sshKeyRootAccess);
    		
    		if ( SMTOtuType::isCOtu())
    		{
	    		$otu = new SMTOtu();
    			$otu->setContext( $this->getContext() );
    			$dto->setOtuBlinking( $otu->isBlink() );
    		}
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $dto->getJsonData();
    }    
    
    /**
     * Set OTU Status LED blinking status
     *
     * @url POST /blink
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function blink($request_data)
    {
    	try
    	{
    		$dto = SMTMaintenanceDto::getInstance( $request_data );
    		
    		//Blink the OTU
    		$otu = new SMTOtu();
    		$otu->setContext( $this->getContext() );
    		$otu->blink($dto->isOtuBlinking());
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    }
    
    /**
     * Get OTU Status LED blinking status
     *
     * @url GET /blink
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getBlinkStatus()
    {
    	try
    	{
    		//Blink the OTU
    		$otu = new SMTOtu();
    		$otu->setContext( $this->getContext() );
    		$blink = $otu->isBlink();
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return array( SMTOtu::BLINK => $blink);
    }
    
    /**
     * Reboot OTU
     *
     * @url POST /system/reboot
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function reboot()
    {
    	try
    	{
    		//Reboot the OTU
    		$otu = new SMTOtu();
    		$otu->setContext( $this->getContext() );
    		$otu->reboot();
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return array( SMTOtu::REBOOT => SMTOtuApi::RES_OK );
    }
    
    /**
     * Retrieve OTU local mode
     *
     * @url GET /system/localmode
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function isOtuLocalMode()
    {
        try
        {
            //force local mode retrieve
            $socket = $this->getContext()->getOtuSocket();
            $localMode = SMTEthernet::isLocalMode( $socket );
        }
        catch ( \Exception $e )
        {
            $this->handleException($e);
        }
    	return $localMode;
    }
    
    /**
     * Request to download an OTU snapshot of log files...
     * 
     * @url GET /system/snapshot
     * @status 200
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function downloadSnapshot()
    {
        try 
        {
            $snapshotName = SMTOtuDebug::retrieveSnapshotName();
            
            if ( $snapshotName != NULL )
            {
                $page = new SMTFileDownloadPage($snapshotName);
                $this->getContext()->getHttpResponse()->send( $page );
            }
            else
            {
                $e = new SMTIOException( SMTIOException::COULD_NOT_CREATE_SNAPSHOT );
                $this->handleException($e);
            }    	
        }
    	catch( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    }  
      
//     /**
//      * Request to download the given log file...
//      * @param $fileName
//      * @url GET /log
//      * @status 200
//      * @access protected
//      * @class SMTAccessControl {@requires install}
//      */
//     function downloadLogFile( $fileName)
//     {    	
//     	try
//     	{
//     	    $logFileName = SMTOtuDebug::retrieveLogFileName($fileName);
    	    
//         	if ( $logFileName != NULL )
//         	{
//         		$page = new SMTFileDownloadPage($logFileName);
//         		$this->getContext()->getHttpResponse()->send( $page );
//         	}
//         	else
//         	{
//         		$e = new SMTIOException( SMTIOException::FILE_NOT_FOUND );
//         		$this->handleException($e);
//         	}
//     	}
//     	catch( \Exception $e )
//     	{
//     		$this->handleException($e);
//     	}
//     }

    /**
     * Fetch log file names and log level
     *
     * @url GET /logfilenames
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function listLogFiles()
    {
        try 
        {
    	    $fileNames = SMTOtuDebug::listLogFileName();
    	    $logFileDto = new SMTLogFilesDto();
    	    $logFileDto->setLogFilesNames( $fileNames );
    	    $logFileDto->setLogLevel( SMTSmartOtuSettings::getLogLevel() );
        }
        catch( \Exception $e )
        {
            $this->handleException($e);
        }
    	return $logFileDto->getJsonData();
    }
    
    /**
     * Change logging level and return the log files dto
     *
     * @url PUT /loglevel/{logLevel}
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function updateLogLevel( $logLevel )
    {
    	try
    	{
    		SMTSmartOtuSettings::setLogLevel( $logLevel, $this->getContext() );
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $this->listLogFiles();
    }    
    
    /**
     * Init TEST IP function for SmartOEM
     *
     * @url POST /system/inittestip
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function initTestIp()
    {
    	try
    	{
    		$testIpDto = new SMTTestIpDto();
    		$portFO = SMTOtuSocketManager::retrieveFOPort( $this->getContext() );
    		$testIpDto->setPortFO( $portFO );
    		
    	}
    	catch ( \Exception $e )
    	{
    		$this->getContext()->getLogger()->traceException( $e );
    	}
    	return $testIpDto->getJsonData();
    }
    /**
     * Send the command encoded in POST to OTU (TEST IP function)
     *
     * @url POST /system/testip
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function testIp( $request_data )
    {
        try 
        {
            $result = SMTOtuApi::RES_OK;
            $testIpDto = SMTTestIpDto::getInstance( $request_data );

            $socketManager = new SMTOtuSocketManager( $this->getContext() );
            $socket = NULL;
            if ( $testIpDto->isOtuPort() == FALSE )
            {
                $port = $testIpDto->getPort();                
                if ( $port !== NULL && SMTUtil::isInt($port) )
                {
                    //retrieve connection from the socket manager
            	    $socket = $socketManager->openSocket( $port );
                }
            }
            
            //by default retrieve socket with current OTU port
            if ( $socket == NULL )
            {
                $socket = $socketManager->restoreSocket( FALSE );
            }
            
            if ( $testIpDto->isQuery() )
            {
                $result = $socket->sendReceive( $testIpDto->getCommand(), SMTOtuSocket::DEFAULT_SLEEP_FOR_ISU );
            }
            else
            {
                $socket->send( $testIpDto->getCommand() );
                
                //test whether the command deals with OTU sequensor
                if ( strpos( strtoupper( $testIpDto->getCommand() ), substr( SMTOtuApi::CMD_set_mon_seq_status, 0, -2) ) !== FALSE )
                {
                    //notify link-tests full refresh needed
                    SMTEventMessageManager::getInstance()->createAndSendLinkTestUpdateAllEvent();
                }                                 
            }
            $testIpDto->setResult($result);
            $testIpDto->setEsrCode( $socket->getLastEsrCode() );
            $testIpDto->setSuccess( TRUE );
            
            $socket->close();            
    	}
    	catch ( \Exception $e )
    	{
    	    $testIpDto->setSuccess( FALSE );
    	    $testIpDto->setResult( $e->getMessage() );
    	    if ( $socket !== NULL )
    	    {
    	        $testIpDto->setEsrCode( $socket->getLastEsrCode() );
    	        $socket->close();
    	    }    	    
    	}    			
    	return $testIpDto->getJsonData();
    }
    
    /**
     * Fetch Smart Access Anywhere connection code if it exists.
     *
     * @url GET /saa
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchSaaSetup()
    {
        $dto = new SMTSmartAccessAnywhereDto();
        $saa = new SMTSmartAccessAnywhere();
    	    	
    	try
    	{
    	    $saa->setContext( $this->getContext() );
            $dto = $saa->fetchSaaSetup();    	    
        } 
        catch ( \Exception $e ) 
        {
            //When saa is not enabled, the function call returns ESR 16: catch error and do nothing
            $dto->setAccessCode( NULL );
        }
        return $dto->getJsonData();
    }
    
    /**
     * Enable Smart Access Anywhere connection.
     *
     * @url PUT /saa/connect
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function connectSaa()
    {
    	$saa = new SMTSmartAccessAnywhere();
    	    	
    	try
    	{
    	    $saa->setContext( $this->getContext() );
            $dto = $saa->connectSaa();    	    
    	}
    	catch( \Exception $e)
    	{
    	    $this->handleException($e);    
    	}
    	
    	return $dto->getJsonData();
    }
    
    /**
     * Disable Smart Access Anywhere connection code if it exists.
     *
     * @url PUT /saa/disconnect
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function disconnectSaa()
    {
    	$saa = new SMTSmartAccessAnywhere();
    
    	try
    	{
    		$saa->setContext( $this->getContext() );
    		$dto = $saa->disconnectSaa();
    	}
    	catch( \Exception $e)
    	{
    		$this->handleException($e);
    	}
    	 
    	return $dto->getJsonData();
    }
    
    /**
     * Test SmartOtu - OTU communication and database access without access control.
     *
     * @url POST /system/ping
     * 
     */
    function ping()
    {
        $dto = new SMTOtuInfoDto();        
    	try
    	{
    	    $otuInfo = $this->sendReceive( SMTOtuApi::getOtuInfo() );
            list( $company, $otuType, $serialNumber, $otuMode, $otuReleaseVersion ) = explode(",", $otuInfo);
            
        	$dto->setCompany($company);
        	$dto->setOtuType(SMTOtuType::getOtuType());
        	$dto->setSerialNumber($serialNumber);
        	$dto->setOtuMode($otuMode);
        	$dto->setOtuSoftwareRelease( SMTSmartOtuSettings::getOtuVersion() );
        	$dto->setSmartOtuSoftwareRelease( SMTSmartOtuSettings::getSmartOtuVersion() );
        	        	 
        	$timestampUTC = SMTUtil::getTimestampUTC( time(), $this->getContext()->getDatabase() );
        	$dto->setCurrentTimeStampUTC( $timestampUTC );
        	$this->getContext()->getDatabase()->close();
    	}
    	catch( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $dto->getJsonData();
    }
    
    /**
     * Return the list of links with their test ids without test and alarm infos
     *
     * @url GET /links
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchLinkList()
    {
    	try
    	{
    		$linkHelper = new SMTLinkTest();
    		$linkHelper->setContext( $this->getContext() );
    		 
    		//retrieve the number of ports available:
    		$portCount = $linkHelper->getPortsCount();
    		 
    		//retrieve links ids
    		$linksId = explode(",", $this->sendReceive( SMTOtuApi::getLinkListCommand() ) );
    		$linkTestListDto = new SMTLinkTestListDto();
    		$linkTestListDto->setPortCount( $portCount );
    
    		$linkTestDto = NULL;
    		foreach ( $linksId as $linkId)
    		{
    			if ( SMTLinkTest::checkLinkTestId( $linkId, FALSE) )
    			{
    				$linkTestDto = $linkHelper->retrieveLinkInfo( $linkId );
    				if ( $linkTestDto != NULL )
    				{
    					$linkTestListDto->addLinkTest( $linkTestDto );
    				}
    			}
    		}
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $linkTestListDto->getJsonData();
    }    
        
    /**
     * Return the list of links in alarm with their test ids without test infos but with alarm infos
     *
     * @url GET /linksinalarm
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchLinkInAlarmList()
    {
    	try
    	{
    		$linkHelper = new SMTLinkTest();
    		$linkHelper->setContext( $this->getContext() );
    		 
    		//retrieve the number of ports available:
    		$portCount = $linkHelper->getPortsCount();
    		 
    		//retrieve links ids
    		$linksId = explode(",", $this->sendReceive( SMTOtuApi::getLinkListCommand() ) );
    		$linkTestListDto = new SMTLinkTestListDto();
    		$linkTestListDto->setPortCount( $portCount );
    
    		$linkTestDto = NULL;
    		foreach ( $linksId as $linkId)
    		{
    			if ( SMTLinkTest::checkLinkTestId( $linkId, FALSE) )
    			{
    				$linkTestDto = $linkHelper->retrieveLinkInfo( $linkId );
    				if ( $linkTestDto != NULL )
    				{
    					$linkTestListDto->addLinkTest( $linkTestDto );
    				}
    			}
    		}
    		
			$alarmHelper = new SMTAlarm();
			$alarmHelper->setContext( $this->getContext() );
			$alarmsDtoList = $alarmHelper->fetchOpticalAlarms();
				
			foreach ( $alarmsDtoList as $opticalAlarm)
			{
				if ( $opticalAlarm != NULL && !$opticalAlarm->isCleared())
				{
					$linkTestListDto->addOpticalAlarm( $opticalAlarm );
				}
			}
			$alarmsDtoList = $alarmHelper->fetchPeakAlarms();
			
			foreach ( $alarmsDtoList as $opticalAlarm)
			{
				if ( $opticalAlarm != NULL )
				{
					$linkTestListDto->addOpticalAlarm( $opticalAlarm );
				}
			}
			$linkTestListDto->removeLinkTestsWithoutAlarm();
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $linkTestListDto->getJsonData();
    }
    
    /**
     * Update license file
     * filename = license file
     *
     * @url POST /system/license/{filename}
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function updateLicense( $filename )
    {
    	try
    	{
    	    $dto = new SMTLicenseListDto();
    		$licenseHelper = new SMTLicence();
    		$licenseHelper->setContext( $this->getContext() );
    
    		$filename = strip_tags($filename);
    		
    		if ( !empty( $filename ) )
    		{
  			    $licenseHelper->processLicense($filename);
    		}
    		else
    		{
    		    throw new SMTIOException(SMTIOException::INVALID_FILE, "empty file");
    		}
    		
    		$softwareOptions = SMTLicence::retrieveSoftwareOptions($this->getContext());
    		$dto->setSoftwareOptions($softwareOptions);
    		SMTLicence::retrieveSoftwareOptionList($this->getContext(), $dto); 
    		
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    
    	return $dto->getJsonData();
    }

    /**
     * Request to retrieve the OTU license
     *
     * @url GET /system/license
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getLicense()
    {
    	try
    	{
    		$context = $this->getContext();
    		$dto = new SMTLicenseListDto();
    		$softwareOptions = SMTLicence::retrieveSoftwareOptions($context);
    		$dto->setSoftwareOptions($softwareOptions);
    		SMTLicence::retrieveSoftwareOptionList($context, $dto);    		
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $dto->getJsonData();
    }
    
    /**
     * Start the upgrade of the OTU file system
     *
     * @url POST /system/filesystem
     * 
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function startFileSystemUpgrade()
    {
        try
        {
            $upgradeDto = SMTUpgrade::getCurrentUpgrade();
            if ( $upgradeDto->isUpgradeInProgressOnOtu() != TRUE )
            {
                SMTUpgrade::startFileSystemUpgrade( $this->getContext() );
                
                //retrieve result                
                $upgradeDto = SMTUpgrade::getCurrentUpgrade();
                SMTUpgrade::getCurrentUpgradeProgressLog( $upgradeDto, $this->getContext() );          
            }
            else if ( $upgradeDto->getCurrentUpgradeName() == NULL )
            {
            	$upgradeDto->setUpgradeError( SMTUpgradeDto::NO_UPGRADE_ARCHIVE_ERROR );
            }
            else if ( $upgradeDto->isDuplicationInProgressOnOtu() )
            {
            	$upgradeDto->setUpgradeError( SMTUpgradeDto::DUPLICATION_ALLREADY_RUNNING_ERROR );
            }
            else
            {
                $upgradeDto->setUpgradeError( SMTUpgradeDto::UPGRADE_ALLREADY_RUNNING_ERROR );
            }
        }
        catch( \Exception $e )
        {
        	$this->handleException($e);
        }
        
        return $upgradeDto->getJsonData();
    }
    
    /**
     * Start the upgrade of the OTU rescue
     *
     * @url POST /system/rescue
     * 
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function startRescueUpgrade()
    {
    	try
    	{
            $upgradeDto = SMTUpgrade::getCurrentUpgrade();
            if ( $upgradeDto->isUpgradeInProgressOnOtu() != TRUE )
            {
                SMTUpgrade::startRescueUpgrade( $this->getContext() );
                
                //retrieve result
                $upgradeDto = SMTUpgrade::getCurrentUpgrade();
                SMTUpgrade::getCurrentUpgradeProgressLog( $upgradeDto, $this->getContext() );
            }
    	    else if ( $upgradeDto->getCurrentUpgradeName() == NULL )
            {
            	$upgradeDto->setUpgradeError( SMTUpgradeDto::NO_UPGRADE_ARCHIVE_ERROR );
            }
            else
            {
                $upgradeDto->setUpgradeError( SMTUpgradeDto::UPGRADE_ALLREADY_RUNNING_ERROR );
            }
        }
        catch( \Exception $e )
        {
        	$this->handleException($e);
        }
        
        return $upgradeDto->getJsonData();
    }    
    
    /**
     * Request to retrieve the name of the current upgrade file, the filesystem version, the rescue version to start a new upgrade
     *
     * @url POST /system/upgrade
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getUpgradeInfo()
    {
    	try
    	{
    		$upgradeDto = SMTUpgrade::getCurrentUpgrade();
    		SMTUpgrade::getCurrentUpgradeProgressLog( $upgradeDto, $this->getContext() );
    		SMTUpgrade::getCurrentDuplicationProgressLog( $upgradeDto, $this->getContext() );
    		$upgradeDto->setCurrentOtuRescueVersion( SMTSmartOtuSettings::getOtuRescueVersion() );
    		$upgradeDto->setCurrentOtuVersion( SMTSmartOtuSettings::getOtuVersion() );
    	}
    	catch( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $upgradeDto->getJsonData();
    }
    
    /**
     * Request to retrieve the name and the status of the current upgrade 
     *
     * @url GET /system/upgrade
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getCurrentUpgrade()
    {
    	try
    	{
    		$upgradeDto = SMTUpgrade::getCurrentUpgrade(FALSE);
    		SMTUpgrade::getCurrentUpgradeProgressLog( $upgradeDto, $this->getContext() );
    		$upgradeDto->setCurrentOtuRescueVersion( SMTSmartOtuSettings::getOtuRescueVersion() );
    		$upgradeDto->setCurrentOtuVersion( SMTSmartOtuSettings::getOtuVersion() );
    	}
    	catch( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $upgradeDto->getJsonData();
    }
       
    /**
     * Request to retrieve upload progress
     * id= OTU_UPLOAD_PROGRESS_KEY = otu-8kv2
     *
     * @url GET /upload/{id}
     * 
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getUploadStatus( $id )
    {
        try
        {
        	$uploadStatusDto = SMTUploadStatus::getUploadProgress( $id );
        	
        	if ( !$uploadStatusDto->isUploadInProgress() || $uploadStatusDto->hasErrorStatus() )
        	{
        	    //remove upload status from cache if upload has ended
        	    $uploadStatusDto->delete();
        	}
        }
        catch( \Exception $e )
        {
        	$this->handleException($e);
        }
        return $uploadStatusDto->getJsonData();
    }
    
    /**
     * Request to generate and download an OTU backup...
     * 
     * @url GET /system/backup
     * @status 200
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function getSmartOtuBackup()
    {
        $backupFullPath = NULL;
        try 
        {
            $backupHelper = new SMTOtuBackupRestore();
            $backupHelper->setContext( $this->getContext() );
            $backupFullPath = $backupHelper->retrieveSmartOtuBackup();

            if ( $backupFullPath != NULL )
            {
            	$page = new SMTFileDownloadPage( $backupFullPath );
            	$this->getContext()->getHttpResponse()->send( $page );
            }
            else
            {
            	$e = new SMTIOException( SMTIOException::COULD_NOT_CREATE_BACKUP );
            	$this->handleException($e);
            }
        }
        catch( \Exception $e )
        {
        	$this->handleException($e);
        }
    }
    
    /**
     * Request to restore the given smartOTU backup...
     *
     * @url POST /system/restore
     * @status 200
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function restoreSmartOtuBackup( $request_data )
    {
    	try
    	{
    	    $backupDto = SMTBackupDto::getInstance( $request_data );
    		$backupHelper = new SMTOtuBackupRestore();
    		$backupHelper->setContext( $this->getContext() );
    		$backupHelper->restoreSmartOtuBackup( $backupDto->getBackupFileName() );
    	}
    	catch( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    }
    
    /**
     * Request to retrieve the last backup available on OTU disk or NULL if none is found
     *
     * @url GET /system/backup/current
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function getCurrentBackup()
    {
    	try
    	{
    		$backupName = SMTOtuBackupRestore::getCurrentBackupName();
    		$backupFullPath = SMTOtuBackupRestore::checkBackupFileName( $backupName );
    		
    		if ( $backupFullPath != NULL )
    		{
    			$page = new SMTFileDownloadPage( $backupFullPath );
    			$this->getContext()->getHttpResponse()->send( $page );
    		}
    		else
    		{
    			$e = new SMTIOException( SMTIOException::FILE_NOT_FOUND );
    			$this->handleException($e);
    		}
    		
    	}
    	catch( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $backupNameDto->getJsonData();
    }

    /**
     * Request to retrieve the name of the last backup available on OTU disk or NULL if none is found
     *
     * @url GET /system/backup/current/name
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function getCurrentBackupName()
    {
    	try
    	{
    		$backupNameDto = SMTOtuBackupRestore::getLastAvailableBackupNameDto();
    	}
    	catch( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $backupNameDto->getJsonData();
    }
    
    /**
     * Request to update ssh public keys
     *
     * @url POST /system/certificate/ssh
     * @status 200
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function setSshPublicCertificate( $request_data )
    {
    	try
    	{
    		$dto = SMTCertificateDto::getInstance( $request_data );
    		$certificateHelper = new SMTCertificate();
    		$certificateHelper->setContext( $this->getContext() );
    		$certificateHelper->updateSshPublicKeys( $dto->getSshPublicKeys() );
    	}
    	catch( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	
    	//on retourne ce qui est lu dans le fichier pour vérification
    	return $this->getCertificate();
    }
    
    /**
     * Request to retrieve the ssh public keys (and https certificates in the future?)
     *
     * @url GET /system/certificate
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function getCertificate()
    {
    	try
    	{
    	    $context = $this->getContext();
    	    
    	    $certificateHelper = new SMTCertificate();
    		$certificateHelper->setContext( $context );
    		$lines = $certificateHelper->readSshPublicKeys();
    		$dto = new SMTCertificateDto();
    		$dto->setSshPublicKeys($lines);
    		
    		$sshKeyRootAccess = SMTLicence::isSshKeyRootAccessOption( $context )? SMTLicence::SSH_KEY_ROOT_ACCESS_ON : SMTLicence::SSH_KEY_ROOT_ACCESS_OFF;
    		$dto->setSshKeyRootAccess($sshKeyRootAccess);
    		
    		$highSecurityMode = SMTLicence::isHighSecurityMode($context)? SMTLicence::ENHANCED_SECURITY_ON : SMTLicence::ENHANCED_SECURITY_OFF;
    		$dto->setEnhancedSecurity($highSecurityMode);
    	}
    	catch ( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    	return $dto->getJsonData();
    }
    
    /**
     * Example: /maintenance/test/testip/otu:link:get/0/true
     * 
     * @url GET /test/testip/{command}/{parameters}/{isquery}
     */
    function testTestIp($command,$parameters,$isquery)
    {    
    	if ( SMTSmartOtuSettings::isDebugMode() )
    	{
	    	$url = '/maintenance/system/testip';
	    	$isQuery = filter_var($isquery, FILTER_VALIDATE_BOOLEAN);
	    	if ($isQuery)
	    	{
	    	    $command = urldecode( $command )."? ".$parameters;
	    	    SMTLogger::getInstance()->trace("command: ".$command);
	    	}
	    	else
	    	{
	    	    $command = urldecode( $command )." ".$parameters;
	    	    SMTLogger::getInstance()->trace("command: ".urldecode( $command ) );
	    	}
	    	$testIpDTO = new SMTTestIpDto();
	    	$testIpDTO->setCommand( $command );    	
	    	$testIpDTO->setQuery( $isQuery );
	    	
	    	$json_string = json_encode( $testIpDTO->getJsonData() );
	    	$page = new SMTTestRestApiPage($url, $json_string, 'POST');
	    	$this->getContext()->getHttpResponse()->send( $page );
    	}
    }   
    
    /**
     *
     * @url GET /test/log
     */
    function testTestListLogFiles()
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
            $url = '/maintenance/logfilenames';
            $page = new SMTTestRestApiPage($url, "", 'GET');
            $this->getContext()->getHttpResponse()->send( $page );
        }        
    }
    
//     /**
//      *
//      * @url GET /test/log/{filename}
//      */
//     function testTestDownloadLogFile( $filename )
//     {
//         if ( SMTSmartOtuSettings::isDebugMode() )
//         {
//         	$url = '/maintenance/log?fileName='.$filename;
//         	$page = new SMTTestRestApiPage($url, null, 'GET');
//         	$this->getContext()->getHttpResponse()->send( $page );
//         }
//     }
    
    
    /**
     * 
     * @param $stringToTest
     * 
     * @url GET /test/regexp
     */
    function testRegexp( $stringToTest)
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
            var_dump(SMTUtil::splitStringByComma($stringToTest));
        }
    }    
    
    /**
     *
     * @url GET /test/ping
     */
    function testPing()
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/maintenance/system/ping';
        	$page = new SMTTestRestApiPage($url, null, 'POST');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }    
    
    /**
     *
     * @url GET /test/snapshot
     */
    function testSnapshot()
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/maintenance/system/snapshot';
        	$page = new SMTTestRestApiPage($url, null, 'GET');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }
    
    /**
     *
     * @url GET /test/backup
     */
    function testBackup()
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/maintenance/system/backup';
        	$page = new SMTTestRestApiPage($url, null, 'GET');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }    
    
    /**
     *
     * @url GET /test/backup/current
     */
    function testCurrentBackup()
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
        	$url = '/maintenance/system/backup/current';
        	$page = new SMTTestRestApiPage($url, null, 'GET');
        	$this->getContext()->getHttpResponse()->send( $page );
        }
    }    
}

?>