<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2012. All rights reserved.
// *********************************************************
namespace app\services\alarm;

use app\serviceshelper\media\SMTWebService;

use app\error\SMTAccessForbiddenRestException;
use app\events\SMTEventMessageManager;
use app\parser\SMTOtuApi;
use app\services\security\SMTAccessControl;
use app\services\SMTIService;
use app\services\SMTService;
use app\serviceshelper\alarm\SMTAlarm;
use app\serviceshelper\alarm\SMTAlarmFactory;
use app\serviceshelper\monitoring\SMTGpsRoute;
use app\serviceshelper\monitoring\SMTLinkTest;
use app\settings\SMTSmartOtuSettings;
use app\util\SMTIOException;
use app\util\SMTLogger;
use app\util\SMTStatusDto;
use app\util\SMTUtil;
use app\view\SMTFileDownloadPage;
use app\view\SMTTestRestApiPage;

/**
 * Service to clear and list alarms.
 *
 * @author Sylvain Desplat
 */
class SMTAlarmService extends SMTService implements SMTIService
{        
    /**
     *
     * @url GET /optical
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchOpticalAlarms()
    {
        $alarmsList = array();
        
        try
        {    
            $alarmHelper = new SMTAlarm();
            $alarmHelper->setContext( $this->getContext() );
            $alarmsDtoList = $alarmHelper->fetchOpticalAlarms();
            
            foreach ( $alarmsDtoList as $alarmDto)
            {
                if ( $alarmDto != NULL )
                {
                    array_push( $alarmsList, $alarmDto->getJsonData() );
                }
            }
        }
        catch ( \Exception $e )
        {
            $this->handleException($e);
        }
    
        return $alarmsList;
    }
    
    /**
     *
     * @url GET /system
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchActiveSystemAlarms()
    {
        $alarmsList = new SMTSystemAlarmListDto();
        
        try
        {    
            $alarmHelper = new SMTAlarm();
            $alarmHelper->setContext( $this->getContext() );
            $alarmsDtoList = $alarmHelper->fetchActiveSystemAlarms();
            
            foreach ( $alarmsDtoList as $alarmDto)
            {
                if ( $alarmDto != NULL )
                {
                    $alarmsList->addSystemAlarm( $alarmDto );
                }
            }
        }
        catch ( \Exception $e )
        {
            $this->handleException($e);
        }
    
        return $alarmsList->getJsonData();
    }
    
    /**
     *
     * @url GET /optical/{id}
     *
     * @access protected
     * @class SMTAccessControl {@requires user}
     */
    function fetchOpticalAlarmDetail( $id )
    {
        try
        {    
            $alarmHelper = new SMTAlarm();
            $alarmHelper->setContext( $this->getContext() );
            $alarmDto = $alarmHelper->fetchOpticalAlarmDetail( $id );            
        }
        catch ( \Exception $e )
        {
            $this->handleException($e);
        }
    
        return ( $alarmDto != NULL)? $alarmDto->getJsonData() : NULL;
    }
    
    /**
     *
     * @url POST /
     * @status 200
     *
     * Access granted to everybody (used by OTU application), but control on host name sending the request: only loopback address allowed.
     */
    function sendAlarm( $request_data )
    {       
        $result = new SMTStatusDto();
        try
        {
            $request = $this->getContext()->getHttpRequest();
            
            $request = $this->getContext()->getHttpRequest();
            if ( !$request->isLocalHTTPQuery() && !SMTSmartOtuSettings::isDebugMode() )
            {
            	$this->getContext()->getLogger()->trace( sprintf("Remote host %s attempt to send autotest: forbidden!",$request->getClientIpAddress() ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            	throw new SMTAccessForbiddenRestException( MSG_ERROR_REMOTE_HOST_ATTEMPT_TO_SEND_DATA, "Host: ".$request->getClientIpAddress() );
            }
                        
                $this->getContext()->getLogger()->trace("receiving alarm..." );
                $this->getContext()->getLogger()->trace( print_r( $request_data, true ),SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
                
                $alarmOtuDto = SMTAlarmFactory::forgeOtuAlarmDto( $request_data );
                $this->getContext()->getLogger()->trace( "DTO Alarm:".print_r( $alarmOtuDto, true ),SMTLogger::INFO );
                
                $alarmHelper = new SMTAlarm();
                $alarmHelper->setContext( $this->getContext() );
                
                $alarms = $alarmHelper->processAlarm( $alarmOtuDto );                         
                $result->setStatusOK();
                
                //try notify alarms to webservice
                try 
                {
                    //notify alarms created or updated
                    foreach( $alarms as $alarm)
                    {
                    	SMTWebService::sendAlarmByWebService( $this->getContext(), $alarm );
                    }                       
                }
                catch(\Exception $ex)
                {
                    $this->getContext()->getLogger()->traceException( $ex );
                }
                
                try 
                {
                	SMTEventMessageManager::getInstance()->sendAlarmEvent( $alarms, $alarmOtuDto );
                    //notify alarms created or updated
//                     foreach( $alarms as $alarm)
//                     {                    
//                         SMTEventMessageManager::getInstance()->sendAlarmEvent( $alarm );
//                     }
                }
                catch ( \Exception $ex )
                {
                    $this->getContext()->getLogger()->traceException( $ex );
                }          
        }
        catch ( \Exception $e )
        {
            $result->setStatusKO();
            $this->handleException( $e );
        }
    
        return $result->getJsonData();
    }

    /**
     * WARNING EPT 5370 "otu:alarm:init" function on OTU application set the test status to "Clear" in memory, but doesn't save that test on disk.
     * 
     * @url DELETE /
     * @status 200
     *
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function clearAllAlarms()
    {
    	$result = new SMTStatusDto();
    	$alarmHelper = new SMTAlarm();
    	$alarmHelper->setContext( $this->getContext() );
    	     	
//that workaround generates clear alarms on OTU: if a reboot occurs just after that command, clears status on test are still not saved 
//     	try
//     	{    	    
//     	    //Workaround EPT 5370: try to clear individually optical alarms because "otu:alarm:init" function on OTU application 
//     	    //set the test status to "Clear" in memory, but doesn't save that test on disk 
//     	    $alarms = $alarmHelper->fetchOpticalAlarms();    	    
// 	        foreach ( $alarms as $alarm )
// 	        {
// 	            if ( !$alarm->isCleared() && SMTUtil::isInt( $alarm->getTestId() ) )
// 	            {
// 	                $alarmHelper->deleteOpticalAlarmOnOtu( $alarm->getTestId() );
// 	            }
// 	        }
//     	}
//     	//if individual clearing of optical alarms on OTU fails, don't exit processing and try to clear all alarms (optical and system)
//     	catch( \Exception $e )
//     	{
//     	    $this->getContext()->getLogger()->traceException( $ex );
//     	}
    	
    	try 
    	{
    	    //clear all alarms on OTU and smartOTU
    	    //WARNING EPT 5370 "otu:alarm:init" function on OTU application 
    	    //sets the test status to "Clear" in memory, but doesn't save that test on disk
    		$alarmHelper->clearAllAlarms();
    
    		//notify listeners that optical and system alarms should be refreshed
    		SMTEventMessageManager::getInstance()->sendRefreshAllAlarms();
    		$result->setStatusOK();
    	}
    	catch ( \Exception $e )
    	{
    		$result->setStatusKO();
    		$this->handleException( $e );
    	}
    	return $result->getJsonData();
    }
    
    /**
     * Delete only smartOTU alarms (not OTU alarms). Used by OTU application
     *
     * @url DELETE /smartotu
     * @status 200
     *
     * Access granted to everybody (used by OTU application), but control on host name sending the request: only loopback address allowed.
     */
    function clearAllSmartOTUAlarms()
    {
    	$result = new SMTStatusDto();
    	$alarmHelper = new SMTAlarm();
    	$alarmHelper->setContext( $this->getContext() );
    	
    	try
    	{
    		$request = $this->getContext()->getHttpRequest();
    		if ( !$request->isLocalHTTPQuery() && !SMTSmartOtuSettings::isDebugMode() )
    		{
    			$this->getContext()->getLogger()->trace( sprintf("Remote host %s attempt to send autotest: forbidden!",$request->getClientIpAddress() ), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
    			throw new SMTAccessForbiddenRestException( MSG_ERROR_REMOTE_HOST_ATTEMPT_TO_SEND_DATA, "Host: ".$request->getClientIpAddress() );
    		}
    		
    		$alarmHelper->clearAllAlarms(FALSE);
    		
    		//notify listeners that optical and system alarms should be refreshed
    		SMTEventMessageManager::getInstance()->sendRefreshAllAlarms();
    		$result->setStatusOK();
    	}
    	catch ( \Exception $e )
    	{
    		$result->setStatusKO();
    		$this->handleException( $e );
    	}
    	return $result->getJsonData();
    }
    
    /**
     *
     * @url DELETE /system/{id}
     * @status 200
     *
     * @access protected
     * @class SMTAccessControl {@requires install}
     */
    function clearSystemAlarm( $id )
    {
        $result = new SMTStatusDto();
    	try
    	{
    		$alarmHelper = new SMTAlarm();
    		$alarmHelper->setContext( $this->getContext() );
    		$alarmHelper->clearSystemAlarm( $id );    		
    		
    		//notify listeners that optical and system alarms should be refreshed
    		SMTEventMessageManager::getInstance()->sendRefreshAllAlarms();
    		$result->setStatusOK();
    	}
    	catch ( \Exception $e )
    	{
            $result->setStatusKO();
            $this->handleException( $e );
    	}
    	return $result->getJsonData();
    }    
    
    /**
     * Request to download the given kml file...
     * id = alarm identifier
     * Can be accessed outside SmartOTU: no security access
     *
     * @param $serverName
     * @param $linkId
     * 
     * @url GET /kml/{id}
     */
    function downloadKmlAlarm( $id, $serverName, $linkId )
    {
    	try
    	{
    	    $linkHelper = new SMTLinkTest();
    	    $linkHelper->setContext( $this->getContext() );
    	    $linkDto = $linkHelper->retrieveLinkInfo( $linkId );
    	    
    		$alarmHelper = new SMTAlarm();
    		$alarmHelper->setContext( $this->getContext() );
    		$alarmDto = $alarmHelper->fetchOpticalAlarmDetail( $id );
    		if ( ($alarmDto != NULL) && ($alarmDto->getCurrentAlarmEvent() != NULL) )
    		{
    		    
    		    $alarmId = $alarmDto->getId();
        		$gpsX = $alarmDto->getCurrentAlarmEvent()->getGpsX();
        		$gpsY = $alarmDto->getCurrentAlarmEvent()->getGpsY();        		
        		$description = $alarmDto->getSpecificProblemText();
        		$severity = $alarmDto->getCurrentAlarmEvent()->getSeverity();
        		$cleared = $alarmDto->isCleared();
        		$date = $alarmDto->getCurrentAlarmEvent()->getTime();
        		$hostName = $this->sendReceive( SMTOtuApi::CMD_get_otu_name );
        		$linkName = $linkDto->getName();        		
        		$date = SMTUtil::getTimestamp( $date, $this->getContext()->getDatabase() );
        		$this->getContext()->getDatabase()->close();
        		$kmlFileName = SMTGpsRoute::createKmlAlarm($alarmId, $linkId, $gpsX, $gpsY, $description, $severity, $cleared, $date, $linkName, $serverName, $hostName, $this->getContext());
    		}
    		 
    		if ( $kmlFileName != NULL )
    		{
    			$page = new SMTFileDownloadPage($kmlFileName);
    			$this->getContext()->getHttpResponse()->send( $page );
    		}
    		else
    		{
    			$e = new SMTIOException( SMTIOException::FILE_NOT_FOUND );
    			$this->handleException($e);
    		}
    	}
    	catch( \Exception $e )
    	{
    		$this->handleException($e);
    	}
    }    
    
    /**
     * @url GET /test/fetchopticalalarms
     * @status 200
     */
    function testFetchOpticalAlarms()
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
            $url = '/alarms/optical';
            $page = new SMTTestRestApiPage($url, NULL,'GET');
            $this->getContext()->getHttpResponse()->send( $page );
        }
    }
    
    /**
     * @url GET /test/fetchsystemalarms
     * @status 200
     */
    function testFetchSystemAlarms()
    {
        if ( SMTSmartOtuSettings::isDebugMode() )
        {
            $url = '/alarms/system';
            $page = new SMTTestRestApiPage($url, "",'GET');
            $this->getContext()->getHttpResponse()->send( $page );
        }
    }
    
    /**
     * @url GET /test/sendalarm
     * @status 200
     */
    function testSendAlarm()
    {       
        //if ( SMTSmartOtuSettings::isDebugMode() )
        {
            $json_string ="{"
            ."\"Type\":\"optical\","
            ."\"AlarmEventType\":\"QUALITY_OF_SERVICE_ALARM\","
            ."\"Date\":\"1371143249\","
            ."\"Test\":0,"
            ."\"TSDOperationId\":0,"
            ."\"Detection\":\"\","
            ."\"Localisation1\":\"\","
            ."\"Code\":1200,"
            ."\"CodeText\":\"Confirmed Alarm\","
            ."\"Version\":123456";
            
            if (FALSE)
            {
                $json_string.=",\"extension fiber\": {"
                ."\"Length\":0.0,"
                ."\"Severity\":,"
                ."\"IsValid\":,"
                ."\"SeverityText\":\"\""
                ."}";
            }
            if (FALSE)
            {
                $json_string.=",\"attenuation\": {"
                ."\"LevelValid\":\"true\","
                ."\"DistanceValid\":\"true\","
                ."\"Distance\":100.05,"
                ."\"Level\":4.35,"
                ."\"Severity\":\"MAJOR\","
                ."\"Specific\":\"INJECTION\""
                ."}";
            }
    
            if (TRUE)
            {
            	$json_string.=",\"Peaks\": ["
            	."{"
            	."\"ReferenceTopDistanceM\":12335.67,"
                ."\"MeasuredTopDistanceM\":12345.67,"
            	."\"MeasuredBottomDistanceM\":12349.99,"
            	."\"MeasuredLevelDb\":-5.99,"
            	."\"MeasuredLevelValid\":true,"
            	."\"Severity\":\"CRITICAL\","
            	."\"PeakName\":\"toto\""
            	."}"
            	."]";
            }
                $json_string.=",\"Deviation\": {"
                ."\"FirstMarkerValid\":\"true\","
                ."\"LinkLossValid\":\"true\","
                ."\"FirstMarker\":0.12,"
                ."\"LinkLoss\":-1.89";
                $json_string.="}";
    
            $json_string.="}";
            
            $url = '/alarms/';
            $page = new SMTTestRestApiPage($url, $json_string,'POST');
            $this->getContext()->getHttpResponse()->send( $page );
        }
    }
 
}