#
# Copyright (c) 2004 Conectiva, Inc.
#
# Written by Gustavo Niemeyer <niemeyer@conectiva.com>
#
# This file is part of Smart Package Manager.
#
# Smart Package Manager is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License as published
# by the Free Software Foundation; either version 2 of the License, or (at
# your option) any later version.
#
# Smart Package Manager is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
# General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Smart Package Manager; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#

from smart.interface import Interface, getScreenWidth
from smart.util.strtools import sizeToStr, printColumns
from smart.const import OPTIONAL, ALWAYS
from smart.fetcher import Fetcher
from smart.report import Report
from smart import *
import getpass
import sys
import os
import gobject
from smart.interface import getScreenWidth
from smart.util.strtools import ShortURL
from smart.progress import Progress
import posixpath
import time
import sys
import signal

class DBusProgress(Progress):

    def __init__(self, di):
        Progress.__init__(self)
        self._lasttopic = None
        self._lastsubkey = None
        self._lastsubkeystart = 0
        self._fetchermode = False
        self._seentopics = {}
        self._addline = False
        self.__di = di
        self.setScreenWidth(getScreenWidth())

    def setScreenWidth(self, width):
        self._screenwidth = width
        self._topicwidth = int(width*0.4)
        self._hashwidth = int(width-self._topicwidth-1)
        self._topicmask = "%%-%d.%ds" % (self._topicwidth, self._topicwidth)
        self._topicmaskn = "%%4d:%%-%d.%ds" % (self._topicwidth-5,
                                               self._topicwidth-5)
        self._shorturl = ShortURL(width-4)

    def handleScreenWidth(self, signum, frame):
        self.setScreenWidth(getScreenWidth())

    def setFetcherMode(self, flag):
        self._fetchermode = flag

    def stop(self):
        Progress.stop(self)
        self._shorturl.reset()
        print

    def expose(self, topic, percent, subkey, subtopic, subpercent, data, done):
        if subkey is None:
            subkey_ = ""
        else:
            subkey_ = subkey[1]
        if subtopic is None:
            subtopic_ = ""
        else:
            subtopic_ = subtopic
        if subpercent is None:
            subpercent_ = 0
        else:
            subpercent_ = subpercent
        self.__di.get_manager().expose(topic, percent, subkey_, subtopic_, subpercent_)
        out = sys.stdout
        if not out.isatty() and not done:
            return
        dispmsg = topic
        if self.getHasSub():
            if topic != self._lasttopic:
                self._lasttopic = topic
                out.write(" "*(self._screenwidth-1)+"\r")
                if self._addline:
                    print
                else:
                    self._addline = True
                print topic
            if not subkey:
                return
            if not done:
                now = time.time()
                if subkey == self._lastsubkey:
                    if (self._lastsubkeystart+2 < now and
                        self.getSubCount() > 1):
                        return
                else:
                    if (self._lastsubkeystart+2 > now and
                        self.getSubCount() > 1):
                        return
                    self._lastsubkey = subkey
                    self._lastsubkeystart = now
            elif subkey == self._lastsubkey:
                    self._lastsubkeystart = 0
            current = subpercent
            topic = subtopic
            if self._fetchermode:
                if topic not in self._seentopics:
                    self._seentopics[topic] = True
                    out.write(" "*(self._screenwidth-1)+"\r")
                    print "->", self._shorturl.get(topic)
                    dispmsg = self._shorturl.get(topic)
                topic = posixpath.basename(topic)
        else:
            current = percent
        n = data.get("item-number")
        if n:
            if len(topic) > self._topicwidth-6:
                topic = topic[:self._topicwidth-8]+".."
            out.write(self._topicmaskn % (n, topic))
        else:
            if len(topic) > self._topicwidth-1:
                topic = topic[:self._topicwidth-3]+".."
            out.write(self._topicmask % topic)
        if not done:
            speed = data.get("speed")
            if speed:
                suffix = "(%d%% %s %s)\r" % (current, speed, data.get("eta"))
            else:
                suffix = "(%3d%%)\r" % current
        elif subpercent is None:
            suffix = "[%3d%%]\n" % current
        else:
            suffix = "[%3d%%]\n" % percent

        hashwidth = self._hashwidth-len(suffix)

        hashes = int(hashwidth*current/100)
        out.write("#"*hashes)
        out.write(" "*(hashwidth-hashes+1))
        self.__di.get_manager().log("%s: %s"%(dispmsg, suffix))
        out.write(suffix)
        out.flush()

class DBusInterface(Interface):

    def __init__(self, ctrl):
        Interface.__init__(self, ctrl)
        self._progress = DBusProgress(self)
        self._activestatus = False
        self.__ctrl = ctrl

    def get_manager(self):
        return self.__ctrl.manager

    def iface_log(self, msg):
        sys.stdout.write(msg)
        sys.stdout.flush()
        self.get_manager().log(msg)

    def error(self,msg):
        self.iface_log("ERROR: %s"%msg)

    def getProgress(self, obj, hassub=False):
        self._progress.setHasSub(hassub)
        self._progress.setFetcherMode(isinstance(obj, Fetcher))
        return self._progress

    def getSubProgress(self, obj):
        return self._progress

    def showStatus(self, msg):
        if self._activestatus:
            print
        else:
            self._activestatus = True
        self.iface_log(msg)

    def hideStatus(self):
        if self._activestatus:
            self._activestatus = False
            print

    def askYesNo(self, question, default=False):
        self.hideStatus()
        mask = default and _("%s (Y/n): ") or _("%s (y/N): ")
        try:
            res = raw_input(mask % question).strip().lower()
        except (KeyboardInterrupt, EOFError):
            print
            return False
        print
        if res:
            return (_("yes").startswith(res) and not
                    _("no").startswith(res))
        return default

    def askContCancel(self, question, default=False):
        self.hideStatus()
        if default:
            mask = _("%s (Continue/cancel): ")
        else:
            mask = _("%s (continue/Cancel): ")
        try:
            res = raw_input(mask % question).strip().lower()
        except (KeyboardInterrupt, EOFError):
            print
            return False
        print
        if res:
            return (_("continue").startswith(res) and not
                    _("cancel").startswith(res))
        return default

    def askOkCancel(self, question, default=False):
        self.hideStatus()
        mask = default and _("%s (Ok/cancel): ") or _("%s (ok/Cancel): ")
        try:
            res = raw_input(mask % question).strip().lower()
        except (KeyboardInterrupt, EOFError):
            print
            return False
        print
        if res:
            return (_("ok").startswith(res) and not
                    _("cancel").startswith(res))
        return default

    def confirmChangeSet(self, changeset):
        return self.showChangeSet(changeset, confirm=True)

    def askInput(self, prompt, message=None, widthchars=None, echo=True):
        self.error("askInput (%s)"%prompt)
        return ""

    def askPassword(self, location, caching=OPTIONAL):
        self.error("askPassword")
        return ""

    def insertRemovableChannels(self, channels):
        self.error("insertRemovableChannels")
        return False

    # Non-standard interface methods:
        
    def showChangeSet(self, changeset, keep=None, confirm=False):
        self.hideStatus()
        report = Report(changeset)
        report.compute()


        if not sysconf.get("explain-changesets", False):
            screenwidth = getScreenWidth()
            hideversion = sysconf.get("text-hide-version", len(changeset) > 40)
            def showPackages(pkgs, showrelations=None):
                if hideversion:
                    pkgs = [x.name for x in pkgs]
                pkgs.sort()
                printColumns(pkgs, indent=2, width=screenwidth)
        else:
            def being(pkg):
                if (pkg in report.installing or
                    pkg in report.upgrading or
                    pkg in report.downgrading):
                    return _("(installed)")
                elif pkg in report.upgraded:
                    return _("(upgraded)")
                elif pkg in report.removed:
                    return _("(removed)")
                elif pkg in report.downgraded:
                    return _("(downgraded)")
                else:
                    return "" # Shouldn't happen
            def showPackages(pkgs, showrelations=True):
                pkgs.sort()
                for pkg in pkgs:
                    channels = []
                    for loader in pkg.loaders:
                        channels.append(loader.getChannel().getAlias())
                        channels.sort()
                    print " ", pkg, ("[%s]" % ", ".join(channels))
                    if showrelations:
                        if pkg in report.upgrading:
                            print "   ", _("Upgrades:")
                            for upgpkg in report.upgrading[pkg]:
                                print "     ", upgpkg, being(upgpkg)
                        if pkg in report.downgrading:
                            print "   ", _("Downgrades:")
                            for dwnpkg in report.downgrading[pkg]:
                                print "     ", dwnpkg, being(dwnpkg)
                        if pkg in report.requires:
                            print "   ", _("Requires:")
                            for reqpkg in report.requires[pkg]:
                                print "     ", reqpkg, being(reqpkg)
                        if pkg in report.requiredby:
                            print "   ", _("Required By:")
                            for reqpkg in report.requiredby[pkg]:
                                print "     ", reqpkg, being(reqpkg)
                        if pkg in report.conflicts:
                            print "   ", _("Conflicts:")
                            for cnfpkg in report.conflicts[pkg]:
                                print "     ", cnfpkg, being(cnfpkg)

        print
        if keep:
            print _("Kept packages (%d):") % len(keep)
            showPackages(keep, False)
            print
        pkgs = report.upgrading.keys()
        if pkgs:
            print _("Upgrading packages (%d):") % len(pkgs)
            showPackages(pkgs)
            print
        pkgs = report.downgrading.keys()
        if pkgs:
            print _("Downgrading packages (%d):") % len(pkgs)
            showPackages(pkgs)
            print
        pkgs = report.installing.keys()
        if pkgs:
            print _("Installing packages (%d):") % len(pkgs)
            showPackages(pkgs)
            print
        pkgs = report.removed.keys()
        if pkgs:
            print _("Removing packages (%d):") % len(pkgs)
            showPackages(pkgs)
            print

        dsize = report.getDownloadSize() - report.getCachedSize()
        size = report.getInstallSize() - report.getRemoveSize()
        if dsize:
            sys.stdout.write(_("%s of package files are needed. ") %
                             sizeToStr(dsize))
        if size > 0:
            sys.stdout.write(_("%s will be used.") % sizeToStr(size))
        elif size < 0:
            size *= -1
            sys.stdout.write(_("%s will be freed.") % sizeToStr(size))
        if dsize or size:
            sys.stdout.write("\n\n")
        if confirm:
            return self.askYesNo(_("Confirm changes?"), True)
        return True

# vim:ts=4:sw=4:et
