import shutil, tarfile, os, io, subprocess, glob
import urllib.request as urlreq
from jdsu.mts.bootutils import bootenv
from jdsu.mts.blockutils import mount, umount, mount_bind, mounted_point
from jdsu.mts.flashutils import mount_chroot, umount_chroot, chroot_exec, format_device
from jdsu.mts.upgrade import PrintException, run_hook, split_url, md5

def synchronize(source, destination, overwrite=False):
    if os.path.isdir(source):
        if os.path.isdir(destination):
            for item in os.listdir(source):
                s = os.path.join(source, item)
                d = os.path.join(destination, item)
                synchronize(s, d)
        else:
            shutil.copytree(source, destination, symlinks=True)
    else:
        if not os.path.lexists(destination) or overwrite:
            if os.path.islink(source):
                linkto = os.readlink(source)
                os.symlink(linkto, destination)
            else:
                if not os.path.exists(os.path.dirname(destination)):
                    os.makedirs(os.path.dirname(destination))
                if os.path.exists(source):
                    shutil.copy2(source, destination)


class MyFileObj(io.FileIO):
    def setCls(self, cls, fsize, name):
        self.__cls = cls
        self.__fsize = fsize
        self.__rname = name
        self.lastpercent = 1000

    def read(self, size):
        percent = int(self.tell() * 100 / self.__fsize)
        if percent != self.lastpercent:
            if self.__cls is not None:
                self.__cls.expose("Upgrade", 66, self.__rname, "Installation", percent)
            self.lastpercent = percent
        return io.FileIO.read(self, size)

def download_file(src, dest, proxy, chunk_size=8192, report_hook=None):
    with open(dest, "wb") as destination:
        if proxy is not None and proxy.strip() != "DIRECT":
            print("Enable proxy %s" % proxy)
            proxy_handler = urlreq.ProxyHandler({'http': proxy})
        else:
            print("Disable proxy")
            proxy_handler = urlreq.ProxyHandler({})
        opener = urlreq.build_opener(proxy_handler)
        urlreq.install_opener(opener)
        response = urlreq.urlopen(src);
        total_size = response.info().get('Content-Length').strip()
        total_size = int(total_size)
        bytes_so_far = 0

        while 1:
            chunk = response.read(chunk_size)
            bytes_so_far += len(chunk)
            destination.write(chunk)

            if not chunk:
                break

            if report_hook:
                report_hook(bytes_so_far, chunk_size, total_size)

    return bytes_so_far



# smartmanager:
#  expose
#  set_action
#  log
#  upgrade_by_tar_terminate

# options:
#  proxy
#  daemon
#  flash_block_device
#  auto_switch
#  test_update_part
def start_upgrade_tar(xmlrelease, hook = None, options = {}, smartmanager = None):
    if not run_hook(hook, 'pre_tar_upgrade'):
        return False
    # Option default values
    if 'auto_switch' not in options.keys():
        options['auto_switch'] = True
    if 'test_update_part' not in options.keys():
        options['test_update_part'] = False
    if 'proxy' not in options.keys():
        options['proxy'] = ''
    if 'current_rootfs' not in options.keys():
        options['current_rootfs'] = ''
    if 'syncronise' not in options.keys():
        options['syncronise'] = True

    result = True
    tarf = xmlrelease.xpath("repositories/archive/@file")[0]
    name = xmlrelease.xpath("@name")[0]
    try:
        size = xmlrelease.xpath("repositories/archive/@size")[0]
        size = int(size) / (1024*1024)
        size = int(size*1.1)
    except :
        size = 256
    if size < 256:
        size = 256
    url = xmlrelease.url()
    remote = xmlrelease.remote()
    (xmlpath, slot) = split_url(url)
    path = "/".join(xmlpath.split("/")[:-1]) + "/" + tarf
    if remote:
        try:
            subprocess.check_output(["mount", "-t", "tmpfs", "-o", "size={}M".format(size), "tmpfs", "/mnt/tmp"])
        except:
            PrintException()
            return False
        lpath = "/mnt/tmp"
    else:
        lpath = "/".join(xmlpath.split("/")[:-1]) + "/"

    def chunk_report(bytes_so_far, chunk_size, total_size):
        percent = float(bytes_so_far) / total_size
        percent = round(percent * 100, 2)
        if int(percent) != int(chunk_report.lastpercent):
            if smartmanager is not None:
                smartmanager.expose("Upgrade", 33, name, "Download release", percent)
            chunk_report.lastpercent = percent

    chunk_report.lastpercent = 1000
    run_hook(hook, 'pre_tar_download')

    if remote:
        if smartmanager is not None:
            smartmanager.set_action("1: Download release")
            smartmanager.log("Download %s" % path)
        download_file(path, "%s/%s" % (lpath, tarf), options['proxy'], report_hook=chunk_report)

    if smartmanager is not None:
        smartmanager.set_action("2: Check release")
    try:
        if not os.path.exists("/etc/release/no_check"):
            if remote:
                if smartmanager is not None:
                    smartmanager.log("Download %s.sig" % path)
                download_file(path + ".sig", "%s/%s.sig" % (lpath, tarf), options['proxy'], report_hook=chunk_report)

            if smartmanager is not None:
                smartmanager.expose("Upgrade", 50, name, "Check integrity", 0)
            print(
                subprocess.check_output(["gpg", "--verify", "%s/%s.sig" % (lpath, tarf), "%s/%s" % (lpath, tarf)]))
            if smartmanager is not None:
                smartmanager.expose("Upgrade", 50, name, "Check integrity", 100)
    except subprocess.CalledProcessError as err:
        print("BAD SIGNATURE")
        if options['daemon']:
            if smartmanager is not None:
                smartmanager.upgrade_by_tar_terminate("BAD SIGNATURE")
        return
    try:
        try:
            bootenv.refresh()
            mmc_part = int(bootenv.mmc_part)
            if mmc_part == options['parts'][0]:
                mmc_part_new = options['parts'][1]
            else:
                mmc_part_new = options['parts'][0]
        except:
            mmc_part_new = options['parts'][1]

        print("mmc_part is %s" % mmc_part_new)

        # Unmount the partition about to be formatted (all mounts)
        for mount_point in mounted_point(options['flash_block_device'] + "p%s" % mmc_part_new):
            umount(mount_point)

        run_hook(hook, 'pre_tar_install')
        if smartmanager is not None:
            smartmanager.set_action("3: Format partitions")
        if not os.path.exists("/mnt/fs"):
            os.makedirs("/mnt/fs")
        if smartmanager is not None:
            smartmanager.expose("Upgrade", 60, name, "Prepare filesystem %s" % mmc_part_new, 10)
        format_device(mmc_part_new)
        if smartmanager is not None:
            smartmanager.expose("Upgrade", 60, name, "Prepare filesystem", 50)
        mount(options['flash_block_device'] + "p%s" % mmc_part_new, "/mnt/fs/", "ext4")
        if smartmanager is not None:
            smartmanager.expose("Upgrade", 60, name, "Prepare filesystem", 60)
        if not os.path.exists("/mnt/fs/user/disk"):
            os.makedirs("/mnt/fs/user/disk")
        if smartmanager is not None:
            smartmanager.expose("Upgrade", 60, name, "Prepare filesystem", 70)
        mount_bind("/user/disk", "/mnt/fs/user/disk")
        if smartmanager is not None:
            smartmanager.expose("Upgrade", 60, name, "Prepare filesystem", 100)
            smartmanager.set_action("4: Install release")

        run_hook(hook, 'pre_tar_untar')
        total_size = os.path.getsize("%s/%s" % (lpath, tarf))
        with MyFileObj("%s/%s" % (lpath, tarf)) as fileobj:
            fileobj.setCls(smartmanager, total_size, name)
            with tarfile.open(fileobj=fileobj, errorlevel=1) as tar:
                tar.extractall("/mnt/fs")
        run_hook(hook, 'post_tar_untar')

        if smartmanager is not None:
            smartmanager.set_action("5: Setup release")
        try:
            shutil.copyfile("/etc/resolv.conf", "/mnt/fs/etc/resolv.conf")
        except shutil.SameFileError:
            pass
        except FileNotFoundError:
            try:
                open("/mnt/fs/etc/resolv.conf", 'a').close()
            except:
                print("Warning: Can't create /mnt/fs/etc/resolv.conf")
                pass

        # Copy all file explicitly listed in /etc/release/backup
        run_hook(hook, 'pre_synchronize')
        if options['syncronise']:
            if smartmanager is not None:
                smartmanager.set_action("6: Synchronize")
            backup = []
            if os.path.exists("{}/etc/release/backup".format(options['current_rootfs'])):
                for f in os.listdir("{}/etc/release/backup".format(options['current_rootfs'])):
                    with open("{}/etc/release/backup/{}".format(options['current_rootfs'], f)) as fi:
                        backup += [line.strip() for line in fi if line.strip() != '']
            if os.path.exists("/mnt/fs/etc/release/backup"):
                for f in os.listdir("/mnt/fs/etc/release/backup"):
                    with open("/mnt/fs/etc/release/backup/%s" % f) as fi:
                        backup += [line.strip() for line in fi if line.strip() != '']
            run_hook(hook, 'pre_synchronize_list')

            for bk in set(backup):
                override = False
                if bk.startswith("@"):
                    override = True
                    bk = bk[1:].strip()
                for fn in glob.glob(options['current_rootfs'] + bk):
                    if len(options['current_rootfs']) > 0:
                        print("Sync {} {}".format(fn, fn.replace(options['current_rootfs'], "/mnt/fs")))
                        synchronize(fn, fn.replace(options['current_rootfs'], "/mnt/fs"), override)
                    else:
                        print("Sync {} {}".format(fn, "/mnt/fs/%s" % fn))
                        synchronize(fn, "/mnt/fs/%s" % fn, override)
        run_hook(hook, 'post_synchronize')

        if smartmanager is not None:
            smartmanager.set_action("7: Run post install (don't turn off device)")
        id = xmlrelease.xpath("@id")[0]
        xmlrelease.save("/mnt/fs/etc/release/%s.xml" % id)
        os.system("rm -f /mnt/fs/etc/release/current.xml")
        os.system("ln -s /etc/release/%s.xml /mnt/fs/etc/release/current.xml" % id)

        # Remove old files form /users/disk declared in /etc/release/remove
        if os.path.isdir("/mnt/fs/etc/release/remove"):
            for f in os.listdir("/mnt/fs/etc/release/remove"):
                with open("/mnt/fs/etc/release/remove/%s"%f) as rf:
                    for line in [l.strip() for l in rf.readlines() if len(l.strip()) > 0 and not l.startswith("#")]:
                        tab = line.split(" ")
                        if os.path.isfile(tab[1]):
                            if md5(tab[1]) == tab[0]:
                                print("Delete removable file: %s"%tab[1])
                                os.unlink(tab[1])

        # Manage Downgrade to fido (cortexa9_vfp_neon architecture)
        if "cortexa9_vfp_neon" in xmlrelease.xpath("/repositories/rpm/@id"):
            bootenv.addmmc = 'setenv bootargs ${bootargs} root=/dev/mmcblk0p${mmc_part} rootwait'
            bootenv.bootcmd = 'run upgenv; run defaultboot'
            bootenv.bootdelay = '0'
            bootenv.defaultboot = 'get_vext 6000 8000 || run mmcboot'
            bootenv.env_version = '1'
            bootenv.getitb = 'ext2load mmc 0:${mmc_part} ${kernel_addr_r} /boot/mts1000.itb'
            bootenv.kernel_addr_r = '0x2000000'
            bootenv.kernel_options = 'sdhci.debug_quirks=0x8000 coherent_pool=0x400000'
            bootenv.mmcboot = 'run getitb; run addtty addmmc addfixes;bootm ${kernel_addr_r}'
            bootenv.rescue = 'setenv mmc_part 7; run mmcboot'
            bootenv.rescuefs = 'run addtty addfixes;bootm ${kernel_addr_r}'
            bootenv.rescuefs_version = '1'
            bootenv.stderr = 'serial'
            bootenv.stdin = 'serial'
            bootenv.stdout = 'serial'
            bootenv.upgenv = 'ext2load mmc 0:${mmc_part} ${kernel_addr_r} /boot/configs.txt && env import ${kernel_addr_r} ${filesize}'

        try:
            mount_chroot()
            os.system("mount > /tmp/mount.log")
            chroot_exec("/mnt/fs", ["/usr/sbin/run-postinsts"])
        except:
            result = False
            PrintException()
        finally:
            umount_chroot()
    except subprocess.CalledProcessError as err:
        result = False
        PrintException()
    except:
        result = False
        PrintException()
    finally:
        if smartmanager is not None:
            smartmanager.set_action("8: Clean installation")
            smartmanager.expose("Upgrade", 90, name, "Clean installation", 100)
        umount("/mnt/fs/user/disk")
        if not os.path.isdir("/mnt/fs/var/lib/release-manager"):
            os.mkdir("/mnt/fs/var/lib/release-manager")
        with open("/mnt/fs/var/lib/release-manager/upgrade_status", "w") as f:
            f.write("Success")
        umount("/mnt/fs")
        if remote:
            umount("/mnt/tmp")
        if result:
            run_hook(hook, 'pre_enable')
            if smartmanager is not None:
                smartmanager.set_action("8: Enable new installation")
                smartmanager.expose("Upgrade", 100, name, "Enable new installation", 100)
            if options['auto_switch']:
                print("Setting permanent boot part")
                bootenv.mmc_part = mmc_part_new
            if options['test_update_part']:
                print("Setting temp boot part")
                bootenv.mmc_part_volatile = mmc_part_new
            if smartmanager is not None:
                smartmanager.expose("Upgrade", 100, name, "Finished successful", 100)
            run_hook(hook, 'post_enable')
    if options['daemon']:
        if smartmanager is not None:
            smartmanager.set_action("Upgrade Terminate")
        if result:
            if smartmanager is not None:
                smartmanager.upgrade_by_tar_terminate("OK")
        else:
            if smartmanager is not None:
                smartmanager.upgrade_by_tar_terminate("FAIL")
    if not run_hook(hook, 'post_tar_upgrade'):
        return False
    return result



