#!/usr/bin/python3

from __future__ import print_function
import parted, os, sys, time
import tarfile
import subprocess
from shutil import copyfile, SameFileError
import xml.etree.ElementTree as etree
try:
    from urllib.request import urlopen
except ImportError:
    from urllib2 import urlopen
try:
    from .blockutils import get_block_devices, get_partition_plan, mount, umount, mount_bind, get_partition_plan
except:
    from blockutils import get_block_devices, get_partition_plan, mount, umount, mount_bind, get_partition_plan

GRAY =   (0xBB, 0xBB, 0xBB)
BLACK =   (0x0, 0x0, 0x0)
BLUE =   (0x0, 0x0, 0xFF)

global last_pbar
last_pbar = 0

def progress_bar(gui, value):
    global last_pbar
    value = int(value)
    if gui is not None:
        if value != last_pbar:
            last_pbar = value
            gui.rectangle(50,250, int((value * 700) / 100),50,BLUE)
            print("Install %i%%"%value)
            sys.stdout.flush()

def sync():
    try:
        os.sync()
    except:
        os.system("sync")


def part_type(typ):
    if typ == 'p':
        return parted.PARTITION_NORMAL
    if typ == 'x':
        return parted.PARTITION_EXTENDED
    if typ == 'l':
        return parted.PARTITION_LOGICAL

def create_partitions_table(mmc, partitionPlan):
    device = parted.getDevice("/dev/%s"%mmc)
    disk = parted.freshDisk(device, 'msdos')
    formatlist = []
    for part in partitionPlan:
        geometry = parted.Geometry(device=device, start=part['start'], end=part['end'])
        if part['format'] != 'raw':
            filesystem = parted.FileSystem(type=part['format'], geometry=geometry)
        else:
            filesystem = None
        partition = parted.Partition(disk=disk, type=part_type(part['type']), fs=filesystem, geometry=geometry)
        disk.addPartition(partition=partition, constraint=device.minimalAlignedConstraint)
        if 'label' in part.keys():
            formatlist.append("mkfs.%s -F -L %s %s"%(part['format'], part['label'], partition.path))
    disk.commit()
    sync()
    for action in formatlist:
        print(action)
        os.system(action)
    sync()

def re_format_device(mmc, partitionPlan, select=None):
    formatlist = []
    for partid in range(len(partitionPlan)):
        part = partitionPlan[partid]
        if 'label' in part.keys() and not part['protected'] and (select == None or partid == (select - 1)):
            formatlist.append("mkfs.%s -F -L %s %s"%(part['format'], part['label'], "/dev/%sp%i"%(mmc, partid + 1) ))
    for action in formatlist:
        print(action)
        os.system(action)
    sync()

def format_device(select):
    mmc = list(get_block_devices("MMC").keys())[0]
    plan = get_partition_plan()
    re_format_device(mmc, plan, select)

def mount_target(mmc, fs):
    mount("/dev/%sp%i"%(mmc, fs), "/mnt/fs/", "ext4")
    mount("/dev/%sp%i"%(mmc, 2), '/mnt/fs/cal', 'ext4')
    mount("/dev/%sp%i"%(mmc, 3), '/mnt/fs/user/disk', 'ext4')

def umount_target():
    umount('/mnt/fs/user/disk')
    umount('/mnt/fs/cal')
    umount('/mnt/fs')

def StreamDecorator(original, tarsize, notify):
    setattr(original.__class__,"oldread",original.read)
    setattr(original.__class__,"notify",notify)
    original.lastpercent = 0
    original.tarsize = int(tarsize)
    original.currentsize = 0

    def myread(self, size):
        percent = self.currentsize * 100 / self.tarsize
        if percent != self.lastpercent:
            self.notify(percent)
            self.lastpercent = percent
        self.currentsize += size
        return self.oldread(size)

    myread.__doc__ = "My read"
    myread.__name__ = "read"
    setattr(original.__class__,"read",myread)
    return original

def printme(self, percent):
    if(percent % 5 != 0):
        return

    if percent != printme.last:
        printme.last = percent
        print("# Install %i%%"%percent)
printme.last = 120


def install_image(uri, notifier=printme, gui=None):
    sync()
    if gui:
        def print_gui(self,percent):
            progress_bar(gui, percent)
        notifier = print_gui
    f = urlopen(uri)
    xmlstring = f.read()
    releasexml = etree.fromstring(xmlstring)
    archive = releasexml.findall("./release/repositories/archive")[0]
    tarsize = archive.attrib['size']
    tarfilename = archive.attrib['file']
    taruri = uri.replace('release.xml', tarfilename)
    f = StreamDecorator(urlopen(taruri), tarsize, notifier)
    tar = tarfile.open(fileobj=f, mode='r|gz', errorlevel=1)
    tar.extractall("/mnt/fs")
    sync()
    with open("/mnt/fs/etc/release/current.xml", "wb") as currentrelease:
        currentrelease.write(xmlstring)
    sync()

def mount_chroot(path="/mnt/fs"):
    mount_bind("/dev/", "%s/dev/"%path)
    mount("none", "%s/sys"%path, "sysfs")
    mount("none", "%s/proc"%path, "proc")
    try:
        copyfile("/etc/resolv.conf", "%s/etc/resolv.conf"%path)
    except SameFileError:
        pass
    except FileNotFoundError:
        open("%s/etc/resolv.conf"%path, 'a').close()

def umount_chroot(path="/mnt/fs"):
    umount("%s/dev"%path)
    umount("%s/sys"%path)
    umount("%s/proc"%path)

def chroot_exec(path, command):
    real_root = os.open("/", os.O_RDONLY)
    os.chroot(path)
    subprocess.check_output(command)
    sync()
    os.fchdir(real_root)
    os.chroot(".")
    os.close(real_root)

def install_release(releasexml, firstinstall=True, part=5, gui=None):
    result = False
    mmc = list(get_block_devices("MMC").keys())[0]
    partitiontable = get_partition_plan()
    if firstinstall:
        create_partitions_table(mmc, partitiontable)
    else:
        re_format_device(mmc, partitiontable, part)
    try:
        mount_target(mmc, part)
        if gui:
            progress_bar(gui, 2)
        install_image(releasexml, gui=gui)
        try:
            mount_chroot()
            if not os.path.exists("/mnt/fs/var/run/lock"):
                os.makedirs("/mnt/fs/var/run/lock")
            if not os.path.exists("/mnt/fs/boot/env.d"):
                os.makedirs("/mnt/fs/boot/env.d")
            chroot_exec("/mnt/fs", ["/usr/sbin/run-postinsts"])
            chroot_exec("/mnt/fs", ["/sbin/fw_setenv", "mmc_part", "%i"%part])
            result = True
        finally:
            umount_chroot()
    finally:
        umount_target()
    return result

def install_pmulti(uri, firstinstall=True, part=7):
    mmc = list(get_block_devices("MMC").keys())[0]
    if firstinstall:
        create_partitions_table(mmc, get_partition_plan())
    else:
        re_format_device(mmc, get_partition_plan(), part)
    try:
        mount("/dev/%sp%i"%(mmc, part), "/mnt/fs/", "ext4")
        sync()
        f = urlopen(uri)
        xmlstring = f.read()
        releasexml = etree.fromstring(xmlstring)
        pmulti = releasexml.findall("./release/pmulti")[0]
        pmultiname = pmulti.attrib['file']
        pmultiuri = uri.replace('release.xml', pmultiname)
        r = urlopen(pmultiuri)
        with open("/mnt/fs/pmulti.itb", "wb") as pm:
            while(True):
                stream = r.read(1*1024*1024) # Read 1MB
                if(len(stream) == 0):
                    break
                pm.write(stream)
        sync()
    finally:
        umount("/mnt/fs")


if __name__ == "__main__":
    if len(sys.argv) < 2:
        print("%s [-f] http://server/release.xml\n\t-f: Force full format device (!!! Will loose Calibration !!!)"%sys.argv[0])
        exit()
    if sys.argv[1] == "-f":
        install_release(sys.argv[2], True)
    elif sys.argv[1] == "-g":
        # install With gui
        from jdsu.mts.framebuffer import FrameBuffer
        gui = FrameBuffer("/dev/fb0")
        gui.rectangle(0,300,800,180,BLACK)
        gui.rectangle(50,250,700,50,GRAY)
        if install_release(sys.argv[2], False, gui=gui):
            print("Filesystem successfully installed")
            exit(0)
        exit(-1)
    else:
        install_release(sys.argv[1], False)
