import threading
import time
import os
import signal
import sys
from queue import Queue
from datetime import datetime

from .tools import CommandInvoker


def press_ctrl_c(signum, frame):
    __builtins__["session"].send_event("__SYS_CTRL", chr(3))


def press_ctrl_z(signum, frame):
    __builtins__["session"].send_event("__SYS_CTRL", chr(26))


class _Event(object):
    '''
    Class used to contain information on an event
    '''

    def __init__(self, evtype, value, level=50):
        '''
        Constructor
        '''
        self.__type = evtype
        self.__value = value
        self.__time = datetime.now()
        self.__level = level

    def __getattr__(self, name):
        '''
        For give read only access of privates attributes
        we overload this methode
        '''
        if name == "type":
            return self.__type
        elif name == "value":
            return self.__value
        elif name == "time":
            return self.__time
        elif name == "level":
            return self.__level
        else:
            raise AttributeError


class _EventsManager(threading.Thread):
    '''
    This class in for the Events manager thread
    This thread put each event in the fifo and run it
    '''

    def __init__(self, evfifo):
        '''
        Constructor, need session and the event fifo
        '''
        self.__eventsfifo = evfifo
        threading.Thread.__init__(self)

    def run(self):
        '''
        Main function, core thread
        '''
        event = _Event("START", "")
        while event.type != "EXIT":
            event = self.__eventsfifo.get()
            __builtins__["session"].trig_event(event)


class Session(object):
    '''
    Session is the class of the main object of the application.
    This is the interface for stream and event manager.
    '''
    def __init__(self):
        '''
        Constructor
        '''
        self.__stream = {}
        self.__current_stream = None
        self.__eventsfifo = Queue()
        self.__evmanager = _EventsManager(self.__eventsfifo)
        self.__slot = []
        self.__threads = []

        self.__modulepaths = []
        signal.signal(signal.SIGINT, press_ctrl_c)
        signal.signal(signal.SIGTSTP, press_ctrl_z)
        self.add_module_path(os.path.dirname(__file__))

    def manage(self, th):
        if isinstance(th, threading.Thread):
            self.__threads.append(th)

    def add_module_path(self, path):
        sys.path.append(path)
        self.__modulepaths.append(path)

    def get_path(self):
        return self.__modulepaths

    def send_event(self, name, value, level=50):
        '''
        Send an event in the system
        '''
        ev = _Event(name, value, level)
        self.__eventsfifo.put(ev)
        self.__eventsfifo.put(_Event("Event", ev, 1))

    def stop(self):
        '''
        Send a stop into all flux port
        '''
        for device in list(self.__stream.keys()):
            self.__stream[device].stop()

    def start(self):
        '''
        Start all thread
        '''
        for th in self.__threads:
            th.start()

        for device in list(self.__stream.keys()):
            self.__stream[device].start()
        self.__evmanager.start()

    def join(self):
        '''
        Wait end of thread writer (>Exit command)
        '''
        try:
            for th in self.__threads:
                th.join()
        except KeyboardInterrupt:
            pass
        self.__evmanager.join()
        for device in list(self.__stream.keys()):
            self.__stream[device].join()

    def trig_event(self, event):
        for obj, func in self.__slot:
            f = CommandInvoker(obj, func, {})
            f(event)

    def connect_event(self, obj, funcname):
        self.__slot.append((obj, funcname))

    def make_stream(self, name, stream, default=None, mode=True):
        if self.__current_stream is None:
            default = True
        elif default is None:
            default = False
        O().create_flux(name, mode)
        self.__stream[name] = stream
        if default or name == self.env("flux.default.stdin"):
            self.set_stream(name)

    def delete_stream(self, name):
        if self.__current_stream == self.__stream[name]:
            self.__stream[name] = None
            for fluxname in list(self.__stream.keys()):
                if fluxname != name:
                    self.__current_stream = self.__stream[fluxname]
                    break
        O().delete_flux(name)
        del self.__stream[name]

    def get_stream(self, name=None):
        if name is None:
            return list(self.__stream.keys())
        if name in self.__stream.keys():
            return self.__stream[name]
        return None

    def set_current_stream(self, name):
        self.__current_stream = self.__stream[name]

    def get_current_stream(self):
        return self.__current_stream

    @classmethod
    def create(cls):
        if "session" not in __builtins__.keys():
            __builtins__["session"] = cls()
        return __builtins__["session"]
