# -*- coding: utf-8 -*-
from threading import BoundedSemaphore
import sys
from . import E, Log
from .tools import CommandInvoker

COLORS = {
    "default": "\033[0m",
    # style
    "bold": "\033[1m",
    "underline": "\033[4m",
    "blink": "\033[5m",
    "reverse": "\033[7m",
    "concealed": "\033[8m",
    # couleur texte
    "black": "\033[30m",
    "red": "\033[31m",
    "green": "\033[32m",
    "yellow": "\033[33m",
    "blue": "\033[34m",
    "magenta": "\033[35m",
    "cyan": "\033[36m",
    "white": "\033[37m",
    # couleur fond
    "on_black": "\033[40m",
    "on_red": "\033[41m",
    "on_green": "\033[42m",
    "on_yellow": "\033[43m",
    "on_blue": "\033[44m",
    "on_magenta": "\033[45m",
    "on_cyan": "\033[46m",
    "on_white": "\033[47m"}


class ConsoleOutput(object):
    '''
    This class manage stdout console
    '''

    def __init__(self, colormode=True):
        self.__printermutex = BoundedSemaphore(1)
        self.__printerflag = False
        self.__color = colormode
        self.__colormode = {
            'default': 'default',
            'error': 'red;on_blue',
            'ok': 'green',
            'help': 'yellow',
            'info': 'magenta',
            # Set default flux mode (start with # like #<flux_name>
            '#console': 'default'
        }

        self.__stream = {
            'std': True,
            'console': True
        }

        self.__currentmode = "default"
        self.__return_line = True
        self.trigger = None

    def __write(self, chars, flush=False, stream="std", color=None,
                nline=False):
        if nline and self.__color:
            sys.stdout.write("\033[0K")

        colorvalue = None

        if color is None or color not in self.__colormode:
            if "#"+stream in self.__colormode:
                colorvalue = self.__colormode["#"+stream]
        else:
            colorvalue = self.__colormode[color]
        if colorvalue is None:
            colorvalue = self.__colormode["default"]

        if colorvalue != self.__currentmode:
            self.changecolormode(colorvalue)

        if not self.get_stream_state(stream):
            return
        sys.stdout.write(chars)
        if flush:
            sys.stdout.flush()

    def write(self, buf):
        if self.__current_stream is not None:
            self.__current_stream.write(buf)

    def flush(self):
        if self.__current_stream is not None:
            self.__current_stream.flush()

    def changecolormode(self, color):
        setcolor = "".join(
            [COLORS[t] for t in color.split(";") if t in COLORS])
        if self.__color:
            sys.stdout.write(setcolor)
        self.__currentmode = color

    def setexclusive(self, mode):
        '''
        if mode == True  : Lock mutex to print somethings in display console,
        if mode == False : Release mutex
            Use by all command to block flux output during execution
        '''
        if mode:
            self.__printerflag = True
            self.__printermutex.acquire()
        else:
            self.__printerflag = False
            self.__printermutex.release()
            self.__return_line = True

    def cleanline(self):
        if self.__color:
            self.__write("\033[0K\n\033[0K\033[1A")
        else:
            self.__write("\n")

    def get_stream_state(self, state):
        if state in self.__stream:
            return self.__stream[state]
        else:
            return False

    def print_str(self, msg, lock=True, stream="std", color=None, nline=False):
        if lock:
            self.__printermutex.acquire()

        if self.trigger:
            if stream == self.trigger[0]:
                if len(msg) > 0:
                    self.trigger[1]()
                    self.trigger = None
        self.__write(msg, flush=True, stream=stream, color=color, nline=nline)
        if stream == "tty" and msg == "\n":
            fluxdev = E().get_stream(stream)
            if fluxdev is not None:
                self.__write(fluxdev.startNewLine())
        if lock:
            self.__printermutex.release()

    def print_msg(self, msg, lock=True, stream="std", color=None):
        '''
        Print message, if lock == True, we use the mutex
        '''
        Log(msg.strip())
        if lock and not self.__printerflag:
            self.__printermutex.acquire()
        self.__write(msg, stream=stream, nline=True, color=color)
        self.__write("\n", flush=True, stream=stream)
        if lock and not self.__printerflag:
            self.__printermutex.release()

    def __getattr__(self, name):
        if name[0:6] == "print_":
            return CommandInvoker(self, "print_msg", {"color": name[6:]})

    def create_colormode(self, name, colormode):
        self.__colormode[name] = colormode

    def delete_colormode(self, name):
        del self.__colormode[name]

    def create_flux(self, name, mode=True):
        self.__stream[name] = mode

    def delete_flux(self, name):
        del self.__stream[name]

    def print_and_return(self, obj):
        savemode = self.__currentmode
        if self.__color:
            sys.stdout.write("\033[s\0337\n")
        self.changecolormode("bold;white;on_magenta")
        sys.stdout.write(str(obj))
        self.changecolormode(savemode)
        if self.__color:
            sys.stdout.write("\033[0K\0338\033[u")
            sys.stdout.write("\033[1A")
        if self.__return_line is True:
            self.__return_line = False
        sys.stdout.flush()

    def set_flux_mode(self, name, mode):
        self.__stream[name] = mode

    @classmethod
    def create(cls, color):
        if "output" not in __builtins__.keys():
            __builtins__["output"] = cls(color)
        return __builtins__["output"]
