<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of JDSU
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from JDSU
// Copyright JDSU 2013. All rights reserved.
// *********************************************************
namespace app\util;

use app\database\SMTIExecuteInTransaction;

use app\database\SMTDatabaseException;

use app\database\SMTSmartOtuDB;

//WARNING: SMTClassTableNameAnnotation is managed outside namespace to avoid to have to put the full qualified name in the name of the annotation
require_once 'app/util/SMTClassTableNameAnnotation.php';

/**
 * Base Class for dto objects built from database resultset. 
 * The DTO is sent to the client application and sent back to the server.
 * It computes a checksum to track object changes since the last fetch.
 *
 * @author Sylvain Desplat
 */
abstract class SMTResultSetDto extends SMTDto
{    
    /**
     * table primary key name
     * 
     * @var string
     */
    const PRIMARY_KEY_ID = "id";
    
    /**
     * Boolean value code is converted to int because SqLite3 doesn't handle boolean values
     */
    const BOOLEAN_VALUE_TRUE = 1;

    /**
     * Boolean value code is converted to int because SqLite3 doesn't handle boolean values
     */
    const BOOLEAN_VALUE_FALSE = 0;
    
    /**
     * Object Identifier
     * @var integer
     */
    protected $id;
    
    const INSERT_STATEMENT = "INSERT INTO %s(%s) VALUES(%s)";
    
    const UPDATE_STATEMENT = "UPDATE %s SET %s WHERE %s";
    
    const DELETE_STATEMENT = "DELETE from %s where %s";
    
    /**
     * Object identifier
     *
     * @return integer
     */
    public function getId()
    {
        return $this->id;
    }
    
    /**
     * Delete dto and its attached dto properties in a transaction
     *
     * @param \SQLite3 $dbConnection
     * @throws SMTDatabaseException
     */
    public function delete( SMTSmartOtuDB $dbConnection)
    {
    	$dbConnection->runInTransaction( new SMTDelete( $this ) );
    }
    
    /**
     * Delete dto.
     *
     * @param SMTSmartOtuDB $dbConnection
     */
    function performDelete( SMTSmartOtuDB $dbConnection )
    {
        $dtoClassName = $this->getDtoClassName();
        //search if the dto is mapped by a database table
        $tableName = self::getTableName( $dtoClassName );         
        SMTLogger::getInstance()->trace(sprintf("Found table Name: %s for dto class %s.", $tableName, $dtoClassName), SMTLogger::DEBUG);
        if ( !isset( $tableName ) )
        {
            throw new SMTDatabaseException( SMTDatabaseException::TABLE_NOT_FOUND, $tableName );
        }
        SMTLogger::getInstance()->trace(sprintf("Delete record of id: %s for table %s.", $this->id , $tableName), SMTLogger::DEBUG);
        
        $tablecolumn = self::PRIMARY_KEY_ID;        
        $query = $this->formatDeleteQuery( $tableName, $tablecolumn );        
        $statement = $dbConnection->prepare($query);
        $propertyName = $this->checkDtoPropertyName( $tablecolumn );
        
        $bindParameters= " :".$tablecolumn."=>".$this->$propertyName;        
        $statement->bindValue(":".$tablecolumn, $this->$propertyName );
        $statement->bindValue(":".self::PRIMARY_KEY_ID, $this->id );

        self::handleStatementError( $dbConnection, $statement->execute(), "Delete", $query, $bindParameters, __FILE__,__METHOD__,__LINE__ );        
        
    }
    
    /**
     * Save one property of the current dto instance in a database table mapping the class hierarchy.
     *
     * @param SMTSmartOtuDB $dbConnection
     * @param string $propertyName The property name to update
     * 
     * @throws SMTDatabaseException
     */
    public function updateSingleProperty( SMTSmartOtuDB $dbConnection, $propertyName )
    {
        $dbConnection->runInTransaction( SMTSave::createSaveDtoForOneProperty( $this, $this->getDtoClassName(), FALSE, $propertyName ) );
    }        
        
    /**
     * Save in a transaction current dto instance in database tables mapping the class hierarchy and generate objectId primary key if needed.
     * 
     * @param SMTSmartOtuDB $dbConnection
     * 
     * @throws SMTDatabaseException
     */
    public function save( SMTSmartOtuDB $dbConnection  )
    {
        $dbConnection->runInTransaction( new SMTSave( $this ) );          
    }
    
    /**
     * Save current dto in database tables mapping the class hierarchy and generate objectId primary key if needed.
     *
     * @param SMTSmartOtuDB $dbConnection
     * @param string $dtoClassName The dto class name to save
     * @param bool $isNew Whether data to save are new data to insert or existing data to update
     * @param string $singlePropertyName update only the given DTO parameters.
     * 
     * @throws SMTDatabaseException
     */
    function performSave( SMTSmartOtuDB $dbConnection, $dtoClassName, $isNew, $singlePropertyName = NULL )
    {               
        //First, recursively call performSave on super class (table mapping super-class can be referenced by tables mapping sub-classes):
        $reflectionClass = new \ReflectionClass( $dtoClassName );
        $parentDtoClass = $reflectionClass->getParentClass();
         
        //stop recursive save if base class is reached
        if ( $parentDtoClass->getName() != __CLASS__ )
        {
            $this->performSave($dbConnection, $parentDtoClass->getName(), $isNew, $singlePropertyName );
        }                        

        //search if the dto is mapped by a database table
        SMTLogger::getInstance()->trace( sprintf("performSave: Search table Name for dto class %s.", $dtoClassName), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        $tableName = self::getTableName( $dtoClassName );
                        
        if ( $tableName != NULL)
        {
            SMTLogger::getInstance()->trace( sprintf("performSave: Found table Name: %s for dto class %s.", $tableName, $dtoClassName), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
            
            //INSERT: case of new record to insert
            if ( $isNew )
            {
                $this->insertNewRecord( $dbConnection, $tableName );
            }
            //UPDATE: existing record
            else
            {
                $this->updateRecord($dbConnection, $tableName, $singlePropertyName);
            }
        }
        else
        {
            SMTLogger::getInstance()->trace( sprintf("performSave: table Name not found for dto class %s.", $dtoClassName), SMTLogger::DEBUG);
        }    
    }    

    /**
     * Insert new record in database with new Id generation
     * 
     * @param SMTSmartOtuDB $dbConnection
     * @param string $tableName
     */
    protected function insertNewRecord( SMTSmartOtuDB $dbConnection, $tableName )
    {
        //if id is not set, generate a new one from database sequences
        if ( !isset( $this->id ) )
        {
        	$this->id = $dbConnection->generateId( $this );
        }
        SMTLogger::getInstance()->trace( sprintf("Create new record of id: %s for table %s.", $this->id , $tableName), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        //search properties bounded to columns
        $boundColumns= $this->bindTableParameters($dbConnection, $tableName );
        //create insert query
        $query = $this->formatInsertQuery( $tableName, array_keys( $boundColumns ) );
        SMTLogger::getInstance()->trace( sprintf("Insert query: %s", $query), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
         
        $bindParameters = "";
        $statement = $dbConnection->prepare($query);
        
        self::handleStatementError( $dbConnection, $statement, "Prepare Insert", $query, NULL, __FILE__,__METHOD__,__LINE__ );
        
        foreach ( $boundColumns as $column=>$propertyValue )
        {
        	//for logging
        	$bindParameters.= " :".$column."=>".$propertyValue;
        
        	$statement->bindValue(":".$column, $propertyValue );
        }
        
        self::handleStatementError( $dbConnection, $statement->execute(), "Insert", $query, $bindParameters, __FILE__,__METHOD__,__LINE__ );        
    }
    
    /**
     * Update record in database
     *
     * @param SMTSmartOtuDB $dbConnection
     * @param string $tableName
     * @param $singlePropertyName update only the given DTO parameters.
     */
    protected function updateRecord( SMTSmartOtuDB $dbConnection, $tableName, $singlePropertyName )
    {
        SMTLogger::getInstance()->trace(sprintf("Update record of id: %s for table %s.", $this->id , $tableName), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        
        //search properties bounded to columns
        $boundColumns= $this->bindTableParameters($dbConnection, $tableName, $singlePropertyName );
         
        //In the special case of an update of one column (singlePropertyName != NULL), that column may not be found on the table (tableName) but on another table mapped to the class hierarchy
        //If we are not in an update of a single column, the list should never be empty (don't test it and let the update crash and throw an exception).
        if ( $singlePropertyName == NULL || count( $boundColumns ) > 0 )
        {
        	//create update query
        	$query = $this->formatUpdateQuery( $tableName, array_keys( $boundColumns ) );
        	SMTLogger::getInstance()->trace(sprintf("Update query: %s", $query), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        
        	$bindParameters = "";
        	$statement = $dbConnection->prepare($query);
        
        	self::handleStatementError( $dbConnection, $statement, "Prepare Update", $query, NULL, __FILE__,__METHOD__,__LINE__ );
        
        	foreach ( $boundColumns as $column=>$propertyValue )
        	{
        		//for logging
        		$bindParameters.= " :".$column."=>".$propertyValue;
        
        		$statement->bindValue(":".$column, $propertyValue );
        	}
        
        	self::handleStatementError( $dbConnection, $statement->execute(), "Update", $query, $bindParameters, __FILE__,__METHOD__,__LINE__ );
        
        	if ( $statement->execute() === FALSE )
        	{
        		SMTLogger::getInstance()->trace( sprintf("Update failed. Query: %s. Bind parameters: %s", $query, $bindParameters), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
        		throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
        	}
        	else
        	{
        		SMTLogger::getInstance()->trace( sprintf("Update successfull for Query: %s. Bind parameters: %s", $query, $bindParameters), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        	}
        }
    }
    
    /**
     * Search properties (optionally limited to the property"singlePropertyName" ) of the current dto instance bound to table (tableName)
     * 
     * @param SMTSmartOtuDB $dbConnection
     * @param string $tableName The database table to map 
     * @param string $singlePropertyName Limit the binding to the given property (NULL by default)
     * 
     * @return array Database columns bound with current dto instance properties
     */
    protected function bindTableParameters( SMTSmartOtuDB $dbConnection, $tableName, $singlePropertyName = NULL )
    {
        $boundColumns= array();
        $dtoClassName = $this->getDtoClassName();
        SMTLogger::getInstance()->trace( sprintf("Search properties of dto instance: %s, bound to table: %s.", $dtoClassName, $tableName), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        $tablecolumns = $this->getTableColumns( $dbConnection, $tableName );
        //SMTLogger::getInstance()->trace( sprintf("Table columns for query: %s", print_r( $tablecolumns, TRUE) ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
         
        foreach ( $tablecolumns as $column )
        {
            $propertyName = $this->checkDtoPropertyName( $column );
            if ( ( $propertyName != NULL && $singlePropertyName == NULL ) || 
                 ( $propertyName != NULL && $propertyName == $singlePropertyName ) ||
                 ( $propertyName == self::PRIMARY_KEY_ID )
               )
            {
                $bindParameterValue = self::getBindParameterValue( $propertyName, $column, $this );
                if ( $bindParameterValue !== NULL )
                {
                    $boundColumns[$column] = $bindParameterValue;
                    //SMTLogger::getInstance()->trace( sprintf("Found bound column %s in class %s for property %s with value %s", $column, $dtoClassName, $propertyName, $bindParameterValue ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);                     
                }
                else
                {
                    SMTLogger::getInstance()->trace( sprintf("Database column %s not bound in class %s", $column, $dtoClassName), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
                }
            }
            //don't log if column is not found when we search to bind a single property.
            else if ( $singlePropertyName == NULL)
            {
                SMTLogger::getInstance()->trace( sprintf("Database column %s not found in class %s", $column, $dtoClassName), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            }
        }
        
        return $boundColumns;
    }
    
    
    /**
     * Search property (propertyName) of the given dto instance bound to the given database column (column).
     * If binding is not possible (property with no value or column not mapping the property), returns NULL otherwise, returns the value of the bound property. 
     * 
     * @param string $propertyName The property name to search
     * @param string $column database column to map to the property
     * @param $dto The dto The dto instance where the property and its annotations are searched 
     * 
     * @return string The bound parameter value or NULL if no binding is possible.
     */
    protected static function getBindParameterValue( $propertyName, $column, $dto )
    {
        $bindParameterValue = NULL;
        $dtoClassName = $dto->getDtoClassName();
        SMTLogger::getInstance()->trace( sprintf("Try binding parameter %s on column %s for dto: %s", $propertyName, $column, $dtoClassName ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__ );
        if ( isset( $dto->$propertyName ) )
        {
            //if property found is a resultset DTO, search its identifier
            if ( $dto->$propertyName instanceof SMTResultSetDto )
            {
                //SMTLogger::getInstance()->trace( sprintf("Property %s found is a dto.", $propertyName ), SMTLogger::DEBUG );
                $dtoPropertyIdName = self::searchPropertyMappingForeignKeyInClass( $dtoClassName, $propertyName, $column );
                
                SMTLogger::getInstance()->trace( sprintf("Identifier of dto property %s found is: %s.", $propertyName, $dtoPropertyIdName ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
                if ( $dtoPropertyIdName !== NULL  )
                {
                    $bindParameterValue = self::getBindParameterValue($dtoPropertyIdName, $column, $dto->$propertyName);
                }
                else
                {
                    SMTLogger::getInstance()->trace( sprintf("ResultSet dto property %s mapping database column %s not mapped in class %s", $propertyName, $column, $dtoClassName), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
                }
            }
            else if ( $dto->$propertyName instanceof SMTDto  )
            {
                SMTLogger::getInstance()->trace( sprintf("DTO property %s cannot map database column %s in class %s", $propertyName, $column, $dtoClassName), SMTLogger::ERROR, __FILE__,__METHOD__,__LINE__);
            }
            else
            {
                $bindParameterValue= $dto->$propertyName;
                SMTLogger::getInstance()->trace( sprintf("DTO property %s of class %s mapping database column %s has value %s", $propertyName, $dtoClassName, $column, $bindParameterValue ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
            }
        }
        else
        {
            SMTLogger::getInstance()->trace( sprintf("Property %s mapping database column %s is not set (no value) in class %s", $propertyName, $column, $dtoClassName), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        }
        
        return $bindParameterValue;
    }        

    /**
     * Check that the given database column can be mapped in the object properties.
     * Handles columns which are foreign keys mapped by object properties which are also DTOs
     * 
     * @param string $columnName
     * 
     * @return The property name of the current DTO or NULL if not found
     */
    protected function checkDtoPropertyName ( $columnName )
    {
        $propertyName = self::fromColumnNameToPropertyName( $columnName );
        
        //if property is not found in dto instance, try to check if property is a foreign key associated to another DTO.
        if ( !property_exists( $this, $propertyName) )
        {
            //search property associated to foreign key (search if column ends with "id" )
            if ( preg_match("/(\w+)".self::PRIMARY_KEY_ID."$/i", $columnName, $matches) )
            {
                //Remove "id" from property name to handle foreign key associated to another DTO
                $propertyName = self::fromColumnNameToPropertyName( $matches[1] );            
                if ( property_exists( $this, $propertyName) )
                {
                    return $propertyName;
                }
            }
        }
        else
        {
            return $propertyName;
        }
        
        SMTLogger::getInstance()->trace( sprintf("Property %s mapping database column %s not found in class %s", $propertyName, $columnName, $this->getDtoClassName() ), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        return NULL;
    }    
    
    /**
     * Search the given property (propertyName) mapping a foreign key (columnName) in the given dto class (dtoClassName) and recurse in the super-dto-classes.
     * If the property is found, return its name
     * 
     * 
     * @param string $dtoClassName Class where we are looking for the attribute(property) annotation describing how that attributes maps the foreign key.
     * @param string $propertyName The property name to search
     * @param string $columnName Database column searched for logging purpose
     *  
     * @return The property name  or NULL if not found
     */
    private static function searchPropertyMappingForeignKeyInClass( $dtoClassName, $propertyName, $columnName )
    {
        if ( property_exists( $dtoClassName, $propertyName) )
        {
            $propertyAnnotation = new \ReflectionAnnotatedProperty( $dtoClassName, $propertyName );
            if ( $propertyAnnotation->hasAnnotation( 'SMTAttributeInfosAnnotation' ) )
            {
                $attributeInfos = $propertyAnnotation->getAnnotation('SMTAttributeInfosAnnotation');
                if ( isset ($attributeInfos->id) )
                {
                    return $attributeInfos->id;
                }
            }
        }
        else
        {
            //recursively call on super class:
            $reflectionClass = new \ReflectionClass( $dtoClassName );
            $parentDtoClass = $reflectionClass->getParentClass();
             
            //stop recursive search if base class is found
            if ( $parentDtoClass->getName() != __CLASS__ )
            {
                return self::searchPropertyMappingForeignKeyInClass( $parentDtoClass->getName(), $propertyName, $columnName );
            }
        }
        
        SMTLogger::getInstance()->trace( sprintf("Property %s mapping database column %s not found in class %s", $propertyName, $columnName, $dtoClassName), SMTLogger::DEBUG, __FILE__,__METHOD__,__LINE__);
        //try to execute the insert or update statement without that property
        return NULL;
    }
    
    
    
    
    /**
     * Retrieve table columns from database
     * 
     * @param SMTSmartOtuDB $dbConnection
     * @param string $tableName
     * 
     * @return array of strings with table columns
     */
    protected function getTableColumns( SMTSmartOtuDB $dbConnection, $tableName )
    {
        $tablecolumns = array();
        //retrieve table columns from database
        $descTableQuery = "PRAGMA table_info( ".$tableName." )";
        $resultSet = $dbConnection->queryWithTrace( $descTableQuery, SMTLogger::DEBUG );
        while ( $res = $resultSet->fetchArray(SQLITE3_ASSOC) )
        {
            array_push( $tablecolumns, $res["name"] );
        }
                
        return $tablecolumns;
    }
    
    /**
     * Build the insert statement to insert the new dto but not its attached dto properties
     *
     * @param string $tableName the database table to update
     * @param string $tablecolumns the columns of the database table to fill
     *
     * @return insert sql statement with bindings
     */    
    protected function formatInsertQuery( $tableName, $tablecolumns )
    {
        $insertKeys = "";
        $insertBindKeys = "";
    
        foreach ( $tablecolumns as $columnName )
        {
            $insertKeys = $insertKeys.$columnName."," ;
            $insertBindKeys = $insertBindKeys.":".$columnName."," ;
        }
        //remove trailing comma
        $insertKeys = trim($insertKeys,",");
        $insertBindKeys = trim($insertBindKeys,",");
    
        return sprintf( self::INSERT_STATEMENT, $tableName, $insertKeys, $insertBindKeys);
    }
    
    /**
     * Build the update statement to update the dto but not its attached dto properties
     *
     * @param string $tableName the database table to update
     * @param string $tablecolumns the columns of the database table to update
     *
     * @return update sql statement with bindings
     */    
    protected function formatUpdateQuery( $tableName, $tablecolumns )
    {
        $updateKeysAndBindKeys = "";
    
        foreach ( $tablecolumns as $columnName )
        {
            //never update primary key
            if ( $columnName != self::PRIMARY_KEY_ID )
            {
                $updateKeysAndBindKeys = $updateKeysAndBindKeys.$columnName."=:".$columnName."," ;
            }
        }
        //remove trailing comma
        $updateKeysAndBindKeys = trim($updateKeysAndBindKeys,",");
        
        $whereClause = self::PRIMARY_KEY_ID."=:".self::PRIMARY_KEY_ID;
    
        return sprintf( self::UPDATE_STATEMENT, $tableName, $updateKeysAndBindKeys, $whereClause );
    }
    
    /**
     * Build the delete statement to remove the dto and its attached dto properties by cascade
     *
     * @param string $tableName the table associated to the dto to delete
     * @param string $tablecolumn the table column key used to remove the record
     *
     * @return delete sql statement with bindings
     */
    protected function formatDeleteQuery( $tableName, $tablecolumn )
    {
        $whereClause = $tablecolumn."=:".$tablecolumn;
        
        return sprintf( self::DELETE_STATEMENT, $tableName, $whereClause );
    }    
    
    /**
     * Database table name mapping the given DTO class
     * 
     * @param string $dtoClassName
     * @return The table name mapping the given DTO or NULL if none is set
     */
    private static function getTableName( $dtoClassName )
    {
        $classAnnotation = new \ReflectionAnnotatedClass( $dtoClassName );
        if ( $classAnnotation->hasAnnotation( 'SMTClassTableNameAnnotation' ) )
        {
            $classInfos = $classAnnotation->getAnnotation('SMTClassTableNameAnnotation');
            if ( isset ( $classInfos->tablename ) )
            {
                return $classInfos->tablename;
            }
        }
        return NULL;
    }
    
    /**
     * Generate new database dto Id (moke database sequence)
     * WARNING Should be called from a critical section!
     *
     * @param SMTSmartOtuDB $dbConnection
     *
     * @return integer
     */
    abstract protected function generateId( SMTSmartOtuDB $dbConnection );
    
    /**
     * Create a dto resultset instance from the given resulset and the given dto resultset class
     * 
     * @param \SQLite3Stmt $resultSet
     * @param string $dtoResultSetClass
     * 
     * @return SMTResultSetDto
     */
    static function getSingleInstanceFromResultSet( \SQLite3Result $resultSet, $dtoResultSetClass )
    {        
        $dto = NULL;
        $result = $resultSet->fetchArray();
        if ( is_array( $result) && count( $result ) > 0 )
        {
            $dto = new $dtoResultSetClass;
            $dtoProperties = get_object_vars( $dto );
        
            self::forgeDtoFromResultSet( $result, $dto, $dtoProperties);
        }
        return $dto;
    }
    
    /**
     * Create a list of dto resultset instances from the given resulset and the given dto resultset class
     *
     * @param \SQLite3Stmt $resultSet
     * @param string $dtoResultSetClass
     *
     * @return array of SMTResultSetDto or empty array if resulset is empty
     */
    static function getListFromResultSet( \SQLite3Result $resultSet, $dtoResultSetClass )
    {
    	$dtoList = array();
    	while ( $result = $resultSet->fetchArray() )
    	{
        	if ( is_array( $result) && count( $result ) > 0 )
        	{
        		$dto = new $dtoResultSetClass;
        		$dtoProperties = get_object_vars( $dto );
        
        		self::forgeDtoFromResultSet( $result, $dto, $dtoProperties);
        		array_push( $dtoList, $dto );
        	}
    	}
    	return $dtoList;
    }
    
    /**
     * 
     * @param array $result
     * @param SMTResultSetDto $dto
     * @param array $objectProperties
     */
    private static function forgeDtoFromResultSet( $result, SMTResultSetDto $dto, $objectProperties )
    {    
        //to be case insensitive to database column names in keys of $result, map those keys to lower case keys
        $keys = array_keys( $result );
        $mapKeysToLowerCaseKeys = array();
        foreach( $keys as $key )
        {
            //SMTLogger::getInstance()->trace( sprintf( $queryAction. " ResultSet DTO: %s => %s", $key, $result[ $key ] ), SMTLogger::DEBUG, $file, $method, $line);
        	$mapKeysToLowerCaseKeys[strtolower($key)] = $key;
        }        
        
        foreach( $objectProperties as $objectKey =>&$property )
        {
            //properties referencing another dto are not processed because they are not mapped in database: 
            //Indeed, the property name doesn't match the foreign key due to "id" at the end of the database column: 
            //currentAlarmEvent != fromPropertyNameToColumnName( current_alarm_event_id )
            $databaseColumnName = self::fromPropertyNameToColumnName($objectKey);
            if ( isset( $mapKeysToLowerCaseKeys[ $databaseColumnName ] ) )
            {
                SMTLogger::getInstance()->trace( sprintf( " Forge DTO: %s => %s", $mapKeysToLowerCaseKeys[$databaseColumnName], $result[ $mapKeysToLowerCaseKeys[$databaseColumnName] ] ), SMTLogger::DEBUG );                
            	$dto->$objectKey = $result[ $mapKeysToLowerCaseKeys[$databaseColumnName] ];
            }                
        }
    }
    
    /**
     * Utility method to generate a database column name with underscores and lower case characters from a DTO property name.
     *
     * @param name The DTO property name
     * @return The database column name
     */
    private static function fromPropertyNameToColumnName( $propertyName )
    {
        for ( $i = 1; $i < strlen( $propertyName ); $i++ )
        {
            if ( ( ctype_lower( $propertyName[ $i - 1 ] ) ||
                    ctype_digit( $propertyName[ $i - 1 ] )) &&
                    ctype_upper( $propertyName[ $i ] ) &&
                    ( ( ($i + 1) == strlen( $propertyName ) ) || ctype_lower( $propertyName[ $i + 1 ] ) ) )
            {
                $propertyName = substr($propertyName, 0, $i).'_'.( substr($propertyName, $i) );
                $i++;
            }
        }
        return strtolower( $propertyName );
    }
    
    /**
     * Utility method to generate DTO property name from a database column name.
     *
     * @param name The database column name
     * @return The DTO property name
     */
    private static function fromColumnNameToPropertyName( $columnName )
    {
        $propertyName = $columnName;
        for ( $i = 1; $i < strlen( $propertyName ) - 1; $i++ )
        {
            if ( $propertyName[ $i ] == "_" )
            {
                $propertyName[ $i + 1 ] = strtoupper( $propertyName[ $i + 1 ] ); 
                $i++;
            }
        }
        return str_replace( "_", "", $propertyName);
    }
    
    /**
     * Check query result and log query in debug mode on success or log error message with sql error and throw exception on failure.
     * 
     * @param \SQLite3 $dbConnection
     * @param mixed $statementResult FALSE on failure or other value and type (SQLite3Result or TRUE) in case of success 
     * @param string $queryAction (prepare) select, (prepare) insert, (prepare) update or delete
     * @param string $query Query string
     * @param string $bindParameters Bind parameters string if there ae bond parameters or NULL
     * @param string $file
     * @param string $method
     * @param string $line
     * @throws SMTDatabaseException
     */
    public static function handleStatementError( \SQLite3 $dbConnection, $statementResult, $queryAction, $query, $bindParameters, $file, $method, $line )
    {
        if ( $statementResult === FALSE )
        {
            if ( $bindParameters !== NULL )
            {
                SMTLogger::getInstance()->trace( sprintf( $queryAction." statement failed. SQL Error: %s; Query: %s. Bind parameters: %s", $dbConnection->lastErrorMsg(), $query, $bindParameters), SMTLogger::ERROR, $file, $method, $line);
            }        
            else 
            {
        	    SMTLogger::getInstance()->trace( sprintf( $queryAction." statement failed. SQL Error: %s; Query: %s.", $dbConnection->lastErrorMsg(), $query), SMTLogger::ERROR, $file, $method, $line);
            }
        	
        	throw new SMTDatabaseException( SMTDatabaseException::EXECUTE_STATEMENT_FAILURE, $query );
        }
        else
        {
            if ( $bindParameters !== NULL )
            {
            	SMTLogger::getInstance()->trace( sprintf( $queryAction. " statement Successfull. Query: %s. Bind parameters: %s", $query, $bindParameters), SMTLogger::DEBUG, $file, $method, $line);
            }
            else
            {
            	SMTLogger::getInstance()->trace( sprintf( $queryAction. " statement Successfull. Query: %s.", $query), SMTLogger::DEBUG, $file, $method, $line);
            }
        }
    }
    
    /**
     * Convert a boolean into an int to be stored in database
     * 
     * @param boolean $booleanValue
     *
     * @return integer ( 0 or 1 )
     */
    public static function fromBoolean( $booleanValue )
    {
    	return ( filter_var($booleanValue, FILTER_VALIDATE_BOOLEAN) ? SMTResultSetDto::BOOLEAN_VALUE_TRUE : SMTResultSetDto::BOOLEAN_VALUE_FALSE );
    }
    
    /**
     * Convert an int into a boolean
     *
     * @param integer $intValue
     *
     * @return boolean
     */
    public static function ToBoolean( $intValue )
    {
    	return ( $intValue == SMTResultSetDto::BOOLEAN_VALUE_TRUE );
    }    
}
/**
 * Save dto in a transaction
 *
 * @author Sylvain Desplat
 */
class SMTSave implements SMTIExecuteInTransaction
{
	/**
	 * @var SMTResultSetDto
	 */
	var $dto;
	
	/** 
	 * @var string
	 */
	var $dtoClassName;
	
	/**
	 * @var boolean
	 */
	var $isNew = TRUE;
	
	/**
	 * @var string
	 */
	var $propertyName = NULL;

	/**
	 *
	 * @param SMTResultSetDto $dto
	 */
	function __construct( SMTResultSetDto $dto )
	{
		$this->dto = $dto;
		$this->dtoClassName = $dto->getDtoClassName();
		$this->isNew = ( $dto->getId() === NULL );
		$this->propertyName = NULL;
	}

	/**
	 * Only one constructor in PHP => factory method
	 * 
	 * @param SMTResultSetDto $dto
	 * @param string $dtoClassName
	 * @param boolean $isNew
	 * @param string $propertyName
	 * 
	 * @return SMTSave
	 */
	static function createSaveDtoForOneProperty( SMTResultSetDto $dto, $dtoClassName, $isNew, $propertyName )
	{
	    $save = new SMTSave($dto);		
		$save->dtoClassName = $dtoClassName;
		$save->isNew = $isNew;
		$save->propertyName = $propertyName;
		return $save;
	}
	
	/**
	 *
	 */
	public function run( SMTSmartOtuDB $dbConnection )
	{
	    $this->dto->performSave( $dbConnection, $this->dtoClassName, $this->isNew, $this->propertyName );
	}

	/**
	 * @return SMTResultSetDto
	 */
	public function getResultSetDto()
	{
		return $this->dto;
	}
}
/**
 * Delete dto in a transaction
 *
 * @author Sylvain Desplat
 */
class SMTDelete implements SMTIExecuteInTransaction
{
	/**
	 *
	 * @var SMTResultSetDto
	 */
	var $dto;

	/**
	 *
	 * @param SMTResultSetDto $dto
	 */
	function __construct( SMTResultSetDto $dto )
	{
		$this->dto = $dto;
	}

	/**
	 *
	 */
	public function run( SMTSmartOtuDB $dbConnection )
	{
		$this->dto->performDelete($dbConnection);
	}

	/**
	 * @return SMTResultSetDto
	 */
	public function getResultSetDto()
	{
		return $this->dto;
	}
}
?>