<?php
// *********************************************************
// NOTICE: All rights reserved. This material contains the
// trade secrets and confidential information of VIAVI
// which embody substantial creative effort,
// ideas and expressions. No part of this material may be
// reproduced or transmitted in any form or by any means,
// electronic, mechanical, optical or otherwise, including
// photocopying and recording or in connection with any
// information storage or retrieval system, without
// specific written permission from VIAVI Solutions
// Copyright VIAVI Solutions 2017. All rights reserved.
// *********************************************************
namespace app\util;

use app\http\SMTISerializable;

/**
 * DTO used for pagination
 * 
 * @author sdesplat
 */
class SMTPaginationDto extends SMTRwDto implements SMTISerializable
{
    /**
     * The first page constant index
     */
    const DEFAULT_FIRST_PAGE = 1;
    
    /**
     * The default value for the last page (no last page)
     */
    const DEFAULT_LAST_PAGE = -1;
    
    /**
     * The default number of rows per page.
     */
    const MAX_ROW_COUNT_PER_PAGE = 48;
    
    /**
     * Page to display 
     * 
     * @var integer
     */
    protected $firstPage = self::DEFAULT_FIRST_PAGE;

    /**
     * Number of row per page to display
     *
     * @var integer
     */    
    protected $rowPerPage = self::MAX_ROW_COUNT_PER_PAGE;
    
    /**
     * Whether the pagination is due to a navigation 
     * 
     * @var boolean
     */
    protected $isNavigation = TRUE;
    
    /**
     * Whether the value is a valid number 
     * 
     * @param string $value
     * @return boolean
     */
    public static function isValid( $value )
    {
        if ( SMTUtil::isInt( $value ) && ($value >= 0) )
        {
            return TRUE;
        }
        return FALSE;
    }
    
    /**
     * Returns the row count per page
     * 
     * 
     * @return the row count per page
     */
    public function getRowPerPage()
    {
        return self::isValid($this->rowPerPage)? intval($this->rowPerPage): self::MAX_ROW_COUNT_PER_PAGE;
    }
    
    /**
     * Set the row count per page
     *
     *
     * @return the row count per page
     */
    public function setRowPerPage($rowPerPage)
    {
    	if ( self::isValid($rowPerPage) )
    	{ 
    	    $this->rowPerPage = intval($rowPerPage);
    	}
    }
    
    /**
     * Returns the page to display
     *
     *
     * @return the page to display
     */
    public function getFirstPage()
    {
    	return self::isValid($this->firstPage)? $this->firstPage: self::DEFAULT_FIRST_PAGE;
    }
    
    /**
     * Force the page to display
     *
     * @param  the page to display
     */
    public function setFirstPage($page)
    {
    	if ( self::isValid($page) )
    	{
    	    $this->firstPage = intval($page);
    	}
    }
    
    /**
     * Returns the id of the first element to fetch
     *
     * @return the the id of the first element to fetch
     */
    public function getFirstElementId()
    {
    	return ( ($this->getFirstPage() - 1) * $this->getRowPerPage() ) + 1;
    }
    
    /**
     * Returns the id of the first element to fetch
     *
     * @return the the id of the first element to fetch
     */
    public function getLastElementId()
    {
    	return ( $this->getFirstPage() * $this->getRowPerPage() );
    }
    
    /**
     * Whether the pagination is due to a navigation
     * @return boolean
     */
    public function isNavigationPagination()
    {
        return SMTUtil::convertToBoolean($this->isNavigation);
    }
    
    /**
     * Serialize as Json data the role object.<br> 
     * 
     * @return $objectProperties the object properties serialized as a Json string.
     */
    function getJsonData()
    {
    	$objectProperties = get_object_vars( $this );
        $this->serializeObjectAsJsonData( $objectProperties );
    	return $objectProperties;
    }
    
    /**
     * Returns Dto class name.
     *
     * @return string the dtop class name
     */
    function getDtoClassName()
    {
    	return get_class();
    }    
    
    /**
     * Returns the serializable properties of the object.
     *
     * @see \app\http\SMTISerializable::getSerializableProperties()
     * 
     * @return array with the object properties to serialize.
     */
    public function getSerializableProperties()
    {
    	$objectProperties = get_object_vars($this);
    	return array_keys( $objectProperties );
    }    
    
    /**
     * Create the Dto from a Json Dto in an array
     *
     * @param array $json_array json data in an array
     *
     * @return \app\util\SMTPaginationDto
     */
    static function getInstance( array &$json_array )
    {
    	return self::forge( get_class(), $json_array );
    }  
}

?>